package jdepend.framework;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;

/**
 * The <code>JavaClassBuilder</code> builds 
 * <code>JavaClass</code> instances from 
 * .class and .jar (or .zip) files.
 * 
 * @author <b>Mike Clark</b> (mike@clarkware.com)
 * @author Clarkware Consulting, Inc.
 */

public class JavaClassBuilder {
    
    private AbstractParser parser;
    private FileManager fileManager;
    
    
    /**
     * Constructs a <code>JavaClassBuilder</code> instance.
     */
    public JavaClassBuilder() {
        this(new ClassFileParser(), new FileManager());
    }
    
    /**
     * Constructs a <code>JavaClassBuilder</code> instance
     * with the specified file manager.
     * 
     * @param fm File manager.
     */
    public JavaClassBuilder(FileManager fm) {
        this(new ClassFileParser(), fm);
    }
  
    /**
     * Constructs a <code>JavaClassBuilder</code> instance
     * with the specified parser and file manager.
     * 
     * @param parser Parser.
     * @param fm File manager.
     */
    public JavaClassBuilder(AbstractParser parser, FileManager fm) {
        this.parser = parser;
        this.fileManager = fm;
    }
    
    /**
     * Returns the number of Java classes
     * to be built.
     *
     * @return Number of classes.
     */
    public int countClasses() {
        AbstractParser counter = new AbstractParser() {
            public JavaClass parse(InputStream is) {
                return new JavaClass("");
            }
        };
        
        JavaClassBuilder builder = 
            new JavaClassBuilder(counter, fileManager);
        Collection classes = builder.build();
        return classes.size();
    }
    
    /**
     * Builds the <code>JavaClass</code> instances.
     *
     * @return Collection of <code>JavaClass</code> instances.
     */
    public Collection build() {
        
        Collection classes = new ArrayList();
        
        Iterator fileIter = fileManager.extractFiles().iterator();
        while (fileIter.hasNext()) {
            
            File nextFile = (File)fileIter.next();
            
            try {

                classes.addAll(buildClasses(nextFile));       
                
            } catch(IOException ioe) {
                System.err.println("\n" + ioe.getMessage());
            }
        }
        
        return classes;
    }
    
    /**
     * Builds the <code>JavaClass</code> instances from
     * the specified file.
     * 
     * @param file Class or Jar file.
     * @return Collection of <code>JavaClass</code> instances.
     */
    public Collection buildClasses(File file) throws IOException {
        
        if (fileManager.acceptClassFile(file)) {
        
            JavaClass parsedClass = 
                parser.parse(new FileInputStream(file));
            Collection javaClasses = new ArrayList();
            javaClasses.add(parsedClass);
            return javaClasses;
        
        } else if (fileManager.acceptJarFile(file)) { 
            
            JarFile jarFile = new JarFile(file);
            return buildClasses(jarFile);
            
        } else {
            throw new IOException("File is not a valid " +
                ".class, .jar, or .zip file: " + 
                file.getPath());
        }
    }
        
    /**
     * Builds the <code>JavaClass</code> instances from
     * the specified jar (or zip) file.
     *
     * @param file Jar (or zip) file.
     * @return Collection of <code>JavaClass</code> instances.
     */
    public Collection buildClasses(JarFile file) throws IOException { 
        
        Collection javaClasses = new ArrayList();
        
        Enumeration entries = file.entries();
        while (entries.hasMoreElements()) {
            ZipEntry e = (ZipEntry)entries.nextElement();
            if (fileManager.acceptClassFileName(e.getName())) {
                InputStream is = null;
                try {
                    is = file.getInputStream(e);
                    JavaClass jc = parser.parse(is);
                    javaClasses.add(jc);
                } finally {
                    is.close();
                }
            }
        }
        
        return javaClasses; 
    }
}
