package jdepend.framework;

import java.io.*;
import java.util.*;

import junit.framework.*;

/**
 * The <code>ExampleTest</code> is an example <code>TestCase</code> 
 * that demonstrates tests for measuring the distance from the main 
 * sequence (D), package dependency constraints, and the existence 
 * of cyclic package dependencies.
 * <p>
 * This test analyzes the JDepend class files.
 *
 * @author <b>Mike Clark</b> (mike@clarkware.com)
 * @author Clarkware Consulting, Inc.
 */
 
public class ExampleTest extends TestCase {

	private JDepend jdepend;
	
	public String BASE_DIR;

	/**
	 * Constructs a <code>ExampleTest</code> with 
	 * the specified name.
	 *
	 * @param name Test case name.
	 */
	public ExampleTest(String name) {
		super(name);
	}

	/**
	 * Sets up the test fixture.
	 *
	 * Called before every test case method.
	 */
	protected void setUp() {
			
        BASE_DIR = System.getProperty("jdepend.home");
        if (BASE_DIR == null) {
            fail("Property 'jdepend.home' not defined");
        }
        
        PackageFilter filter = new PackageFilter();
        filter.addPackage("java.*");
        filter.addPackage("javax.*");
		jdepend = new JDepend(filter);
		
		try { 
		
			String classesDir = BASE_DIR + File.separator + "build";
			
			jdepend.addDirectory(classesDir);
				
		} catch(IOException ioe)  {
			fail(ioe.getMessage());
		} 
	}

	/**
	 * Tears down the test fixture.
	 *
	 * Called after every test case method.
	 */
	protected void tearDown() {
		jdepend = null;
	}
	
	/**
	 * Tests the conformance of a single package to a 
	 * distance from the main sequence (D) within a 
	 * tolerance.
	 */
	public void testOnePackageDistance() {
		
		double ideal = 0.0;
		double tolerance = 0.8;
		
		jdepend.analyze();

		JavaPackage p = jdepend.getPackage("jdepend.framework");
		assertNotNull(p);

		assertEquals("Distance exceeded: " + p.getName(), 
			ideal, p.distance(), tolerance);
	}
	
	/**
	 * Tests that a single package does not contain
	 * any package dependency cycles. 
	 */
	public void testOnePackageCycle() {
		
		jdepend.analyze();

		JavaPackage p = jdepend.getPackage("jdepend.framework");
		assertNotNull(p);
		
		assertEquals("Cycles exist: " + p.getName(), 
			false, p.containsCycle());
	}
	
	/**
	 * Tests the conformance of all analyzed packages to a 
	 * distance from the main sequence (D) within a tolerance.
	 */
	public void testAllPackagesDistance() {

		double ideal = 0.0;
		double tolerance = 1.0;
		
		Collection packages = jdepend.analyze();

		Iterator iter = packages.iterator();
		while (iter.hasNext()) {
			JavaPackage p = (JavaPackage)iter.next();
			assertEquals("Distance exceeded: " + p.getName(), 
				ideal, p.distance(), tolerance);
		}
	}
	
	/**
	 * Tests that a package dependency cycle does not 
	 * exist for any of the analyzed packages.
	 */
	public void testAllPackagesCycles() {
		
		Collection packages = jdepend.analyze();

		assertEquals("Cycles exist", false, jdepend.containsCycles());
	}
	
	/**
	 * Tests that a package dependency constraint is matched
	 * for the analyzed packages.  
	 * <p>
	 * Fails if any package dependency other than those declared 
	 * in the dependency constraints are detected.
	 */
	public void testDependencyConstraint() {
		
		DependencyConstraint constraint = new DependencyConstraint();
		
		JavaPackage junitframework = constraint.addPackage("junit.framework");
		JavaPackage junitui = constraint.addPackage("junit.textui");
		JavaPackage framework = constraint.addPackage("jdepend.framework");
		JavaPackage text = constraint.addPackage("jdepend.textui");
		JavaPackage xml = constraint.addPackage("jdepend.xmlui");
		JavaPackage swing = constraint.addPackage("jdepend.swingui");
		
		framework.dependsUpon(junitframework);	
		framework.dependsUpon(junitui);	
		text.dependsUpon(framework);
		xml.dependsUpon(text);
		swing.dependsUpon(framework);
		
		jdepend.analyze();
		
		assertEquals("Constraint mismatch", 
			true, jdepend.dependencyMatch(constraint));
	}

	public static void main(String args[]) {
		junit.textui.TestRunner.run(ExampleTest.class);
	}
}
