/* EntropyTools.java -- static methods for generating secure PRNGs.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;

import java.security.Security;

import java.util.Arrays;
import java.util.HashMap;

import gnu.crypto.prng.ARCFour;
import gnu.crypto.prng.IRandom;
import gnu.crypto.prng.LimitReachedException;

final class EntropyTools
{

  // Constants.
  // -------------------------------------------------------------------------

  /**
   * A PRNG that can be used by any part of JESSIE that needs random, but
   * not necessarily secure, random numbers.
   */
  static final IRandom SYSTEM_PRNG = new SynchronizedRandom(getSeededRandom());

  private static IRandom fast_seed_prng = getSeededRandom();
  private static int fast_seed_count = 0;
  private static final int FAST_SEED_LIMIT = 65536;

  // Class methods.
  // -------------------------------------------------------------------------

  /**
   * Return a new RNG that has been seeded by a trusted source of entropy.
   *
   * @return The new instance.
   */
  static IRandom getSeededRandom()
  {
    ARCFour prng = new ARCFour();
    HashMap attrib = new HashMap();
    attrib.put(ARCFour.ARCFOUR_KEY_MATERIAL, generateSeed(128));
    prng.init(attrib);
    return prng;
  }

  /**
   * Generates a fast seed from an internal PRNG seeded with a more
   * secure source of entropy. If too many bytes are generated with
   * this method, the PRNG will be created anew with fresh entropy.
   *
   * @return The new seed.
   */
  static synchronized byte[] generateFastSeed(int nbytes)
  {
    byte[] buf = new byte[nbytes];
    generateFastSeed(buf);
    return buf;
  }

  static synchronized void generateFastSeed(byte[] buf)
  {
    generateFastSeed(buf, 0, buf.length);
  }

  static synchronized void generateFastSeed(byte[] buf, int off, int len)
  {
    try
      {
        fast_seed_prng.nextBytes(buf, off, len);
        fast_seed_count += len;
        if (fast_seed_count >= FAST_SEED_LIMIT)
          {
            fast_seed_prng = getSeededRandom();
          }
      }
    catch (LimitReachedException cannotHappen)
      {
        throw new Error(cannotHappen.toString());
      }
  }

  /**
   * Generates a seed based on a strong source of entropy. The entropy is
   * gathered either
   *
   * <ol>
   * <li>From a file, specified by the security property
   * "jessie.entropy.source". This could be, for example, the "/dev/random"
   * device on Linux.</li>
   * <li>Though a thread-timing algorithm. This method is rather slow, and
   * may not produce the most reliable random bits, so the file method is
   * recommended if available.</li>
   * </ol>
   *
   * @param nbytes The number of seed bytes to generate.
   * @return The seed.
   */
  static byte[] generateSeed(int nbytes)
  {
    if (nbytes < 0)
      {
        throw new IllegalArgumentException();
      }
    if (nbytes == 0)
      {
        return new byte[0];
      }

    byte[] buf = new byte[nbytes];
    try
      {
        String file = Security.getProperty("jessie.entropy.source");
        if (file != null)
          {
            FileInputStream fin = new FileInputStream(file);
            fin.read(buf);
            return buf;
          }
      }
    catch (Exception e)
      {
      }

    Runtime r = Runtime.getRuntime();
    byte[] entropy = new byte[128];
    for (int bits = 0; bits < 1024; bits += 32)
      {
        for (int i = 0; i < entropy.length; )
          {
            Spinner[] spinners = new Spinner[32];
            for (int j = 0; j < 32; j++)
              {
                spinners[j] = new Spinner(j);
              }
            try
              {
                Thread.sleep(100);
              }
            catch (InterruptedException ie)
              {
              }
            long t = System.currentTimeMillis();
            long m = r.freeMemory();
            for (int j = 0; j < 32; j++)
              {
                spinners[j].running = false;
                if (i < entropy.length)
                  {
                    entropy[i] ^= spinners[j].count;
                    entropy[i] ^= (byte) spinners[j].hashCode();
                    entropy[i] ^= (byte) (t >>> (i << 3 & 63));
                    entropy[i] ^= (byte) (m >>> (i << 3 & 63));
                  }
                ++i;
              }
          }
      }

    ARCFour prng = new ARCFour();
    HashMap attrib = new HashMap();
    attrib.put(ARCFour.ARCFOUR_KEY_MATERIAL, entropy);
    prng.init(attrib);
    try { prng.nextBytes(buf, 0, nbytes); }
    catch (LimitReachedException lre) { throw new Error(lre); }
    Arrays.fill(entropy, (byte) 0);
    return buf;
  }

  // Inner class.
  // -------------------------------------------------------------------------

  private static final class Spinner extends Thread
  {
    volatile boolean running;
    byte count;

    Spinner(int id)
    {
      super("spinner-" + id);
      running = true;
      count = (byte) (System.currentTimeMillis() ^ id);
      start();
    }

    public void run()
    {
      while (running)
        {
          count++;
          yield();
        }
    }
  }
}
