/* RecordInput.java -- record layer input thread.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;

import java.util.zip.DataFormatException;

import javax.net.ssl.SSLException;

class RecordInput extends Thread
{

  // Constants and fields.
  // -------------------------------------------------------------------------

  private static final boolean DEBUG_RECORD_LAYER = false;
  private static final PrintStream debug = System.err;

  private Session session;
  private final PipedInputStream applicationIn;
  private final PipedOutputStream applicationOut;
  private final PipedInputStream handshakeIn;
  private final PipedOutputStream handshakeOut;
  private final InputStream in;
  private boolean changeCipherSpec;
  private boolean running;
  private RecordOutput recordOutput;
  private boolean closeNotify;
  private int fragmentLength;
  private final Text inText, outText;

  // Constructor.
  // -------------------------------------------------------------------------

  RecordInput(InputStream in, Session session, ThreadGroup group)
  {
    super(group, "record_layer_in");
    setDaemon(true);
    this.in = in;
    this.session = session;
    try
      {
        applicationOut = new PipedOutputStream();
        applicationIn = new PipedInputStream(applicationOut);
        handshakeOut = new PipedOutputStream();
        handshakeIn = new PipedInputStream(handshakeOut);
      }
    catch (IOException ioe)
      {
        throw new Error(ioe);
      }
    changeCipherSpec = false;
    closeNotify = false;
    fragmentLength = 16384;
    inText = new Text(16384 + 2048);
    outText = new Text(16384);
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public void run()
  {
    if (DEBUG_RECORD_LAYER)
      {
        debug.println(currentThread() + " starting");
      }
    running = true;
    try
      {
        while (running)
          {
            ContentType type;
            byte[] frag;
            int len;
            synchronized (inText)
              {
                inText.read(in);
                if (DEBUG_RECORD_LAYER)
                  synchronized (debug)
                    {
                      debug.println(">> READ RECORD <<");
                      debug.println(inText);
                    }
              }
            if (isInterrupted())
              {
                Thread.dumpStack();
                return;
              }
            while (changeCipherSpec)
              {
                synchronized (session.params)
                  {
                    try
                      {
                        session.params.wait(200);
                      }
                    catch (InterruptedException ie) { }
                  }
              }
            synchronized (inText)
              {
                synchronized (outText)
                  {
                    session.params.decrypt(inText, outText);
                    frag = outText.getFragment();
                    len = outText.getLength();
                    type = outText.getType();
                  }
              }

            if (type == ContentType.APPLICATION_DATA)
              {
                applicationOut.write(frag, 0, len);
              }
            else if (type == ContentType.HANDSHAKE)
              {
                handshakeOut.write(frag, 0, len);
              }
            else if (type == ContentType.CHANGE_CIPHER_SPEC)
              {
                changeCipherSpec = true;
              }
            else if (type == ContentType.ALERT)
              {
                Alert alert =
                  Alert.read(new ByteArrayInputStream(outText.getFragment()));
                session.currentAlert = alert;
                if (alert.getDescription() == Alert.Description.CLOSE_NOTIFY)
                  {
                    closeNotify = true;
                    running = false;
                    while (applicationIn.available() > 0) yield();
                    applicationOut.close();
                    return;
                  }
                else if (alert.getLevel() == Alert.Level.FATAL)
                  {
                    handshakeOut.close();
                    applicationOut.close();
                    return;
                  }
              }
            else
              {
                throw new IOException("unknown type " + type);
              }
          }
      }
    catch (MacException me)
      {
        if (DEBUG_RECORD_LAYER)
          {
            me.printStackTrace(debug);
          }
        Alert alert = new Alert(Alert.Level.FATAL,
                                Alert.Description.BAD_RECORD_MAC);
        session.currentAlert = alert;
        session.invalidate();
        if (recordOutput != null)
          {
            try
              {
                recordOutput.sendAlert(alert);
              }
            catch (IOException ignore) { }
          }
        try
          {
            applicationOut.close();
            handshakeOut.close();
          }
        catch (IOException ignore) { }
      }
    catch (OverflowException oe)
      {
        if (DEBUG_RECORD_LAYER)
          {
            oe.printStackTrace(debug);
          }
        Alert alert = new Alert(Alert.Level.FATAL,
                                Alert.Description.RECORD_OVERFLOW);
        session.currentAlert = alert;
        session.invalidate();
        if (recordOutput != null)
          {
            try
              {
                recordOutput.sendAlert(alert);
              }
            catch (IOException ignore) { }
          }
        try
          {
            applicationOut.close();
            handshakeOut.close();
          }
        catch (IOException ignore) { }
      }
    catch (DataFormatException dfe)
      {
        if (DEBUG_RECORD_LAYER)
          {
            dfe.printStackTrace(debug);
          }
        Alert alert = new Alert(Alert.Level.FATAL,
                                Alert.Description.DECOMPRESSION_FAILURE);
        session.currentAlert = alert;
        session.invalidate();
        if (recordOutput != null)
          {
            try
              {
                recordOutput.sendAlert(alert);
              }
            catch (IOException ignore) { }
          }
        try
          {
            applicationOut.close();
            handshakeOut.close();
          }
        catch (IOException ignore) { }
      }
    catch (EOFException eofe)
      {
        if (DEBUG_RECORD_LAYER)
          {
            eofe.printStackTrace(debug);
          }
        if (!closeNotify)
          {
            session.invalidate();
            session.currentAlert =
              new Alert(Alert.Level.FATAL, Alert.Description.INTERNAL_ERROR);
          }
      }
    catch (IOException ioe)
      {
        if (DEBUG_RECORD_LAYER)
          {
            ioe.printStackTrace(debug);
          }
        session.invalidate();
        if (session.currentAlert == null)
          {
            session.currentAlert =
              new Alert(Alert.Level.FATAL, Alert.Description.INTERNAL_ERROR);
          }
      }
  }

  // Package methods.
  // -------------------------------------------------------------------------

  void setFragmentLength(int fragmentLength)
  {
    this.fragmentLength = fragmentLength;
    synchronized (inText)
      {
        inText.setCapacity(fragmentLength);
      }
    synchronized (outText)
      {
        outText.setCapacity(fragmentLength);
      }
  }

  void setRunning(boolean running)
  {
    this.running = running;
  }

  void setRecordOutput(RecordOutput recordOutput)
  {
    this.recordOutput = recordOutput;
  }

  Session getSession()
  {
    return session;
  }

  void setSession(Session session)
  {
    this.session = session;
  }

  InputStream getApplicationStream()
  {
    return applicationIn;
  }

  InputStream getHandshakeStream()
  {
    return handshakeIn;
  }

  boolean checkCloseNotify()
  {
    return closeNotify;
  }

  boolean checkChangeCipherSpec() throws SSLException
  {
    Alert alert = session.currentAlert;
    if (alert != null && alert.getLevel().equals(Alert.Level.FATAL))
      {
        throw new SSLException(alert.getDescription().toString());
      }
    return changeCipherSpec;
  }

  void clearChangeCipherSpec()
  {
    changeCipherSpec = false;
  }
}
