/* RecordOutput.java -- record layer output thread.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.IOException;
import java.io.EOFException;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

final class RecordOutput extends Thread
{

  // Constants and fields.
  // -------------------------------------------------------------------------

  private static final boolean DEBUG_RECORD_LAYER = false;
  private static final PrintStream debug = System.err;

  private OutputStream out;
  private Session session;
  private PipedInputStream applicationIn;
  private PipedOutputStream applicationOut;
  private PipedInputStream handshakeIn;
  private PipedOutputStream handshakeOut;
  private boolean running;
  private boolean sentCloseNotify;
  private int fragmentLength;
  private final Text inText, outText;
  private final List handshakeAvail;

  // Constructor.
  // -------------------------------------------------------------------------

  RecordOutput(OutputStream out, Session session, ThreadGroup group)
  {
    super(group, "record_layer_out");
    setDaemon(true);
    this.out = out;
    this.session = session;
    try
      {
        applicationOut = new PipedOutputStream();
        applicationIn = new PipedInputStream(applicationOut);
        handshakeOut = new PipedOutputStream();
        handshakeIn = new PipedInputStream(handshakeOut);
      }
    catch (IOException ioe)
      {
        throw new Error(ioe);
      }
    fragmentLength = 16384;
    inText = new Text(16384);
    outText = new Text(16384);
    handshakeAvail = Collections.synchronizedList(new ArrayList(5));
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public void run()
  {
    if (DEBUG_RECORD_LAYER)
      {
        debug.println(currentThread() + " starting");
      }
    running = true;
    try
      {
        while (running)
          {
            int len;
            if (handshakeAvail.size() > 0)
              {
                // Many other SSL implementations will go insane (NSS,
                // anyone?) if any handshake message is broken across
                // multiple records. We try to be careful here, then.
                synchronized (inText)
                  {
                    len = ((Integer) handshakeAvail.remove(0)).intValue();
                    len = Math.min(len, fragmentLength);
                    byte[] buf = inText.getFragment();
                    int count = 0;
                    while (count < len)
                      {
                        int l = handshakeIn.read(buf, count, len - count);
                        if (l == -1)
                          {
                            throw new EOFException("unexpected end of data");
                          }
                        count += l;
                      }
                    inText.setLength(len);
                    inText.setType(ContentType.HANDSHAKE);
                    inText.setVersion(session.protocol);
                    send(inText);
                  }
                continue;
              }

            if ((len = applicationIn.available()) > 0)
              {
                synchronized (inText)
                  {
                    len = Math.min(len, fragmentLength);
                    byte[] buf = inText.getFragment();
                    int count = 0;
                    while (count < len)
                      {
                        int l = applicationIn.read(buf, count, len - count);
                        if (l == -1)
                          {
                            throw new EOFException("unexpected end of data");
                          }
                        count += l;
                      }
                    inText.setLength(len);
                    inText.setType(ContentType.APPLICATION_DATA);
                    inText.setVersion(session.protocol);
                    send(inText);
                  }
              }
            else if (isInterrupted())
              {
                return;
              }
            else
              {
                yield();
                continue;
              }
          }
      }
    catch (IOException ioe)
      {
        try
          {
            if (DEBUG_RECORD_LAYER)
              {
                ioe.printStackTrace();
              }
            doClose();
          }
        catch (IOException ignored)
          {
          }
      }
  }

  // Package methods.
  // -------------------------------------------------------------------------

  void setHandshakeAvail(int handshakeAvail)
  {
    this.handshakeAvail.add(new Integer(handshakeAvail));
  }

  void setFragmentLength(int fragmentLength)
  {
    this.fragmentLength = fragmentLength;
    synchronized (inText)
      {
        inText.setCapacity(fragmentLength);
      }
    synchronized (outText)
      {
        outText.setCapacity(fragmentLength);
      }
  }

  void setRunning(boolean running)
  {
    this.running = running;
  }

  Session getSession()
  {
    return session;
  }

  void setSession(Session session)
  {
    this.session = session;
  }

  void sendAlert(Alert alert) throws IOException
  {
    synchronized (inText)
      {
        byte[] b = inText.getFragment();
        b[0] = (byte) alert.getLevel().getValue();
        b[1] = (byte) alert.getDescription().getValue();
        inText.setLength(2);
        inText.setType(ContentType.ALERT);
        inText.setVersion(session.protocol);
        send(inText);
      }
  }

  boolean applicationDataPending() throws IOException
  {
    return applicationIn.available() > 0;
  }

  void doClose() throws IOException
  {
    while (applicationIn.available() > 0)
      {
        Thread.yield();
      }
    if (!sentCloseNotify)
      {
        sendAlert(new Alert(Alert.Level.WARNING,
                            Alert.Description.CLOSE_NOTIFY));
        sentCloseNotify = true;
      }
  }

  void changeCipherSpec() throws IOException
  {
    while (handshakeIn.available() > 0);
    send(new Text(ContentType.CHANGE_CIPHER_SPEC,
                  session.protocol, new byte[] { 1 }));
  }

  OutputStream getApplicationStream()
  {
    return applicationOut;
  }

  OutputStream getHandshakeStream()
  {
    return handshakeOut;
  }

  private synchronized void send(Text text) throws IOException
  {
    session.params.encrypt(text, outText);
    outText.write(out);
    out.flush();
    if (DEBUG_RECORD_LAYER)
      {
        synchronized (debug)
          {
            debug.println(">> WROTE RECORD <<");
            debug.println(outText);
          }
      }
  }
}
