/*
 * Copyright (c) 2007-2008, Dennis M. Sosnoski. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.schema.codegen;

import org.jibx.schema.SchemaUtils;
import org.jibx.schema.elements.AnnotatedBase;

/**
 * Information for a global definition.
 */
public class DefinitionItem extends GroupItem
{
    /** Number of references to this definition. */
    private int m_referenceCount;
    
    /** Inlining not allowed flag. */
    private boolean m_inlineBlocked;
    
    /** Checked flag used by the code generation handling to track which definitions have already been processed, and
     also used during the classification pass. */
    private boolean m_checked;
    
    /**
     * Constructor for new top-level structure. Child structures should always be created using the containing
     * structure's {@link #addGroup(AnnotatedBase)} method.
     * 
     * @param comp schema component
     */
    public DefinitionItem(AnnotatedBase comp) {
        super(comp, null);
    }
    
    /**
     * Constructor from group. This supports replacing an embedded group with a definition, as needed when an embedded
     * group is used in multiple locations and cannot be inlined.
     * 
     * @param group
     */
    DefinitionItem(GroupItem group) {
        super(group, null, null);
    }

    /**
     * Get the number of references to this definition.
     *
     * @return count
     */
    public int getReferenceCount() {
        return m_referenceCount;
    }

    /**
     * Count a reference to this definition.
     */
    public void countReference() {
        m_referenceCount++;
    }

    /**
     * Check if inlining is blocked (due to non-singleton references).
     *
     * @return blocked
     */
    public boolean isInlineBlocked() {
        return m_inlineBlocked;
    }

    /**
     * Set inlining blocked flag.
     *
     * @param blocked
     */
    public void setInlineBlocked(boolean blocked) {
        m_inlineBlocked = blocked;
        if (blocked) {
            System.out.println("Setting inlining blocked for " + SchemaUtils.describeComponent(getSchemaComponent()));
        }
    }

    /**
     * Check if definition has been processed.
     *
     * @return checked
     */
    public boolean isChecked() {
        return m_checked;
    }

    /**
     * Set definition has been processed flag.
     *
     * @param checked
     */
    public void setChecked(boolean checked) {
        m_checked = checked;
    }
    
    /**
     * Classify the content of this item as attribute, element, and/or character data content. For a definition item,
     * this checks if the classification has already been done, and if not flags it done and invokes the superclass
     * handling.
     */
    public void classifyContent() {
        if (!m_checked) {
            m_checked = true;
            super.classifyContent();
        }
    }
    
    /**
     * Build a description of the item, including all nested items.
     *
     * @param depth current nesting depth
     * @return description
     */
    public String describe(int depth) {
        StringBuffer buff = new StringBuffer(depth + 50);
        buff.append(leadString(depth));
        if (isInline()) {
            buff.append("inlined ");
        }
        if (isEnumeration()) {
            buff.append("enumeration ");
        }
        buff.append("definition with ");
        buff.append(m_referenceCount);
        buff.append(" references");
        if (m_inlineBlocked) {
            buff.append(" (inlining blocked)");
        }
        buff.append(", and class name ");
        buff.append(getClassName());
        buff.append(": ");
        buff.append(SchemaUtils.describeComponent(getSchemaComponent()));
        buff.append('\n');
        buff.append(nestedString(depth));
        return buff.toString();
    }
}