//
// neuralnet.hh
//
// Made by Guillaume Stordeur
// Login   <kami@GrayArea.Masaq>
//
// Started on  Thu Aug  1 05:06:30 2002 Guillaume Stordeur
// Last update Mon May  5 21:05:11 2003 Guillaume Stordeur
//

#ifndef   	NEURALNET_HH_
# define   	NEURALNET_HH_

# include <cassert>
# include <string>
# include "neuron.hh"

# define __NN_VERSION "0.5 (July 2003)"
# define __NN_AUTHORS "Guillaume Stordeur <kamih@happycoders.org>"

namespace NeuralNet
{
  // Error functions
  typedef enum e_ErrorFunction
    {
      ERR_DIFF,
      ERR_TRIMMED_DIFF,
      ERR_SCALED_DIFF,
      ERR_ATANH
    } ErrorFunctionType;

  // Training algorithms
  typedef enum e_TrainingAlgorithm
    {
      ALG_NONE,
      ALG_GAUSS_NEWTON,
      ALG_BACKPROP,
      ALG_QUICKPROP,
      ALG_RPROP,
      ALG_TIMELAGGED
    } TrainingAlgorithm;

  typedef std::vector<Neuron*> t_layer;

  const std::string version(__NN_VERSION);

class NeuralNet
{
public:

  //	Constructor
  NeuralNet(const std::vector<unsigned int> &nb,// nb of neurons per layer
	    ActivationFunctionType type, 	// activation function type
	    ActivationFunctionType typeOut);

  //	Load from file Constructor
  NeuralNet(const std::string file);

  //	Destructor
  virtual ~NeuralNet();

  //	add a weighted connection between 2 neurons
  void	addConnection(unsigned int srcLayer,
		      unsigned int src,
		      unsigned int dstLayer,
		      unsigned int dst,
		      float weight);

  //	add a random-weighted connection between 2 neurons
  void	addConnection(unsigned int srcLayer,
		      unsigned int src,
		      unsigned int dstLayer,
		      unsigned int dst);

  //	get the network output for clamped input
  std::vector<float>	output(const std::vector<float> &inputs, int layer = -1);

  // Main training function
  float	trainNetwork(const std::vector<std::vector<float> > &inputs,
		     const std::vector<std::vector<float> > &desired,
		     bool batch,
		     TrainingAlgorithm algorithm);

  // Do Principal Component Analysis
  void  PCAinit(const std::string algo,
		const std::vector<std::vector<float> >& inputs,
		const std::vector<std::vector<float> >& desired);

  //	display info on the network
  virtual void	display() const;

  // Dump the network to file
  virtual void	outputFile(const std::string file) const;

  // Load a neural net description file
  virtual void	inputFile(const std::string file);

  //	dump a response graph
  void	graph2d1output(const std::string file);

  // Interface
  unsigned int getNbInputRecurrentNeurons() const
  {
    unsigned int r = 0;
    for (unsigned int i = 0; i < _layers[0].size(); ++i)
      if (_layers[0][i]->getRecurrent())
	r++;
    return r;
  }
  unsigned int getNbInputNeurons() const { return _layers[0].size() - getNbInputRecurrentNeurons(); }
  unsigned int getNbOutputNeurons() const { return _layers[_layers.size() - 1].size(); }
  float getLastLayerWeight(unsigned int neuron, unsigned int weight) const;
  void	setWeight(unsigned int layer, unsigned int neuron,
		  unsigned int weight, float value);
  float	getLRate(void) const { return _lRate; }
  float	getMomentum(void) const { return _moment; }
  ActivationFunctionType getOutType() const { return _typeOut; }

  void	setLRate(float lrate) { _lRate = lrate; }
  void	setMomentum(float moment) { _moment = moment; }
  void	setNPlus(float nplus) { _nPlus = nplus; }
  void	setNMinus(float nminus) { _nMinus = nminus; }
  void	setDeltaMax(float deltamax) { _deltaMax = deltamax; }
  void	setDeltaMin(float deltamin) { _deltaMin = deltamin; }
  void	setTri0(float tri0) { _tri0 = tri0; }
  void	setMu(float m) { _mu = m; }
  void	setWDecay(float d) { _wdecay = d; }
  void	setMaxIterations(unsigned int i) { _maxIter = i; }
  void	setSkip(unsigned int i) { _skip = i; }
  void	setChangeIterations(unsigned int i) { _changeIter = i; }
  void	setGaussNewton(int b) { _gaussNewton = b; }
  void	setErrorEpsilon(float e) { _epsilon = e; }
  void	setErrorFunction(ErrorFunctionType e) { _errfunc = e; }
  void	setFudge(float f) { _fudge = f; }
  void	resetOutputs() { _clear(OUTPUT); }
  // FIXME : for test only
  void	unsetFixed();
  
private:
  // protect against copy
  NeuralNet& operator=(const NeuralNet& rhs) { assert(0); }
  NeuralNet(const NeuralNet& rhs) { assert(0); }
protected: // Functions

  // For inheritance only
  NeuralNet() {}

  //	error function
  float	_errFunction(float dif);

  float	_computeError(const std::vector<std::vector<float> > &inputs,
		      const std::vector<std::vector<float> > &desired);

  void	_displayError(unsigned int iter, float err)
  {
    //std::cout << "Iter " << iter << " error: " << err << std::endl;
    std::cout << iter << " " << err << std::endl;
  }

  //	train the network on input/desired vectors
  float	_trainStochastic(const std::vector<std::vector<float> > &inputs,
			 const std::vector<std::vector<float> > &desired,
			 bool randPermutations,
			 TrainingAlgorithm algorithm);

  float	_trainBatch(const std::vector<std::vector<float> > &inputs,
		    const std::vector<std::vector<float> > &desired,
		    TrainingAlgorithm algorithm);

  Matrix<double>	_makeJ(Neuron *neuron, 
			       Matrix<double>& weights, 
			       Matrix<double>& exemple);
  
  float	_trainGaussNewton(const std::vector<std::vector<float> > &inputs,
			  const std::vector<std::vector<float> > &inter_exemple,
			  const std::vector<std::vector<float> > &desired);

  float	_trainNewton(const std::vector<std::vector<float> > &inputs,
		     const std::vector<std::vector<float> > &inter_exemple,
		     const std::vector<std::vector<float> > &desired);

  //	final weight change for batch mode
  void	_updateWeights(TrainingAlgorithm algorithm, float err, float lastErr);

  //	updates deltae and changes weights if in stochastic mode
  float	_backwardsPass(const std::vector<float>	&desiredRes, bool batch, TrainingAlgorithm algorithm);

  // Make a random permutation vector
  std::vector<unsigned int>	_make_permutation(unsigned int size);

  // Init function called by the constructors
  void	_init(const std::vector<unsigned int> &nb,
	      ActivationFunctionType type,
	      ActivationFunctionType typeOut);


  // Change a neuron's weight
  void	_changeWeight(unsigned int	layer,
		      unsigned int	neuron,
		      unsigned int	weight,
		      float		value)
  {
    assert(layer != 0 && layer < _layers.size());
    assert(neuron < _layers[layer].size());
    (_layers[layer])[neuron]->setWeight(weight, value);
  }

  //	clears stuff and sets stuff to initial values
  void	_clear(int flags);
  
 
protected: // Data
   
  //	ThresholdNeuron to be connected to all neurons'
  //	threshold connections
  ThresholdNeuron	*_thresholdNeuron;

  //	learning rate for backprop/quickprop
  float		_lRate;

  //	momentum coeff for backprop/quickprop
  float		_moment;

  //	mu coeff for quickprop
  float		_mu;

  //	deltae decay param
  float		_wdecay;

  //	rprop parameters
  float		_nPlus, _nMinus, _deltaMin, _deltaMax, _tri0;

  //	derivative fudge, for preventing zero updates
  float		_fudge;

  //	activation function type
  ActivationFunctionType	_type, _typeOut;

  //	error function type
  ErrorFunctionType		_errfunc;

  //	neuron layers
  std::vector<t_layer>	_layers;

  //	use  1 in every _skip lines of the training data
  unsigned int	_skip;

  //
  bool	_gaussNewton;

  //	training halt criteria:
  //	halts if iter >= _max_iter OR err <= _epsilon OR err hasnt
  //	changed since _change_iter iterations
  unsigned int	_maxIter, _changeIter;
  float		_epsilon;

};

} // end NeuralNet namespace

#endif	    /* !NEURALNET_HH_ */
