/*
** timelaggedneuron.hh
** Login : <nico@Altarion.marmotte.ath.cx>
** Started on  Tue Jul 22 15:32:18 2003 Nicolas
** $Id: timelaggedneuron.hh,v 1.11 2003/09/16 08:56:55 nico Exp $
**
** Copyright (C) 2003 Nicolas
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU Lesser General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef   	TIMELAGGEDNEURON_HH_
# define   	TIMELAGGEDNEURON_HH_

# include <cassert>
# include <list>
# include <cmath>

# include "pair.hh"
# include "neuron.hh"


namespace NeuralNet
{
  typedef std::vector<float>::iterator Iter;
  typedef std::vector<float>::const_iterator CIter;

  class TimeLaggedInputNeuron
  {
  public:
    // Ctor
    TimeLaggedInputNeuron(int size) { assert(size > 0); _size = size; }

  public:

    //	Set up the input
    void setInput(CIter& input_beg, CIter& input_end)
    {
      assert(std::distance(input_beg, input_end) == _size);
      _output.first = input_beg; _output.second = input_end;
    }

    void setInput(pair<CIter >& input)
    { _output = input; }
    void setInput(CIter& in)
    {
      _output.first = in;
      advance(in, _size);
      _output.second = in;
    }


    int getSize() { return _size; }

    //  Process inputs and calculate activation (output)
    pair<CIter> refreshOutput() {return _output;}
    pair<CIter> getOutput() const { return _output; }

    //	Get the derivative F'(s)
    float	getFPrime(void) const { return 1; }

    //	Display info to standard output
    void	display(void) const { std::cout << "TL Input Neuron" << std::endl;}

  protected:
    pair<CIter > _output;
    int		_size; // Size of the window...
  };

  class TimeLaggedTresholdInputNeuron : public TimeLaggedInputNeuron
  {
  public:

    TimeLaggedTresholdInputNeuron(int size):  TimeLaggedInputNeuron(size)
    {
      _voutput.resize(size);
      for (int i = 0; i < size; i++)
	_voutput[i] = 1.0;
      _output.second = _output.first = _voutput.begin();
      advance(_output.second, size);
    }

  protected:
    std::vector<float>	_voutput;
  };

  class TimeLaggedNeuron: public Neuron
  {
  public:
    // Ctor & Dtor
    TimeLaggedNeuron() { _inputNeurons.clear(); }
    virtual ~TimeLaggedNeuron() {}

  public:

    void addConnection(TimeLaggedInputNeuron* in);
    void addConnection(TimeLaggedInputNeuron* in, std::vector<float> weights);

    void updateBackpropStochastic(float	lRate, float moment, float delta);

    void updateBatch(float delta);
    
    // backprop
    void updateWeights(float lRate, float moment);
    // rprop
    void updateWeights(float nPlus, float nMinus,
		       float deltaMin, float deltaMax,
		       bool errUp);
    // quickprop
    void updateWeights(float lRate, float moment, float mu);


    float refreshOutput();

    float getFPrime() const;
    float getOutput() const { return _output; }

    float activation(float s) {  // sigmoid [0..1]
      // avoid overflow
      if (s > 45)
	return 1;
      else if (s < -45)
	return 0;
      else
	return (1 / (1 + expf(-s)));
    }

    std::vector<float>& getWeight(unsigned i)
    {  assert(i < _weights.size()); return _weights[i]; }

    TimeLaggedInputNeuron*	getInputNeuron(unsigned int i) const
    { assert(i < _inputNeurons.size()); return _inputNeurons[i]; }

    void display() const;
    void	clearset(int flags, float decay, float tri);

  protected:
    float	_fprime(unsigned i) const;

  protected:
    //  Neurons that have their output connected to this neuron
    std::vector<TimeLaggedInputNeuron*>		_inputNeurons;

    std::vector< std::vector<float> >	_weights, _momentum, _deltae;

    // For QuickPro && Rprop
    std::vector< std::vector<float> >   _oldDeltae, _tri;

//     float				_treshold;
//     float				_deltaTreshold;
//     float				_trMomentum;

    std::vector< std::list<float> >	_lagDeltae;

    float				_output, _s;
    std::list<float>			_activ;

  }; // class TimeLaggedNeuron

  // This Neuron is the Input neuron for Time Lagged Neural Network
  // Output is the input vector...
  // Works with iterators on the vector, instead of the vector itself...

}// !namespace NeuralNet


#endif	    /* !TIMELAGGEDNEURON_HH_ */
