/*
    Copyright (C) 2000 Steve Brown 
    Copyright (C) 2000,2001,2002 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: shadow.c,v 1.81 2002/09/09 19:32:24 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/src/shadow.c,v $
    $Date: 2002/09/09 19:32:24 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <string.h>
#include <shadow.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>

#ifdef HAVE_CONFIG_H
        #include "config.h"
#endif

#include "nss-mysql.h"

#ifdef INCLUDE_MY_SYS
        #include <my_pthread.h>
#endif

#include "lib.h"
#include "nss-shadow.h"
#include "parser.h"
#include "ent.h"

#if USE_SHADOW


NSS_STATUS _nss_mysql_getspnam_r (const char *, struct spwd *,char *, size_t,int *);
NSS_STATUS _nss_mysql_setspent (void);
NSS_STATUS _nss_mysql_endspent (void);
NSS_STATUS _nss_mysql_getspent_r (struct spwd *result,char *buffer, 
                size_t buflen, int *errnop);
static void prepare_forkhandler(void);
static void parent_forkhandler(void);
static void child_forkhandler(void);
static char * get_sql_query(struct query * q,struct parse_result * pr,
                struct mysql_auth * auth);

/* EOP */

static int forkhandler_isset = 0;
static pthread_mutex_t forkhandler_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;

/* get_sql_query:
 * 
 * Returns a complete SQL query string corresponding to the query
 * 
 * q: request:
 * pr: value gathered by the conf files parsing
 * auth: connection information
 */
char * get_sql_query(struct query * q,struct parse_result * pr,
                struct mysql_auth * auth) {
        struct shadowoptions * soptions = pr->soptions_p;
        char * sql = NULL;
        char * e_name=NULL;
        char buffer[64];
        int call_free;

        if (q->name) {
                /* looking by name */
                if (DEBUG)
                        _nss_mysql_log(LOG_ERR,"shadow.c: get_query "
                                        "called for user %s",q->name);
        } else {
                if (! q->bulk) {
                     _nss_mysql_log(LOG_ERR,"shadow.c: get_sql_query called "
                                     "with NAME == NULL and bulk == 0. Please "
                                     "report this bug.");
                }
        }
        
        /* escaping name */
        if (q->name)	{
                e_name = _nss_mysql_escape_string(q->name,auth,buffer,
                                sizeof buffer,&call_free);
                if (! e_name) 
                        return NULL;
        }


        if (q->name) {
                sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s,%s "
                                "from %s where %s='%s' and %s order by %s",
                        soptions->passwdcolumn,
                        soptions->usercolumn,
                        soptions->lastchange,
                        soptions->min,
                        soptions->max,
                        soptions->warn,
                        soptions->inact,
                        soptions->expire,
                        soptions->table,
                        soptions->usercolumn,
                        e_name,
                        soptions->where[0] ? soptions->where : "1=1",
                        soptions->usercolumn
                        );
                if (call_free)
                        free(e_name);
        } else {
                sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s,%s "
                                "from %s where %s order by %s",
                        soptions->passwdcolumn,
                        soptions->usercolumn,
                        soptions->lastchange,
                        soptions->min,
                        soptions->max,
                        soptions->warn,
                        soptions->inact,
                        soptions->expire,
                        soptions->table,
                        soptions->where[0] ? soptions->where : "1=1",
                        /* order by */
                        soptions->usercolumn
                        );
        }
        return sql;
}

/*
 * spw: to be filled
 * result: MySQL result
 * errnop: pointer to errno
 * buffer: buffer where we store the strings
 * buflen: size of buffer
 * max: the maximum number of rows that result should contain, ignored if 0
 *
 */

NSS_STATUS _nss_mysql_shadow_result_to_struct(struct spwd *spw,
                MYSQL_RES * result,int * errnop, char * buffer, size_t buflen,
                unsigned long max) {
        unsigned long rows;
        int error;
        MYSQL_ROW sql_row;
        MYSQL_ROW_OFFSET initial_offset = mysql_row_tell(result);

        rows = mysql_num_rows(result);

        if (rows == 0) {
                if (DEBUG) _nss_mysql_log(LOG_ERR,"user not found");
                return NSS_STATUS_NOTFOUND;
        } else if (max && rows > max) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct "
                                "has found %d matching rows, was expecting "
                                "%d. Fix your database.",
                                rows,max);
        }
        
        sql_row = mysql_fetch_row(result);

        if (!sql_row) {
                if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_"
                                "to_struct: mysql_fetch_row failed: end of "
                                "bulk retrieval");
                return NSS_STATUS_NOTFOUND;
        }

        if(_nss_mysql_isempty(sql_row[1])) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct:"
                                "empty user name field"
                                " Fix your database.");
                        return NSS_STATUS_UNAVAIL;
        }
        spw->sp_namp = _nss_mysql_copy_to_buffer(&buffer,&buflen,sql_row[1]);
        
        if (! spw->sp_namp)
                goto out_nomem;
                
        if (_nss_mysql_isempty(sql_row[0])) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "Password field for %s is empty or NULL. "
                                "Fix your database.",
                                spw->sp_namp);
                return NSS_STATUS_UNAVAIL;
        }

        spw->sp_pwdp = _nss_mysql_copy_to_buffer(&buffer,&buflen,sql_row[0]);
        if (! spw->sp_pwdp)
                goto out_nomem;

        spw->sp_lstchg = _nss_mysql_strtol(sql_row[2],time(NULL)-24*3600,
                        &error);
        if (error)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "lastchange field empty for %s. Reverting "
                                "to 'yesterday. Fix your database",
                                spw->sp_namp);
        spw->sp_min = _nss_mysql_strtol(sql_row[3],1,&error);
        if (error)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "min field empty for %s. Reverting to 1. Fix "
                                "your database",spw->sp_namp);
        spw->sp_max = _nss_mysql_strtol(sql_row[4],2,&error);
        if (error)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "max field empty for %s. Reverting to 2. Fix "
                                "your database", spw->sp_namp);
        spw->sp_warn = _nss_mysql_strtol(sql_row[5],7,&error);
        if (error)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "warn field empty for %s. Reverting to 7. Fix"
                                " your database",spw->sp_namp);
        
        spw->sp_inact = _nss_mysql_strtol(sql_row[6],-1,&error);
        if (error)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "inact field empty for %s. Reverting to -1. "
                                "Fix your database",spw->sp_namp);
        spw->sp_expire = _nss_mysql_strtol(sql_row[7],-1,&error);
        if (error)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_result_to_struct: "
                                "expire field empty for %s. Reverting to -1. "
                                "Fix your database",spw->sp_namp);

        spw->sp_flag = -1; /* unused atm */

        *errnop = 0;
        return NSS_STATUS_SUCCESS;

out_nomem:
        /* we want a bigger buffer, so we return ERANGE */
        *errnop = ERANGE;
        /* this row will be processed again */
        mysql_row_seek(result,initial_offset);
        return NSS_STATUS_TRYAGAIN;
}

/* getspnam
 * looks for an user by its name
 * Arguments:
 * uid: user's uid
 * spw: struct we'll fill
 * buf: buffer (not used)
 * buflen: sizeof(buffer)
 * errnop: ptr on the application errno
 */


static pthread_mutex_t nam_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;
static struct mysql_auth nam_auth = {NULL,0,&nam_mutex};

NSS_STATUS _nss_mysql_getspnam_r (const char * name, struct spwd *spw,
                char *buffer, size_t buflen,int * errnop) {
        struct shadowoptions soptions;
        struct parse_result pr = {NULL,NULL,&soptions};
        NSS_STATUS status;
        MYSQL_RES * result = NULL;
        char * sql_query = NULL;
        struct query q = {name,0,0,0};
        /*int call_mte = 0;*/

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_getspnam_r called for "
                                "%s",name);
        
        memset(&soptions,0,sizeof(struct shadowoptions));

        if (! _nss_mysql_read_conf_file("shadow",&pr)) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_getspnam_r conf file "
                                "parsing failed");
                status = NSS_STATUS_UNAVAIL;
                *errnop = ENOENT;
                goto out;
        }

        _nss_mysql_set_fork_handler(&forkhandler_isset,&forkhandler_mutex,
                        prepare_forkhandler,
                        parent_forkhandler,
                        child_forkhandler);

        if (! _nss_mysql_check_connection(&nam_auth,&soptions.con)) {
                status = NSS_STATUS_UNAVAIL;
                *errnop = ENOENT;
                goto out;
        }
       
        /*call_mte = 1;*/
        sql_query = get_sql_query(&q,&pr,&nam_auth);
        if (! sql_query) {
                *errnop = EAGAIN;
                status = NSS_STATUS_TRYAGAIN;
                /* we need to unlock the mutex because 
                 * _nss_mysql_check_connection locks it
                 */
                pthread_mutex_unlock(nam_auth.mutex);
                goto out;
        }
        
        status = _nss_mysql_send_query(&nam_auth,sql_query,&result,errnop);
        if (status != NSS_STATUS_SUCCESS)
                goto out;

        status = _nss_mysql_shadow_result_to_struct(spw,result,errnop,buffer,
                        buflen,1);
out:
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_getspnam_r: reached "
                                "\"out\" label, result %p",result);
        s_free(sql_query);
        if (result)
                mysql_free_result(result);
        _nss_mysql_free_shadow(&soptions);
        /*
        if (call_mte)
                my_thread_end();
        */
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_getspnam_r with "
                                "status %d (errno %s)",status,
                                strerror(*errnop));
        return status;
}

/* setspent
 * Initializes data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */

NSS_STATUS _nss_mysql_setspent (void) {
        return _nss_mysql_setent(SHADOW_ENT_TYPE,get_sql_query);
}

/* endspent
 * Kills all data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */
NSS_STATUS _nss_mysql_endspent (void) {
        return _nss_mysql_endent(SHADOW_ENT_TYPE);
}

/* getspent
 * Gets info for every group
 * Arguments:
 * pw: array to fill
 * buffer: buffer, unused 
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

NSS_STATUS _nss_mysql_getspent_r (struct spwd *spw,
                char * buffer, size_t buflen,int * errnop) {
        return _nss_mysql_getent_r(SHADOW_ENT_TYPE,spw,buffer,buflen,errnop);
}


void prepare_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"shadow.c: calling prepare_forkhandler");
        pthread_mutex_lock(nam_auth.mutex);
        pthread_mutex_lock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"shadow.c: end of prepare_forkhandler");
}

void parent_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"shadow.c: calling parent_forkhandler");
        pthread_mutex_unlock(nam_auth.mutex);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"shadow.c: end of parent_forkhandler");
}

void child_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"shadow.c: calling child_forkhandler");
        nam_auth.pid = 0;
        pthread_mutex_unlock(nam_auth.mutex);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"shadow.c: end of child_forkhandler");
}

#endif /* #if USE_SHADOW */

