/*
 * nss_sshsock: NSS module for providing NSS services from a remote
 * ssh server over a SSH socket.
 *
 * Copyright (C) 2011 Scott Balneaves <sbalneav@ltsp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "nss_sshsock.h"

/*
 * fgets_nonl:
 *
 * Uses fgets to grab a line of text.  If it successfully
 * finds a newline character, returns 1, else 0.
 */

int
fgets_nonl (char *s, int size, FILE *stream)
{
  char *ptr, *end;
  size_t len;

  *s = '\0'; /* NULL first byte */

  ptr = fgets (s, size, stream);
  len = strlen (s);

  if (len)
    {
      end = s + (len - 1);
      if ( *end == '\n')
        {
          *end = '\0';
          return 1;
        }
    }

  return 0;
}

/*
 * sshopen:
 *
 * Varargs function to allow for easy opening of the remote ssh command.
 */

FILE *
sshopen (char *remote)
{
  char cmd[BUFSIZ];
  char *socket, *host;

  if (sanity_check (&socket, &host) < 0)
    {
      return NULL;
    }

  snprintf (cmd, sizeof cmd,
            "NSS_SSHSOCK_SOCKET=\"\" NSS_SSHSOCK_HOST=\"\" ssh -S %s %s \"%s\"", socket, host, remote);

  /*
   * Call fflush before the popen command to make sure we're not
   * interfering with any buffered i/o currently in progress.
   */

  fflush (NULL);
  return popen (cmd, "r");
}

void
sshclose (FILE *p)
{
  pclose (p);
}

/*
 * delimit_array
 */

void
delimit_array (char **array, char *buffer, const char *delim)
{
  char **ap = array;   /* array pointer */
  char **bp = &buffer; /* buffer pointer */

  while ((*ap = strsep (bp, delim)) != NULL)
    {
      ap++;
    }
}

/*
 * split:
 * When passed a buffer that contains newline separated strings,
 * and a character to split on, allocate an array of character
 * pointers, and initialize to the beginnings of each line.  Any
 * occurance of the "delim" character is turned into a '\0'
 */

char **
split (char *buffer, const char *delim)
{
  char **array;
  char *p;
  size_t size = 0;

  /*
   * Take a quick pass through the buffer and count the number
   * of delim characters.
   */

  for (p = buffer; *p != '\0'; p++)
    {
      if (*p == *delim)
        {
          size++;
        }
    }

  /*
   * We'll need 2 more than the delim count: also want a null
   * terminator on the end of the array.
   */

  array = malloc (sizeof (char *) * (size + 2));

  if (!array)
    {
      return NULL;
    }

  delimit_array (array, buffer, delim);

  return array;
}

/*
 * sanity_check:
 * This function does some basic checks to make sure all the
 * external pieces we're relying on actually exist.
 */

int
sanity_check (char **socket, char **host)
{
  struct stat statbuf;

  *socket = getenv (SOCKENV);
  *host   = getenv (HOSTENV);

  /*
   * Check to make sure we've got a valid socket env var.
   */

  if (!(*socket))
    {
      return -1;
    }

  /*
   * Got a host environment var?
   */

  if (!*host)
    {
      return -1;
    }

  /*
   * file exists?
   */

  if (stat (*socket, &statbuf) < 0)
    {
      return -1;
    }

  /*
   * Socket?
   */

  if (!(S_ISSOCK (statbuf.st_mode)))
    {
      return -1;
    }

  return 0;  /* OK, passes sanity checks */
}
