(**
   Implements a textinput field. The string gadget is drag source and drop
   destination.

  TODO

  * Optimize redrawing to avoid flicker.

**)

MODULE VO:String;

(*
    Implements a string gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

(*
  Contributions:

  * introducing centered and right alignment. Beat Christen, 08-Dec-97

*)


IMPORT D   := VO:Base:Display,
       DD  := VO:Base:DragDrop,
       E   := VO:Base:Event,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       V   := VO:Model:Value,

       G   := VO:Object,

       CC  := CharClass,
       str := Strings;

CONST
  enteredMsg * = 0;
  escapeMsg  * = 1;

  leftAligned   * = 0;
  rightAligned  * = 1;
  centerAligned * = 2;

  (* mode *)
  normal*   = 0;
  password* = 1;

  passwordChar = "*";

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 gridDisable* : BOOLEAN;
                 baseline*    : BOOLEAN;
               END;

  Alignment* = LONGINT;

  String*     = POINTER TO StringDesc;
  StringDesc* = RECORD (G.GadgetDesc)
                  textAlign   : Alignment;
                  textWidth,
                  visWidth,
                  offset,
                  cursor,
                  markA,markB,
                  textPos     : LONGINT;
                  font        : D.Font;
                  string-     : V.ValueModel;
                  selected    : BOOLEAN;
                  readOnly    : BOOLEAN;
                  mode        : LONGINT;
                END;

  EnteredMsg*     = POINTER TO EnteredMsgDesc;
  EnteredMsgDesc* = RECORD (O.MessageDesc)
                    END;

  EscapeMsg*      = POINTER TO EscapeMsgDesc;
  EscapeMsgDesc*  = RECORD (O.MessageDesc)
                    END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
    p.gridDisable:=TRUE;
    p.baseline:=FALSE;
  END Init;

  PROCEDURE (s : String) Init*;

  BEGIN
    s.Init^;

    s.SetBackground(D.tableBackgroundColor);

    s.SetPrefs(prefs); (* We set the prefs *)

    s.SetFlags({G.canFocus});
    s.RemoveFlags({G.stdFocus});

    s.textWidth:=256;
    s.visWidth:=20;
    s.offset:=0;
    s.cursor:=0;
    s.markA:=-1;
    s.markB:=-1;
    s.selected:=FALSE;
    s.mode:=normal;
    s.string:=NIL;
    s.textAlign:=leftAligned;
    s.font:=D.normalFont;
    s.readOnly:=FALSE;
  END Init;

  (**
    Set a new font to be used by the string gadget.
  **)

  PROCEDURE (s : String) SetFont*(font : D.Font);

  BEGIN
    s.font:=font;
  END SetFont;

  PROCEDURE (s : String) SetMode*(mode : LONGINT);

  BEGIN
    s.mode:=mode;
  END SetMode;

  (**
    Assign a string model to string gadget.
  **)

  PROCEDURE (s : String) SetModel*(model : O.Model);

  BEGIN
    IF s.string#NIL THEN
      s.UnattachModel(s.string);
    END;
    IF (model#NIL) & (model IS V.ValueModel) THEN
      s.string:=model(V.ValueModel);
      s.AttachModel(model);
    ELSE
      s.string:=NIL;
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (s : String) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN s.string=m;
  END ModelAccepted;


  (**
    Set the maximum text with of string gadget and the visible with
    of the string gadget in letters.

    NOTE
    The with in letters will be estimated and thus the real width
    may differ from the given one.
  **)

  PROCEDURE (s : String) SetStringWidth*(width,visWidth : LONGINT);

  VAR
    help,help2 : U.Text;

  BEGIN
    IF (s.string#NIL) & ~s.string.IsNull() THEN
      NEW(help,width);
      help2:=s.string.GetText();
      COPY(help2^,help^);
      s.string.SetText(help);
    END;
    s.textWidth:=width;
    s.visWidth:=visWidth;
  END SetStringWidth;

  (**
    Set the aligment for the text within the string object.
  **)

  PROCEDURE (s : String) SetStringAlignment*(a:Alignment);
  BEGIN
    s.textAlign:=a;
  END SetStringAlignment;

  PROCEDURE (s : String) SetReadOnly*(readOnly : BOOLEAN);
  
  BEGIN
    s.readOnly:=readOnly;
    s.Redraw;
  END SetReadOnly;

  PROCEDURE (s : String) CalcSize*;

  VAR
    font : D.Font;

  BEGIN
    font:=s.font;

    s.width:=s.visWidth*D.display.spaceWidth;
    s.height:=font.height;

    s.minWidth:=s.width;
    s.minHeight:=s.height;

    s.CalcSize^;
  END CalcSize;

  PROCEDURE (s : String) ConvertToPassword(text : U.Text);

  VAR
    x : LONGINT;

  BEGIN
    IF text#NIL THEN
      x:=0;
      WHILE (text[x]#0X) DO
        text[x]:=passwordChar;
        INC(x);
      END;
    END;
  END ConvertToPassword;

  (**
    Sets the cursor under the mouse.
  **)

  PROCEDURE (s : String) GetCursorPos(x : LONGINT):LONGINT;

  VAR
    y,
    widthA,
    widthB  : LONGINT;
    help    : U.Text;
    extent  : D.FontExtentDesc;
    found   : BOOLEAN;
    font    : D.Font;

  BEGIN
    font:=s.font;

    IF x<s.textPos THEN
      RETURN 0;
    END;
    DEC(x,s.textPos);

    help:=s.string.GetTextCopy();

    IF s.mode=password THEN
      s.ConvertToPassword(help);
    END;

    found:=FALSE;
    y:=0;
    WHILE ~found & (y<s.string.GetTextLength()) DO (* TODO: Optimize *)
      font.TextExtent(help^,y,{},extent);
      widthA:=extent.width;
      font.TextExtent(help^,y+1,{},extent);
      widthB:=extent.width;
      IF (widthA<=x) & (x<=widthB) THEN
        IF (x-widthA)>(widthB-widthA) DIV 2 THEN
          INC(y);
        END;
        found:=TRUE;
      ELSE
        INC(y);
      END;
    END;

    RETURN y;
  END GetCursorPos;

  PROCEDURE (s : String) Selected():BOOLEAN;

  BEGIN
    RETURN s.markA>=0;
  END Selected;

  PROCEDURE (s : String) SetSelection(a,b : LONGINT);

  BEGIN
    IF D.display.RegisterSelection(s,s.GetWindow()) THEN
      s.markA:=a;
      s.markB:=b;
    END;
  END SetSelection;

  PROCEDURE (s : String) ClearSelection;

  BEGIN
    IF s.Selected() THEN
      s.markA:=-1;
      s.markB:=-1;
      D.display.CancelSelection;
    END;
  END ClearSelection;

  PROCEDURE (s : String) DeleteSelection;

  VAR
    a,b : LONGINT;

  BEGIN
    IF s.Selected() THEN
      a:=s.markA;
      b:=s.markB;
      s.ClearSelection;
      s.string.Delete(a,b-a);
    END;
  END DeleteSelection;

  (* ---- Drag and drop stuff *)

  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (s : String) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  BEGIN
    IF s.visible & s.PointIsIn(x,y) & (s.string#NIL) & ~s.string.IsNull()
    & (s.mode#password)
    & (drag OR ~s.disabled) THEN
      (* we can drag and drop *)
      RETURN s;
    ELSE
      RETURN NIL;
    END;
  END GetDnDObject;

  (**
    Return a list of supported datatypes.
  **)

  PROCEDURE (s : String) GetDragInfo*(VAR dragInfo : DD.DnDDataInfo);

  BEGIN
    dragInfo.AddDataType(DD.text,DD.none,{DD.copy,DD.move,DD.insert},DD.copy);
  END GetDragInfo;

  (**
    The object gets a list of supported datatypes of the drag object and
    has to choose one of them. If there is no fitting datatype it must return
    FALSE.
  **)

  PROCEDURE (s : String) GetDropDataType*(VAR dragInfo : DD.DnDDataInfo;
                                          VAR group, type, action : LONGINT):BOOLEAN;

  BEGIN
    group:=DD.text;
    type:=DD.joker;
    RETURN dragInfo.FindDataType(group,type,action);
  END GetDropDataType;

  (**
    Called if we want to get the drag & drop data from the object.
    data will we garanted to be initialized.
  **)

  PROCEDURE (s : String) GetDragData*(group, type, action : LONGINT):DD.DnDData;

  VAR
    data : DD.DnDStringData;
    text : U.Text;

  BEGIN
    IF (group=DD.text) & (s.string#NIL) & ~s.string.IsNull() & (s.mode#password) THEN
      NEW(data);
      IF s.Selected() THEN
        NEW(data.string,s.markB-s.markA+2);
        text:=s.string.GetText();
        str.Extract(text^,SHORT(s.markA),SHORT(s.markB-s.markA),data.string^);
        data.string[s.markB-s.markA]:=0X;

        IF action=DD.move THEN
          s.string.Delete(SHORT(s.markA),SHORT(s.markB-s.markA));
        END;
      ELSE
        data.string:=s.string.GetTextCopy();
        IF action=DD.move THEN
          s.string.SetString("");
        END;
      END;
      RETURN data;
    ELSE
      RETURN NIL;
    END;
  END GetDragData;

  PROCEDURE (s : String) HandleDrop*(data : DD.DnDData; action : LONGINT):BOOLEAN;

  VAR
    x : LONGINT;

  BEGIN
    WITH data : DD.DnDStringData DO

      x:=0;
      WHILE data.string[x]#0X DO
        IF CC.IsControl(data.string[x]) THEN
          RETURN FALSE;
        END;
        INC(x);
      END;

      (*
        TODO: Exchange selection if selection exists
      *)

      IF s.string.GetTextLength()+LEN(data.string^)-1>s.textWidth THEN
        RETURN FALSE;
      END;

      CASE action OF
        DD.insert:
          INC(s.cursor,LEN(data.string^)-1);
          s.string.Insert(data.string^,s.cursor-LEN(data.string^)+1);
      | DD.copy,
        DD.move:
          s.string.SetString(data.string^);
      ELSE
        RETURN FALSE;
      END;
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END HandleDrop;

  PROCEDURE (s : String) DrawText;

  VAR
    x,
    width,
    cursorPos,
    markA,
    markB     : LONGINT;
    help      : U.Text;
    help2     : ARRAY 2 OF CHAR;
    extent1,
    extent2   : D.FontExtentDesc;
    string    : U.Text;
    font      : D.Font;
    draw      : D.DrawInfo;

  BEGIN
    font:=s.font;
    draw:=s.GetDrawInfo();

    (* x = left starting position of string, y = top position *)
    x:=s.x+D.display.spaceWidth DIV 2;
    width:=s.width-D.display.spaceWidth;

    IF s.disabled & ~s.prefs(Prefs).gridDisable THEN
      draw.PushForeground(D.backgroundColor);
      draw.FillRectangle(s.x,s.y,s.width,s.height);
      draw.PopForeground;
    ELSE
      s.DrawBackground(s.x,s.y,s.width,s.height);
    END;

    IF s.prefs(Prefs).baseline THEN
      IF s.disabled THEN
        draw.PushForeground(D.disabledColor);
      ELSE
        draw.PushForeground(D.textColor);
      END;

      draw.DrawLine(x,s.y+font.ascent-1,x+width-1,s.y+font.ascent-1);

      draw.PopForeground;
    END;

    IF s.string#NIL THEN
      IF s.string.IsNull() THEN
        NEW(string,1);
        string[0]:=0X;
      ELSE
        IF s.mode=password THEN
          string:=s.string.GetTextCopy();
          s.ConvertToPassword(string);
        ELSE
          string:=s.string.GetText();
        END;
      END;
    ELSE
      string:=NIL;
    END;

    IF string#NIL THEN
      draw.InstallClip(s.x,s.y,s.width,s.height);


      IF s.cursor>str.Length(string^) THEN
        s.cursor:=str.Length(string^);
      END;

      IF s.textAlign = leftAligned THEN
        s.textPos:=x;
      ELSIF s.textAlign = rightAligned THEN
        s.textPos:=x+width -font.TextWidth(string^,str.Length(string^), {});
      ELSE (* center *)
        s.textPos:=x+(width -font.TextWidth(string^, str.Length(string^), {})) DIV 2;
      END;

      (* Copy string from 0 before cursor to help *)
      NEW(help,s.cursor+1);
      str.Extract(string^,0,SHORT(s.cursor),help^);

      (* calculate bounds of help^ *)
      font.TextExtent(help^,s.cursor,{},extent1);

      (* Calculate bound of first letter of help^ *)
      help2[0]:=string[0];
      help2[1]:=0X;
      font.TextExtent(help2,1,{},extent2);
      (*
        correct starting position of string by left hand
        space of starting character
       *)
      DEC(s.textPos,extent2.lbearing);

      (* Calculate cursor position *)
      cursorPos:=s.textPos+extent1.width;

      (* correct cursor position by first character of 2nd string *)
      IF s.cursor<str.Length(string^) THEN
        help2[0]:=string[s.cursor];
        font.TextExtent(help2,1,{},extent2);
        INC(cursorPos,extent2.lbearing);
      END;

      (* Make cursor visible in gadget by correcting starting offset *)
      IF cursorPos-s.offset>x+width-2 THEN
        INC(s.offset,cursorPos-s.offset-(x+width-2));
      ELSIF cursorPos-s.offset<x THEN
        DEC(s.offset,x-cursorPos+s.offset);
      END;

      (* correct textstart and cursorpos, too *)
      DEC(s.textPos,s.offset);
      DEC(cursorPos,s.offset+1);

      IF s.Selected() THEN
        (* Copy string from 0 before cursor to help *)
        NEW(help,s.markA+1);
        str.Extract(string^,0,SHORT(s.markA),help^);

        (* calculate bounds of help^ *)
        font.TextExtent(help^,s.markA,{},extent1);

        (* Calculate cursor position *)
        markA:=s.textPos+extent1.width;

        (* correct cursor position by first character of 2nd string *)
        IF s.markA<str.Length(string^) THEN
          help2[0]:=string[s.markA];
          font.TextExtent(help2,1,{},extent2);
          INC(markA,extent2.lbearing);
        END;


        (* Copy string from 0 before cursor to help *)
        NEW(help,s.markB+1);
        str.Extract(string^,0,SHORT(s.markB),help^);

        (* calculate bounds of help^ *)
        font.TextExtent(help^,s.markB,{},extent1);

        (* Calculate cursor position *)
        markB:=s.textPos+extent1.width;

        (* correct cursor position by first character of 2nd string *)
        IF s.markB<str.Length(string^) THEN
          help2[0]:=string[s.markB];
          font.TextExtent(help2,1,{},extent2);
          INC(markB,extent2.lbearing);
        END;

        DEC(markA,s.offset+1);
        DEC(markB,s.offset+1);

        draw.PushForeground(D.fillColor);
        draw.FillRectangle(markA,s.y,markB-markA+1,s.height);
        draw.PopForeground;
      END;

      (* Draw the string *)
      draw.PushFont(s.font,{});
      IF s.disabled & ~s.prefs(Prefs).gridDisable THEN
        draw.PushForeground(D.disabledColor);
      ELSE
        draw.PushForeground(D.textColor);
      END;

      draw.DrawString(s.textPos,s.y+font.ascent,string^,str.Length(string^));

      draw.PopForeground;
      draw.PopFont;

      (* Drawing the cursor *)
      IF s.selected THEN
        draw.PushDrawMode(D.invert);
        draw.PushForeground(D.textColor);
        draw.DrawLine(cursorPos,s.y,cursorPos,s.y+font.height-1);
        draw.PopForeground;
        draw.PopDrawMode;
      END;

      IF s.disabled & s.prefs(Prefs).gridDisable THEN
        s.DrawDisabled;
      END;

      draw.FreeLastClip;
    END;
  END DrawText;

  (**
    Clear the current selection.
  **)

  PROCEDURE (s : String) Deselect*;

  BEGIN
    s.ClearSelection;
    IF s.visible THEN
      s.DrawText;
    END;
  END Deselect;

  PROCEDURE (s : String) HandleMouseEvent*(event : E.MouseEvent;
                                           VAR grab : G.Object):BOOLEAN;

  VAR
    help : LONGINT;

  BEGIN
    IF ~s.visible OR (s.string=NIL) THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & s.PointIsIn(event.x,event.y)
      & (event.button=E.button1) THEN
        s.selected:=TRUE;
        s.cursor:=s.GetCursorPos(event.x);
        s.ClearSelection;
        s.DrawText;

        grab:=s;
        RETURN TRUE;
      ELSIF (event.type=E.mouseDown) & (event.button=E.dragDropButton) THEN
        s.selected:=TRUE;
        s.cursor:=s.GetCursorPos(event.x);
        s.ClearSelection;
        s.DrawText;
        IF ~D.display.QuerySelection(s.GetWindow(),s,D.text) THEN END;

        grab:=s;
        RETURN TRUE;
      ELSIF (event.button=E.button1) & (event.type=E.mouseUp) THEN
        help:=s.GetCursorPos(event.x);
        IF help>s.cursor THEN
          s.SetSelection(s.cursor,help);
          s.cursor:=help;
          s.DrawText;
        ELSIF help<s.cursor THEN
          s.SetSelection(help,s.cursor);
          s.DrawText;
        END;

        grab:=NIL;
        RETURN TRUE;
      ELSIF (event.type=E.mouseUp) & (event.button=E.dragDropButton) THEN
        IF ~D.display.QuerySelection(s.GetWindow(),s,D.text) THEN
        END;

        grab:=NIL;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF event.qualifier={E.button1} THEN
        help:=s.GetCursorPos(event.x);
        IF help>s.cursor THEN
          s.SetSelection(s.cursor,help);
        ELSIF help<s.cursor THEN
          s.SetSelection(help,s.cursor);
        ELSE
          s.ClearSelection;
        END;
        s.DrawText;

        RETURN TRUE;
      END;
    ELSE
    END;
    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (s : String) DrawFocus*;

  BEGIN
    s.DrawText;
  END DrawFocus;

  PROCEDURE (s : String) HideFocus*;

  BEGIN
    s.DrawText;
  END HideFocus;
  
  PROCEDURE (s : String) RecatchedFocus*;
  
  BEGIN
    s.selected:=TRUE;
    
    s.RecatchedFocus^;
  END RecatchedFocus;
  
  PROCEDURE (s : String) CatchedFocus*;

  VAR
    length : LONGINT;

  BEGIN
    IF (s.string#NIL) & ~s.selected THEN
      length:=s.string.GetTextLength();
      IF length>0 THEN
        s.SetSelection(0,length);
        s.cursor:=0;
      END;  
    END;  

    s.selected:=TRUE;

    s.CatchedFocus^;
  END CatchedFocus;

  PROCEDURE (s : String) LostFocus*;
  
  BEGIN
    s.selected:=FALSE;
    s.LostFocus^;
  END LostFocus;


  PROCEDURE (s : String) HandleKeys(event :  E.KeyEvent):BOOLEAN;

  VAR
    entered : EnteredMsg;
    escape  : EscapeMsg;

  BEGIN
    CASE event.key OF
      E.return:
        IF ~s.HasFocus() THEN
          s.selected:=FALSE;
          s.DrawText;
        ELSE
          s.LeaveFocus;
        END;

        NEW(entered);
        s.Send(entered,enteredMsg);
    | E.escape:
        IF ~s.HasFocus() THEN
          s.selected:=FALSE;
          s.DrawText;
        END;

        NEW(escape);
        s.Send(escape,escapeMsg);
    | E.left:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor>0 THEN
            IF ~s.Selected() THEN
              DEC(s.cursor);
              s.SetSelection(s.cursor,s.cursor+1);
              s.DrawText;
            ELSIF (s.markA+1=s.markB) & (s.markB=s.cursor) THEN
              s.ClearSelection;
              DEC(s.cursor);
              s.DrawText;
            ELSIF s.cursor=s.markA THEN
              DEC(s.cursor);
              s.SetSelection(s.cursor,s.markB);
              s.DrawText;
            ELSIF s.cursor=s.markB THEN
              DEC(s.cursor);
              s.SetSelection(s.markA,s.cursor);
              s.DrawText;
            END;
          END;
        ELSE
          s.ClearSelection;
          IF s.cursor>0 THEN
            DEC(s.cursor);
          END;
          s.DrawText;
        END;
    | E.right:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor<s.string.GetTextLength() THEN
            IF ~s.Selected() THEN
              INC(s.cursor);
              s.SetSelection(s.cursor-1,s.cursor);
              s.DrawText;
            ELSIF (s.markA+1=s.markB) & (s.markA=s.cursor) THEN
              s.ClearSelection;
              INC(s.cursor);
              s.DrawText;
            ELSIF s.cursor=s.markB THEN
              INC(s.cursor);
              s.SetSelection(s.markA,s.cursor);
              s.DrawText;
            ELSIF s.cursor=s.markA THEN
              INC(s.cursor);
              s.SetSelection(s.cursor,s.markB);
              s.DrawText;
            END;
          END;
        ELSE
          s.ClearSelection;
          IF s.cursor<s.string.GetTextLength() THEN
            INC(s.cursor);
          END;
          s.DrawText;
        END;
    | E.home:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor#0 THEN
            s.SetSelection(0,s.cursor);
          END;
          s.cursor:=0;
          s.DrawText;
        ELSE
          s.ClearSelection;
          s.cursor:=0;
          s.DrawText;
        END;
    | E.end:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor#s.string.GetTextLength() THEN
            s.SetSelection(s.cursor,s.string.GetTextLength());
          END;
          s.cursor:=s.string.GetTextLength();
          s.DrawText;
        ELSE
          s.ClearSelection;
          s.cursor:=s.string.GetTextLength();
          s.DrawText;
        END;
    | E.backspace:
        IF ~s.disabled & ~ s.readOnly THEN
          IF s.Selected() THEN
            s.cursor:=s.markA;
            s.DeleteSelection;
          ELSIF s.cursor>0 THEN
            s.ClearSelection;
            DEC(s.cursor);
            s.string.Delete(s.cursor,1);
          END;
        END;  
    | E.delete:
        IF ~s.disabled & ~s.readOnly THEN
          IF s.Selected() THEN
            s.cursor:=s.markA;
            s.DeleteSelection;
          ELSIF s.cursor<s.string.GetTextLength() THEN
            s.ClearSelection;
            s.string.Delete(s.cursor,1);
          END;
        END;  
    | E.insert:
        IF ~s.disabled & ~s.readOnly THEN
          IF s.Selected() THEN
            s.DeleteSelection;
          END;
          IF ~D.display.QuerySelection(s.GetWindow(),s,D.text) THEN
          END;
        END;  
    ELSE
      IF ~s.disabled & ~s.readOnly & (s.string#NIL) THEN
        IF (event.textLength>0) & ~CC.IsControl(event.text[0]) THEN
          IF s.string.GetTextLength()+event.textLength<=s.textWidth THEN
            s.DeleteSelection;
            INC(s.cursor,event.textLength);
            s.string.Insert(event.text^,s.cursor-event.textLength);
          ELSE
            RETURN FALSE;
            D.display.Beep;
          END;
        ELSE
          IF event.text[0]#0X THEN
            (*Err.String(buffer); Err.Ln;
            Err.Hex(keysym,8); Err.Ln;*)
            (*D.display.Beep;*)
            RETURN FALSE;
          END;
        END;
      END;
    END;
    RETURN TRUE;
  END HandleKeys;

  PROCEDURE (s : String) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    IF event.type=E.keyDown THEN
      RETURN s.HandleKeys(event);
    ELSE
      RETURN FALSE;
    END;
  END HandleKeyEvent;

  PROCEDURE (s : String) Draw*(x,y,w,h : LONGINT);

  BEGIN
    s.Draw^(x,y,w,h);

    IF ~s.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    s.DrawText;
  END Draw;

  PROCEDURE (s : String) Hide*;

  BEGIN
    IF s.visible THEN
      s.DrawHide;
      s.ClearSelection;
      s.Hide^;
    END;
  END Hide;

  PROCEDURE (s : String) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF s.visible THEN
      s.DrawText;
    END;
  END Resync;

  PROCEDURE CreateString*():String;

  VAR
    string : String;

  BEGIN
    NEW(string);
    string.Init;

    RETURN string;
  END CreateString;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:String.