/***************************************************************************
 $RCSfile: conf.cpp,v $
                             -------------------
    cvs         : $Id: conf.cpp,v 1.5 2003/07/02 18:22:24 aquamaniac Exp $
    begin       : Tue Dec 13 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : martin@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/




/*
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif

#include "openhbci/error.h"
#include "openhbci/parser.h"
#include "openhbci/conf.h"
#include <stdio.h> // DEBUG


namespace HBCI {

Tree<ConfigNode>::const_iterator 
Config::_findGroup(string name,
		     Tree<ConfigNode>::const_iterator where) const {
  if (!where.isValid())
    return where;
  where.child();
  while(where.isValid()) {
    if ((*where).type==CONFIG_TYPE_GROUP)
      if (-1<parser::cmpPattern((*where).data,
                                  name,
                                  !(_mode & CONFIG_MODE_IGNORE_GROUP_CASE)))
        break;
    where++;
  } // while
  return where;
}

Tree<ConfigNode>::iterator Config::_findGroup(string name,
                                                    Tree<ConfigNode>::iterator where){
  if (!where.isValid())
    return where;
  where.child();
  while(where.isValid()) {
    if ((*where).type==CONFIG_TYPE_GROUP)
      if (-1<parser::cmpPattern((*where).data,
                                  name,
                                  !(_mode & CONFIG_MODE_IGNORE_GROUP_CASE)))
        break;
    where++;
  } // while
  return where;
}


Tree<ConfigNode>::const_iterator 
Config::_findVariable(string name,
			Tree<ConfigNode>::const_iterator where) const {
  if (!where.isValid())
    return where;
  where.child();
  while(where.isValid()) {
    if ((*where).type==CONFIG_TYPE_VAR)
      if (-1<parser::cmpPattern((*where).data,
                                  name,
                                  !(_mode & CONFIG_MODE_IGNORE_VAR_CASE)))
        break;
    where++;
  } // while
  return where;
}

Tree<ConfigNode>::iterator Config::_findVariable(string name,
                                                       Tree<ConfigNode>::iterator where){
  if (!where.isValid())
    return where;
  where.child();
  while(where.isValid()) {
    if ((*where).type==CONFIG_TYPE_VAR)
      if (-1<parser::cmpPattern((*where).data,
                                  name,
                                  !(_mode & CONFIG_MODE_IGNORE_VAR_CASE)))
        break;
    where++;
  } // while
  return where;
}


Tree<ConfigNode>::iterator Config::_addGroup(string name,
                                                   Tree<ConfigNode>::iterator where){
  if (!where.isValid())
    return where;
  // check if "where" is a group or root
  if ((*where).type!=CONFIG_TYPE_GROUP &&
      (*where).type!=CONFIG_TYPE_ROOT)
    // return invalid iterator if not
    return Tree<ConfigNode>::iterator();
  // simply add the child and let the iterator select it
  where.addChild(ConfigGroup(name),false,true);
  // return this iterator with the new position
  return where;
}


Tree<ConfigNode>::iterator Config::_addVariable(string name,
                                                      Tree<ConfigNode>::iterator where){
  if (!where.isValid()) {
    return where;
  }
  // check if "where" is a group or root
  if ((*where).type!=CONFIG_TYPE_GROUP &&
      (*where).type!=CONFIG_TYPE_ROOT &&
      (*where).type!=CONFIG_TYPE_VAR)
    // return invalid iterator if not
    return Tree<ConfigNode>::iterator();
  // simply add the child and let the iterator select it
  where.addChild(ConfigVariable(name),false,true);
  // return this iterator with the new position
  return where;
}


Error Config::_parseGroup(string &s,
                              Tree<ConfigNode>::iterator &curr){
  string n;
  unsigned int pos;
  Error err;
  Tree<ConfigNode>::iterator tmpit;
  string prepath;
  string postpath;

  pos=0;
  // skip leading blanks
  while(pos<s.length()){
    if (s.at(pos)>32)
      break;
    pos++;
  }
  if (pos>=s.length())
    return Error("Config::_parseGroup()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "empty group name");

  // check for introducing '['
  if (s.at(pos)!='[')
    return Error("Config::_parseGroup()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "'[' must be the first character");

  // skip '['
  pos++;

  // read the name
  n.erase();
  err=parser::getString(s,
                          n,
                          "]#",
                          "\"\"''",  // nest chars
                          pos,
                          1024);
  if (!err.isOk())
    return err;
  // post process the name
  parser::processString(n,_mode);
  if (n.empty())
    return Error("Config::_parseGroup()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "empty group name after processing");

  // check for trailing ']'
  if (pos>=s.length())
    // is no char left, but we need at least a ']' following, so: error
    return Error("Config::_parseGroup()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "']' must follow group name");
  if (s.at(pos)!=']')
    return Error("Config::_parseGroup()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "']' must follow group name");

  tmpit=createGroup(n,curr);
  if (!tmpit.isValid())
    return Error("Config::_parseGroup()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "could not create group ",
                   n);
  curr=tmpit;
  return Error();
}


Error Config::_parseValues(string &s,
                               unsigned int pos,
                               Tree<ConfigNode>::iterator var){
  bool goon;
  Error err;
  string value;

  if (!var.isValid())
    return Error("Config::_parseValues()",
                   ERROR_LEVEL_INTERNAL,
                   0,
                   ERROR_ADVISE_IGNORE,
                   "where should I add values ?");

  goon=true;
  while(goon) {
    // read value
    value.erase();
    err=parser::getString(s,
			    value,
			    ",#",
			    "\"\"",
			    pos,
			    8192);
    if (!err.isOk())
      return err;

    // post process value
    err=parser::processString(value,_mode);
    if (!err.isOk())
      return err;

    // add value
    if (!value.empty() ||
        _mode & CONFIG_MODE_ALLOW_EMPTY_VALUES)
      var.addChild(ConfigValue(value));

    // skip blanks
    while(pos<s.length()) {
      if (s.at(pos)>32)
        break;
      pos++;
    }

    // check for end of loop
    goon=false;
    if (pos<s.length())
      if (s.at(pos)==',') {
        pos++;
        goon=true;
      }
  } // while goon
  return Error();
}


Error Config::_parseVar(string &s,
                            Tree<ConfigNode>::iterator where){
  string n;
  unsigned int pos;
  Error err;
  Tree<ConfigNode>::iterator tmpit;

  if (!where.isValid())
    return Error("Config::_parseVar()",
                   ERROR_LEVEL_INTERNAL,
                   0,
                   ERROR_ADVISE_IGNORE,
                   "where should I add vars ?");
  pos=0;
  // skip leading blanks
  while(pos<s.length()){
    if (s.at(pos)>32)
      break;
    pos++;
  }
  if (pos>=s.length())
    return Error("Config::_parseVar()",
                   ERROR_LEVEL_INTERNAL,
                   0,
                   ERROR_ADVISE_IGNORE,
                   "empty string");
  // read the name
  n.erase();
  if (_mode & CONFIG_MODE_COLON)
    err=parser::getString(s,
                            n,
                            ":#",
                            "\"\"",  // nest chars
                            pos,
                            1024);
  else
    err=parser::getString(s,
                            n,
                            "=#",
                            "\"\"",  // nest chars
                            pos,
                            1024);
  if (!err.isOk())
    return err;
  // post process the name
  err=parser::processString(n,_mode);
  if (!err.isOk())
    return err;

  if (n.empty())
    // nothing to parse, just return
    return Error();

  // skip "=" or ":"
  pos++;

  tmpit=createVar(n,where);
  if (!tmpit.isValid())
    return Error("Config::_parseVar()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "createVar()");
  return _parseValues(s,pos,tmpit);
}


Tree<ConfigNode>::iterator Config::createVar(string name,
                                                   Tree<ConfigNode>::iterator where){
  Error err;
  Tree<ConfigNode>::iterator tmpit;
  unsigned int i;
  string prepath;
  string postpath;

  if (!where.isValid())
    return where;

  // check if "where" is a group or root or a var
  if ((*where).type!=CONFIG_TYPE_GROUP &&
      (*where).type!=CONFIG_TYPE_ROOT &&
      (*where).type!=CONFIG_TYPE_VAR)
    return Tree<ConfigNode>::iterator();

  // find last "."
  i=name.rfind(".");
  if (i!=string::npos) {
    // we have a full stop, so it seems to be a full path
    if (i)
      prepath=name.substr(0,i); // exclude "."
    postpath=name.substr(i+1);
  }
  else
    // we have no full stop, so it is just a group name
    postpath=name;
  // first select/create the path
  if (!prepath.empty()) {
    tmpit=findPath(prepath,where,true);
    if (!tmpit.isValid())
      return Tree<ConfigNode>::iterator();
    where=tmpit;
  }

  // then create the group in it. should groups be unique ?
  if (_mode & CONFIG_MODE_UNIQUE_VARS) {
    // yes, so look first if there is already a group with that name
    tmpit=_findVariable(postpath,where);
    if (tmpit.isValid()) {
      // yes, we have one, should we clear it ?
      if (_mode & CONFIG_MODE_OVERWRITE_VARS)
        // yes, so be it
        if (!tmpit.clearBranch())
          return Tree<ConfigNode>::iterator();
      return tmpit;
    } // if group found
  }
  // add var
  return _addVariable(postpath,where);
}


Tree<ConfigNode>::iterator Config::createGroup(string name,
                                                     Tree<ConfigNode>::iterator where){
  Error err;
  Tree<ConfigNode>::iterator tmpit;
  unsigned int i;
  string prepath;
  string postpath;

  if (!where.isValid())
    return Tree<ConfigNode>::iterator();

  // check if "where" is a group or root or a var
  if ((*where).type!=CONFIG_TYPE_GROUP &&
      (*where).type!=CONFIG_TYPE_ROOT)
    return Tree<ConfigNode>::iterator();

  // erase eventually existing leading slash
  if (name.at(0)=='/')
    name.erase(0,1);

  // remove trailing slash
  if (name.at(name.length()-1)=='/')
    name.erase(name.length()-1,1);

  // find last "/"
  i=name.rfind("/");

  if (i!=string::npos) {
    // we have a full stop, so it seems to be a full path
    if (i)
      prepath=name.substr(0,i+1); // include "/"
    postpath=name.substr(i+1);
  }
  else
    // we have no full stop, so it is just a group name
    postpath=name;

  // first select/create the path
  if (!prepath.empty()) {
    tmpit=findPath(prepath,where,true);
    if (!tmpit.isValid())
      return Tree<ConfigNode>::iterator();
    where=tmpit;
  }

  // then create the group in it. should groups be unique ?
  if (_mode & CONFIG_MODE_UNIQUE_GROUPS) {
    // yes, so look first if there is already a group with that name
    tmpit=_findGroup(postpath,where);
    if (tmpit.isValid()) {
      // yes, we have one, should we clear it ?
      if (_mode & CONFIG_MODE_OVERWRITE_GROUPS)
        // yes, so be it
        if (!tmpit.clearBranch())
          return Tree<ConfigNode>::iterator();
      return tmpit;
    } // if group found
  }
  // add var
  return _addGroup(postpath,where);
}

Error Config::_parseLine(string &s){
  string tmp;
  unsigned int pos;
  Error err;
  Tree<ConfigNode>::iterator tmpit;

  pos=0;
  tmp.erase();
  if (_mode & CONFIG_MODE_COLON)
      err=parser::getString(s,
                              tmp,
                              ":[#",
                              "\"\"",  // nest chars
                              pos,
                              8192);
  else
      err=parser::getString(s,
                              tmp,
                              "=[#",
                              "\"\"",  // nest chars
                              pos,
                              8192);
  if (!err.isOk())
    return err;

  if (pos<s.length()) {
    // we got one of the stop characters, is it "[" ?
    if (s.at(pos)=='[') {
      // yes, so parse the group name
      tmpit=_parseroot;
      // groups are created under the given root, which has not to be the real root
      err=_parseGroup(s,tmpit);
      // error, so do not change the group
      if (!err.isOk())
	return err;
      // otherwise tell the new group
      _parsewhere=tmpit;
      return Error();
    }
    // if "=" or ":" then handle it as a variable
    else if ((_mode & CONFIG_MODE_COLON &&
	      (s.at(pos)==':')) ||
	     (!(_mode & CONFIG_MODE_COLON) &&
	      s.at(pos)=='=')){
      tmpit=_parsewhere;
      return _parseVar(s,tmpit);
    }
  }
  // otherwise handle it as a value
  if (!tmp.empty()) {
    err=parser::processString(tmp,_mode);
    if (!err.isOk())
      return err;
    if (!tmp.empty())
      _parsewhere.addChild(ConfigValue(tmp));
  }
  return Error();
}


Error Config::beginParsing(Tree<ConfigNode>::iterator where){
  _parsewhere=where;
  _parseroot=where;
  _linebuffer.erase();
  return Error();
}


Error Config::parseLine(string &s){
    Error err;
    unsigned int pos;

  // case 1: new line is empty
  if (s.empty()) {
    // linebuffer is not empty, so go and parse it.
    if (!_linebuffer.empty()) {
      err=_parseLine(_linebuffer);
      if (!err.isOk())
        return err;
      _linebuffer.erase();
      return Error();
    }
    else
      // otherwise there is nothing to do
      return Error();
  }

  // case 2: new line is NOT empty, but linebuffer is
  else if (_linebuffer.empty()) {
    _linebuffer=s;
    return Error();
  }

  // case 3: new line is not empty and linebuffer contains data
  else {
    // check if last char of previous line was "\" and if that leads to continuation
    if (_linebuffer.at(_linebuffer.length()-1)=='\\' &&
        (_mode & CONFIG_MODE_BACKSLASH_CONTINUATION)) {
      // yepp, so remove the "\" and add the new line
      _linebuffer=_linebuffer.erase(_linebuffer.length()-1);
      _linebuffer+=" " + s;
      return Error();
    }
    // check if new line starts with blank and if that leads to continuation
    else if (s.at(0)<33 &&
             (_mode & CONFIG_MODE_BLANK_CONTINUATION)) {
        // yepp, so simply add this line to the previous one
        // first remove leading blanks
        pos=0;
        while(pos<s.length()) {
            if (s.at(pos)>32)
                break;
            pos++;
        }
        // is there still data in the string ?
        if (pos<s.length()) {
            // yes, it is, so insert one blank
            _linebuffer+=" ";
            _linebuffer+=s.substr(pos);
        }
        return Error();
    }
    // ok, no continuation, so first parse last line
    err=_parseLine(_linebuffer);
    if (!err.isOk())
      return err;
    // store next line
    _linebuffer=s;
    return Error();
  }
}


Error Config::endParsing(){
  if (!_linebuffer.empty())
    return _parseLine(_linebuffer);
  return Error();
}


Tree<ConfigNode>::iterator 
Config::findPath(string path,
		   Tree<ConfigNode>::iterator where,
		   bool crea){
  string tmp;
  unsigned int pos;
  Error err;
  Tree<ConfigNode>::iterator tmpit;
  bool havevar;

  pos=0;
  havevar=false;
  while(pos<path.length()) {
    tmp.erase();
    err=parser::getString(path,
                            tmp,
                            "/.",
                            "\"\"",  // nest chars
                            pos,
                            8192);
    if (!err.isOk())
      return Tree<ConfigNode>::iterator();

    // post process the name
    err=parser::processString(tmp,_mode);
    if (!err.isOk())
      return Tree<ConfigNode>::iterator();

    // check if the word read is a group or a variable name
    if (pos<path.length()) {
      if (path.at(pos)=='/') {
        if (havevar)
          // we cannot have a group below a var !
          return Tree<ConfigNode>::iterator();
      }
      else
        havevar=true;
    }
    else
      havevar=true;

    // handle the word read according to its type
    if (!havevar) {
      // we have a group name, handle it
      if (!tmp.empty()) {
        tmpit=_findGroup(tmp,where);
        if (!tmpit.isValid() && crea)
          tmpit=_addGroup(tmp,where);
        if (!tmpit.isValid())
          return Tree<ConfigNode>::iterator();
        where=tmpit;
      }
    }
    else {
      // else we have a variable name
      if (!tmp.empty()) {
        tmpit=_findVariable(tmp,where);
        if (!tmpit.isValid() && crea)
          tmpit=_addVariable(tmp,where);
        if (!tmpit.isValid())
          return Tree<ConfigNode>::iterator();
        where=tmpit;
      } // if word read is not empty
    } // if there is something behind the word read

    // skip "/" or "."
    pos++;
  } // while

  // all fine.
  return where;
}



Tree<ConfigNode>::const_iterator 
Config::findPath(string path,
		   Tree<ConfigNode>::const_iterator where) const {
  string tmp;
  unsigned int pos;
  Error err;
  Tree<ConfigNode>::const_iterator tmpit;
  bool havevar;

  pos=0;
  havevar=false;
  while(pos<path.length()) {
    tmp.erase();
    err=parser::getString(path,
                            tmp,
                            "/.",
                            "\"\"",  // nest chars
                            pos,
                            8192);
    if (!err.isOk())
      return Tree<ConfigNode>::iterator();

    // post process the name
    err=parser::processString(tmp,_mode);
    if (!err.isOk())
      return Tree<ConfigNode>::iterator();

    // check if the word read is a group or a variable name
    if (pos<path.length()) {
      if (path.at(pos)=='/') {
        if (havevar)
          // we cannot have a group below a var !
          return Tree<ConfigNode>::iterator();
      }
      else
        havevar=true;
    }
    else
      havevar=true;

    // handle the word read according to its type
    if (!havevar) {
      // we have a group name, handle it
      if (!tmp.empty()) {
        tmpit=_findGroup(tmp,where);
        if (!tmpit.isValid())
          return Tree<ConfigNode>::iterator();
        where=tmpit;
      }
    }
    else {
      // else we have a variable name
      if (!tmp.empty()) {
        tmpit=_findVariable(tmp,where);
        if (!tmpit.isValid())
          return Tree<ConfigNode>::iterator();
        where=tmpit;
      } // if word read is not empty
    } // if there is something behind the word read

    // skip "/" or "."
    pos++;
  } // while

  // all fine.
  return where;
}


Error Config::readFromStream(Stream *st,
                                 Tree<ConfigNode>::iterator where){
  string line;
  Error err;
  unsigned int pos;
  bool goon;

  if (!st)
    return Error("Config::readFromStream()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "NULL pointer for st");

  // begin parsing
  err=beginParsing(where);
  if (!err.isOk())
    return err;

  goon=true;
  while(goon && !st->eof()) {
    // read line
    pos=0;
    line.erase();
    st->readLine(line,8192);
    // skip leading blanks
    while(pos<line.length()){
      if (line.at(pos)>32)
        break;
      pos++;
    }
    if (pos>=line.length() &&
        (_mode & CONFIG_MODE_STOP_ON_BLANK_LINE))
      // line is blank and that is a stop criteria
      goon=false;
    else {
      // otherwise handle line, even if blank
      err=parseLine(line);
      if (!err.isOk() &&
          (_mode & CONFIG_MODE_RETURN_ON_ERROR))
        // return error if desired, ignore it otherwise
        return err;
    }
  } // while
  return endParsing();
}


Error Config::_writeGroup(Stream *st,
                              Tree<ConfigNode>::const_iterator where) const{

  Tree<ConfigNode>::const_iterator tmpit;
  bool namewritten;
  string name;
  Error err;

  // if group is empty and we should not write empty groups just return
  if (!where.hasChild() && !(_mode & CONFIG_MODE_WRITE_EMPTY_GROUP))
    return Error();

  // get group path
  if (_mode & CONFIG_MODE_WRITE_GROUP_NAME) {
    if (where==_parseroot)
      // do not write the name if we are the root of the branch to be written
      namewritten=true;
    else {
      tmpit=where;
      name=(*tmpit).data;
      tmpit.parent();
      while(tmpit.isValid()) {
        if (tmpit==_parseroot)
          break;
        name=(*tmpit).data+"/"+name;
        tmpit.parent();
      } // while
      namewritten=false;
      name="[" + name + "]";
    }
  }
  else
    namewritten=true;

  where.child();

  // write all vars and values
  tmpit=where;
  while(tmpit.isValid()) {
    if ((*tmpit).type==CONFIG_TYPE_VAR) {
      if (!namewritten) {
	st->writeLine(""); // for better readability
	st->writeLine(name);
	namewritten=true;
      }
      err=_writeVar(st,tmpit);
      if (!err.isOk())
	return err;
    } // if var
    else if ((*tmpit).type==CONFIG_TYPE_VALUE) {
      if (!namewritten) {
        st->writeLine(name);
        namewritten=true;
      }
      if (_mode & CONFIG_MODE_QUOTED_VALUES)
        st->writeLine("\""+(*tmpit).data+"\"");
      else
        st->writeLine((*tmpit).data);
    } // if value
    tmpit++;
  } // while

  // write all sub groups
  tmpit=where;
  while(tmpit.isValid()) {
    if ((*tmpit).type==CONFIG_TYPE_GROUP) {
      err=_writeGroup(st,tmpit);
      if (!err.isOk())
        return err;
    }
    tmpit++;
  } // while
  return Error();
}


Error Config::_writeVar(Stream *st,
                              Tree<ConfigNode>::const_iterator where) const {

  Tree<ConfigNode>::const_iterator tmpit;
  bool namewritten;
  string name;
  string value;
  bool first;
  Error err;

  // if var is empty just return
  if (!where.hasChild())
    return Error();

  // get var path
  if (where==_parseroot)
    // do not write the name if we are the root of the branch to be written
    namewritten=true;
  else {
    tmpit=where;
    name=(*tmpit).data;
    tmpit.parent();
    while(tmpit.isValid()) {
      if (tmpit==_parseroot)
        break;
      if ((*tmpit).type==CONFIG_TYPE_GROUP)
        break;
      else if ((*tmpit).type==CONFIG_TYPE_VAR)
        name=(*tmpit).data+"."+name;
      else
        return Error("Config::_writeVar()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "illegal order in Tree");
      tmpit.parent();
    } // while
    // add pes anserinus
    if (_mode & CONFIG_MODE_QUOTED_NAME)
      name="\""+name+"\"";
    if (_mode & CONFIG_MODE_COLON)
      name+=": ";
    else
      name+="=";
  }

  where.child();

  // write all values
  tmpit=where;
  first=true;
  while(tmpit.isValid()) {
    if ((*tmpit).type==CONFIG_TYPE_VALUE) {
      // create the value string
      value=(*tmpit).data;
      if (_mode & CONFIG_MODE_QUOTED_VALUES)
        value="\""+value+"\"";
      // write the line
      if ((_mode & CONFIG_MODE_ONE_VALUE_PER_LINE)) {
        st->writeString(name);
        st->writeLine(value);
      }
      else {
        if (first) {
          st->writeString(name);
          first=false;
        }
        else
          st->writeString(",");
        st->writeString(value);
      }
    } // if value
    tmpit++;
  } // while
  if (!first)
      st->writeLine("");
  // write all sub-variables
  tmpit=where;
  while(tmpit.isValid()) {
    if ((*tmpit).type==CONFIG_TYPE_VAR) {
      err=_writeVar(st,tmpit);
      if (!err.isOk())
        return err;
    } // if value
    tmpit++;
  } // while

  return Error();
}


Error Config::writeToStream(Stream *st,
                                Tree<ConfigNode>::iterator where){

  if (!st)
    return Error("Config::readFromStream()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "NULL pointer for st");

  if (!where.isValid())
    return Error("Config::writeToStream()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "'where' is invalid");
  // FIXME: Is this assignment really necessary? If it's not, we can
  // switch this method to const and using a const_iterator where.
  _parseroot=where;

  if ((*where).type==CONFIG_TYPE_GROUP)
    return _writeGroup(st,where);
  if ((*where).type==CONFIG_TYPE_ROOT)
    return _writeGroup(st,where);
  else if ((*where).type==CONFIG_TYPE_VAR)
    return _writeVar(st,where);
  else
    return Error("Config::writeToStream()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "can only write groups or variables");
}


Tree<ConfigNode>::iterator Config::findGroup(string path,
                                                   Tree<ConfigNode>::iterator where,
                                                   bool create){
  if (!where.isValid() ||
      path.empty())
    return where;
  if (path=="/" && ((*where).type==CONFIG_TYPE_GROUP ||
                    (*where).type==CONFIG_TYPE_ROOT))
    return where;

  if (path.at(path.length()-1)!='/')
    path+="/";
  return findPath(path,where,create);
}


Tree<ConfigNode>::const_iterator 
Config::findGroup(string path,
		    Tree<ConfigNode>::const_iterator where) const {
  if (!where.isValid() ||
      path.empty())
    return where;
  if (path=="/" && ((*where).type==CONFIG_TYPE_GROUP ||
                    (*where).type==CONFIG_TYPE_ROOT))
    return where;

  if (path.at(path.length()-1)!='/')
    path+="/";
  return findPath(path,where);
}


Tree<ConfigNode>::iterator 
Config::findVariable(string path,
		       Tree<ConfigNode>::iterator where,
		       bool create)
{
  if (!where.isValid() ||
      path.empty())
    return where;
  if (path=="/" && (*where).type==CONFIG_TYPE_VAR)
    return where;

  return findPath(path,where, create);
}

Tree<ConfigNode>::const_iterator 
Config::findVariable(string path,
		       Tree<ConfigNode>::const_iterator where) const
{
  if (!where.isValid() ||
      path.empty())
    return where;
  if (path=="/" && (*where).type==CONFIG_TYPE_VAR)
    return where;

  return findPath(path,where);
}


void Config::clear(){
    _cfg.clear();
}



void Config::dumpCfg(Tree<ConfigNode>::const_iterator where, int level) {
  int i;
  string t;

  if (!where.isValid())
      return;
  while(where.isValid()) {
      for (i=0; i<=level; i++)
          fprintf(stderr,"  ");
      switch((*where).type) {
      case CONFIG_TYPE_VALUE:
          t="Value";
          break;

      case CONFIG_TYPE_VAR:
          t="Variable";
          break;
      case CONFIG_TYPE_GROUP:
          t="Group";
          break;
      case CONFIG_TYPE_ROOT:
          t="Root";
          break;
      default:
          t="<unknown>";
          break;
      } // switch
      if ((*where).type==CONFIG_TYPE_VALUE)
          fprintf(stderr,"- ");
      fprintf(stderr,"\"%s\" (%s)\n",(*where).data.c_str(),t.c_str());
      if (where.hasChild()) {
          where.child();
          dumpCfg(where,level+1);
          where.parent();
      }
      where++;
  }
}

} // namespace HBCI
