/***************************************************************************
 $RCSfile: interactor.h,v $
                             -------------------
    cvs         : $Id: interactor.h,v 1.8 2002/12/27 23:06:53 aquamaniac Exp $
    begin       : Sat Jun 08 2002
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HBCIINTERACTOR_H
#define HBCIINTERACTOR_H

/** @file interactor.h
 *
 * @short HBCI::Interactor and C wrapper @ref HBCI_Interactor. */

#ifdef __cplusplus
namespace HBCI {
  class Interactor;
};
typedef struct HBCI::Interactor HBCI_Interactor;
#else /* __cplusplus */
typedef struct HBCI_Interactor HBCI_Interactor;
#endif /* __cplusplus */

#include <openhbci/dllimport.h>
#include <openhbci/pointer.h>
#include <openhbci/user.h>
#include <openhbci/medium.h>


#ifdef __cplusplus

namespace HBCI {
/**  
 * @short Interface class to provide methods for every user interaction that 
 * is needed by the core OpenHBCI.
 * 
 * The default implementation for every method does nothing, i.e. if
 * you don't provide your own derived class OpenHBCI is unable to do
 * any user interaction at all. You have to derive this class and
 * overload all methods so that you offer customized user interaction
 * in your application. 
 *
 * Again, there is no other user interaction in the core OpenHBCI
 * except through the methods in this class. Note, however, that the
 * non-core class @ref API in fact does provide further user
 * interaction for progress bar support through means of the @ref
 * ProgressMonitor class.
 *
 * @author Martin Preuss<martin@libchipcard.de> */
class DLLIMPORT Interactor {
private:
    bool _aborted;

public:
    /**
     * Ask the user for the pin of a security medium. This method is
     * called by the default Authentifikator. The default
     * implementation does nothing and returns false.
     *
     * @author Martin Preuss<martin@libchipcard.de>
     * @return true if ok, false if the user aborted
     * @param user user for whom the pin is to be asked for.
     * Please note that this pointer may be invalid.
     * @param pin reference to a variable to receive the pin
     * @param minsize minimum size of the pin.
     * @param newPin if true then the user should be asked for a new pin.
     * Most implementations will ask the user for the pin and then ask him
     * to confirm it (thus making the user doubly enter the pin).  */
    virtual bool msgInputPin(Pointer<User> user,
                             string &pin,
                             int minsize,
			     bool newPin);

    /**
     * This method is called directly before the user is supposed to enter the
     * pin into the keypad of a card reader. This is used to inform the user
     * that OpenHBCI started secure pin verification. A graphical application
     * might want to open a window that tells the user about this.
     * (new since 0.9.5)
     * @author Martin Preuss<martin@libchipcard.de>
     * @param user user for whom the pin is to be asked for.
     * Please note that this pointer may be invalid.
     */
    virtual void msgStartInputPinViaKeypad(Pointer<User> user);

    /**
     * This method is called directly after the secure pin verification (via
     * the card readers keypad) is finished. A graphical applicatition most
     * likely closes the notification window opened at
     * @ref msgStartInputPinViaKeypad().
     * (new since 0.9.5)
     * @author Martin Preuss<martin@libchipcard.de>
     * @param user user for whom the pin is to be asked for.
     * Please note that this pointer may be invalid.
     */
    virtual void msgFinishedInputPinViaKeypad(Pointer<User> user);

    /**
     * Tell the user to insert a chip card. The default
     * implementation does nothing and returns false.
     *
     * @param user user for whom the card is to be inserted.
     * Please note that this pointer may be invalid.
     * @param t The MediumType of the medium that should be inserted.
     * @author Martin Preuss<martin@libchipcard.de>
     * @return true if ok, false if user wants to abort.
     */
    virtual bool msgInsertMediumOrAbort(Pointer<User> user,
                                        MediumType t);

    /**
     * Tell the user that:<ul>
     * <li>the currently inserted chip card is not the right one and</li>
     * <li>he/she should now insert the correct card</li>
     * </ul> The default
     * implementation does nothing and returns false.
     *
     * @author Martin Preuss<martin@libchipcard.de>
     * @param t The MediumType of the medium that should be inserted.
     * @return true if ok, false if user wants to abort.
     * @param user user for whom the card is to be inserted.
     * Please note that this pointer may be invalid.
     */
    virtual bool msgInsertCorrectMediumOrAbort(Pointer<User> user,
                                               MediumType t);

    /**  
     * Prints a message that might be used for logging purpose. Does
     * not require interaction. The default implementation does
     * nothing and discards the message.  This method should be
     * overwritten by your application.  
     * @param msg the message to show */
    virtual void msgStateResponse(const string &msg);

    /**
     * During actions which take a long time to perform (some seconds or
     * minutes) this method is called by OpenHBCI every once in a while. 
     * 
     * You can override this method to let your program do some GUI
     * update stuff. Also, if this method returns "false", then the
     * long term action will be aborted. Otherwise the action simply
     * continues.  
     *
     * The default implementation of the Interactor returns the
     * variable set by @ref abort() which is accessible also by @ref
     * aborted(). Thus the Interactor offery you the convenience
     * of only using @ref abort() instead of deriving your own
     * Interactor. If you overload this method, you probably
     * should take the value of @ref aborted() into account to comply
     * to that default behaviour.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @return false if user wants the action to be aborted, false otherwise */
    virtual bool keepAlive();

    /** 
     * Sets an abort flag which is queried by the default
     * implementation of the @ref keepAlive() method. This might be
     * called to signal that the user wants to abort the ongoing
     * action. A call to this method might result from the user
     * pressing an "abort" button.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param a if true then the next call to keepAlive() will return false */
    void abort(bool a=true) {_aborted=a;};

    /** 
     * Returns the abort flag set by @ref abort(). This is used by
     * the default implementation of the @ref keepAlive() method.
     * This flag probably tells you if the user wants to abort the ongoing
     * transaction.  
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @return returns the flag set through @ref abort().  */
    bool aborted() const { return _aborted;};

    Interactor();
    virtual ~Interactor();

};

} /* namespace HBCI */
extern "C" {
#endif /* __cplusplus */
    extern void HBCI_Interactor_msgStateResponse(HBCI_Interactor *i,
						 const char *msg);
    extern void HBCI_Interactor_abort(HBCI_Interactor *i, int a);
    extern int HBCI_Interactor_aborted(const HBCI_Interactor *i);
#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif



