/* oslConfig.cc
 */
#include "osl/oslConfig.h"
#include "osl/config.h"
#include "osl/misc/ncores.h"
#include <map>
#include <boost/filesystem/operations.hpp>
#include <boost/static_assert.hpp>
#include <limits>
#include <iostream>
#include <fstream>
#include <cstdlib>
#ifndef _MSC_VER
#  include <unistd.h>
#endif
#ifdef _WIN32
#  include <windows.h>
#  include <psapi.h>
#else
#include <sys/resource.h>
#ifdef __FreeBSD__
#  include <kvm.h>
#  include <sys/param.h>
#  include <sys/sysctl.h>
#  include <sys/user.h>
#  include <paths.h>
#  include <fcntl.h>
#endif
#ifdef __APPLE__
#  include <sys/types.h>
#  include <sys/sysctl.h>
#  include <mach/task.h>
#  include <mach/mach_init.h>
#endif
#endif

const int osl::OslConfig::MaxThreads; // define
unsigned int osl::OslConfig::eval_random = 0;

bool osl::OslConfig::is_verbose = false;
#ifndef OSL_NCPUS
const int osl::OslConfig::default_ncpus = osl::misc::ncores();
#else
BOOST_STATIC_ASSERT(OSL_NCPUS <= osl::OslConfig::MaxThreads);
const int osl::OslConfig::default_ncpus = OSL_NCPUS;
#endif
int osl::OslConfig::num_cpu = default_ncpus;
volatile bool osl::OslConfig::usi_mode = false, 
  osl::OslConfig::usi_mode_silent = false, 
  osl::OslConfig::use_log_linear_probability = false,
  osl::OslConfig::force_root_null_window = false;
int osl::OslConfig::usi_output_pawn_value = 100; 
volatile int osl::OslConfig::root_null_window = 0; 
boost::mutex osl::OslConfig::lock_io;

namespace
{
  size_t system_memory_use_limit() 
  {
#ifdef _WIN32
    MEMORYSTATUSEX statex;
    statex.dwLength = sizeof(statex);
    GlobalMemoryStatusEx(&statex);
    return statex.ullTotalPhys; // in bytes
#else
    size_t limit_by_rlimit = std::numeric_limits<size_t>::max();
    {
      rlimit rlp;
      if (getrlimit(RLIMIT_AS, &rlp) == 0
	  && rlp.rlim_cur != std::numeric_limits<rlim_t>::max()) {
	limit_by_rlimit = rlp.rlim_cur;
#ifdef __APPLE__
	limit_by_rlimit *= 1024;
#endif
	std::cerr << "rlimit " << limit_by_rlimit << "\n";
      }
    }
#ifdef __APPLE__
    {
      int mib[2];
      unsigned int usermem;
      size_t len=sizeof(usermem);
      mib[0] = CTL_HW;
      mib[1] = HW_USERMEM;
      if (sysctl(mib, 2, &usermem, &len, NULL, 0) == 0
	  && len == sizeof(usermem)) {
	std::cerr << "usermem " << usermem << std::endl;
	return std::min((size_t)usermem, limit_by_rlimit);
      }
    }
#endif
    {
      std::string name, unit;
      size_t value;
      std::ifstream is("/proc/meminfo");
      if (is >> name >> value >> unit
	  && name == "MemTotal:" && unit == "kB")
	return std::min(value * 1024, limit_by_rlimit);
    }
#if (defined __FreeBSD__)
    const long mem = sysconf(_SC_PHYS_PAGES);
    if (mem != -1) 
      return std::min(mem * getpagesize(), limit_by_rlimit);
#endif
    return std::min((rlim_t)limit_by_rlimit, std::numeric_limits<rlim_t>::max());
#endif
  }
}

size_t osl::OslConfig::memory_use_limit = system_memory_use_limit();
const size_t osl::OslConfig::memory_use_limit_system_max = 
#ifdef _WIN32
  3000000000; // 3GB
#else
  std::numeric_limits<rlim_t>::max();
#endif

void osl::OslConfig::setNumCPUs(int ncpu)
{
  if (ncpu > MaxThreads) {
    std::cerr << "ncpu " << ncpu << " > " << "MaxThreads " << MaxThreads << "\n";
    ncpu = MaxThreads;
  }
  num_cpu = ncpu;
}

int osl::OslConfig::numCPUs()
{
  return num_cpu;
}

void osl::OslConfig::setVerbose(bool v)
{
  is_verbose = v;
}

bool osl::OslConfig::verbose()
{
  return is_verbose;
}

bool osl::OslConfig::usiMode()
{
  return usi_mode;
}
void osl::OslConfig::setUsiMode(bool enable)
{
  usi_mode = enable;
}
bool osl::OslConfig::usiModeInSilent()
{
  return usi_mode_silent;
}
void osl::OslConfig::setUsiSilent(bool enable)
{
  usi_mode_silent = enable;
}

bool osl::OslConfig::useLogLinearProbability()
{
  static bool fixed_use_log_linear_probability
    = use_log_linear_probability || getenv("OSL_USE_LOG_LINEAR_PROBABILITY");
  return fixed_use_log_linear_probability;
}
void osl::OslConfig::enableLogLinearProbability(bool enabled)
{
  use_log_linear_probability = enabled;
}

void osl::OslConfig::showOslHome(const std::string& home)
{
  std::cerr << "using " << home << " as OSL_HOME, word size "
	    << OSL_WORDSIZE << std::endl;
}

void osl::OslConfig::showOslHome()
{
  showOslHome(home());
}

bool osl::OslConfig::isGoodDir(const std::string& dir)
{
  return boost::filesystem::exists(dir)
    && boost::filesystem::is_directory(dir);
}

void osl::OslConfig::trySetDir(std::string& dir, const std::string& candidate)
{
  if (isGoodDir(candidate))
  {
    dir = candidate;
    return;
  }
  if (verbose())
    std::cerr << "skipping " << candidate << std::endl;
}

const std::string osl::OslConfig::makeHome()
{
  std::string result;
  if (const char *env = getenv("GPSSHOGI_HOME"))
    trySetDir(result, env);
  
#if defined GPSSHOGI_HOME
  if (result.empty())
    trySetDir(result, GPSSHOGI_HOME);
#endif

  if (result.empty())
    if (const char *env = getenv("OSL_HOME"))
      trySetDir(result, env);

  if (result.empty())
    result = OSL_HOME;

  if (verbose())
    showOslHome(result);
  return result;
}

const std::string& osl::OslConfig::home()
{
  static const std::string home_directory = makeHome();
  return home_directory;
}

const char * osl::OslConfig::home_c_str()
{
  return home().c_str();
}

const std::string osl::OslConfig::gpsusiConf()
{
  // issue:
  // - 開発者には $HOME ではなく OSL_HOME の方が使い分ける都合が良い
  // - 一方、配布版では OSL_HOME は共有ディレクトリで書き込めないかもしれない
#ifdef OSL_PUBLIC_RELEASE
  // for personal users
  if (const char *env = getenv("HOME"))
    return std::string(env) + "/gpsusi.conf";
  if (const char *env = getenv("USERPROFILE"))
    return std::string(env) + "/gpsusi.conf";
#endif
  // for developpers
  static const std::string home_directory = makeHome();
  return home_directory + "/gpsusi.conf";
}

const std::string osl::OslConfig::makeTest()
{
  std::string result;
  if (const char *env = getenv("OSL_TEST"))
    trySetDir(result, env);

  if (result.empty())
    result = home() + "/data";	// 今はdata内に混在

  std::cerr << "using " << result << " as OSL_TEST" << std::endl;
  return result;
}

const std::string osl::OslConfig::test()
{
  static const std::string test_directory = makeTest();
  return test_directory;
}

namespace 
{
  struct NameHolder : std::map<std::string,std::string>
  {
    std::string directory;

    NameHolder(const std::string& d) : directory(d)
    {
      directory += "/";
    }

    iterator add(const std::string& key, const std::string& value)
    {
      return insert(std::make_pair(key, value)).first;
    }
    iterator addRelative(const std::string& key, const std::string& filename)
    {
      std::string value = directory + filename;
      return add(key, value);
    }
    iterator addRelative(const std::string& filename)
    {
      return addRelative(filename, filename);
    }
  };
}

const char * osl::OslConfig::testFile(const std::string& filename)
{
  static NameHolder table(test());
  NameHolder::iterator p=table.find(filename);
  if (p == table.end()) {
    p = table.addRelative(filename);
  }
  return p->second.c_str();
}

const char * osl::OslConfig::testCsaFile(const std::string& filename)
{
  static NameHolder table(test()+"/kifdat");
  NameHolder::iterator p=table.find(filename);
  if (p == table.end()) {
    p = table.addRelative(filename);
  }
  return p->second.c_str();
}

const char *osl::OslConfig::openingBook(const std::string& filename)
{
  static NameHolder table(home()+"/data");
  NameHolder::iterator p=table.find(filename);
  if (p == table.end()) {
    if (! filename.empty() && filename[0] == '/') {
      // absolute path
      p = table.add(filename, filename);
    }
    else {
      // relative path
      p = table.addRelative(filename, 
			    (filename == "" ? "joseki.dat" : filename));
    }
  }
  return p->second.c_str();
}


size_t osl::OslConfig::residentMemoryUse()
{
#if defined(_WIN32)
  static const DWORD process_id = GetCurrentProcessId();
  HANDLE hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ,
                                FALSE, process_id);
  if (NULL == hProcess)
  {
    std::cerr << "Failed to get residentMemoryUse()\n";
    return 0;
  }

  size_t working_set = 0;
  PROCESS_MEMORY_COUNTERS pmc;
  if (GetProcessMemoryInfo(hProcess, &pmc, sizeof(pmc))) {
    working_set = pmc.WorkingSetSize; // in bytes
  }
  CloseHandle(hProcess);
  return working_set;
#else
  // see proc(5)
  // note: < 40000 cycles @macpro2
  std::ifstream is("/proc/self/statm");
  size_t total, resident;
  if (is >> total >> resident)
    return resident*getpagesize();
#ifdef __APPLE__
  mach_msg_type_number_t count = TASK_BASIC_INFO_64_COUNT;
  task_basic_info_64 ti;
  if (task_info(current_task(), TASK_BASIC_INFO_64, (task_info_t)&ti, &count)
      == KERN_SUCCESS)
    return ti.resident_size;
#endif
#ifdef __FreeBSD__
  static kvm_t *kd = kvm_open(NULL, _PATH_DEVNULL, NULL, O_RDONLY, "osl kvm_open");
  int nproc;
  kinfo_proc *pp = kvm_getprocs(kd, KERN_PROC_PID, getpid(), &nproc);
  if (pp)
    return pp->ki_rssize * getpagesize();
#endif  
#endif
  return 0;
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
