/* pieceEval.t.cc
 */
#include "osl/eval/pieceEval.h"
#include "osl/eval/pieceEval.tcc"
#include "osl/state/simpleState.tcc"
#include "osl/state/numEffectState.h"
#include "osl/effect_util/effectUtil.h"
#include "consistencyTest.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>
#include <fstream>

class PieceEvalTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PieceEvalTest);
  CPPUNIT_TEST(testBug);
  CPPUNIT_TEST(testComputeDiffAfterMoveForRPBug030910);
  CPPUNIT_TEST(testComputeDiffAfterMoveForRP);
  CPPUNIT_TEST(testCannotTake);
  CPPUNIT_TEST(testConsistentUpdate);
  CPPUNIT_TEST(testConsistentExpect);
  CPPUNIT_TEST(testSoma);
  CPPUNIT_TEST_SUITE_END();
public:
  void testBug();
  void testSoma();
  void testCannotTake();
  void testComputeDiffAfterMoveForRP();
  void testComputeDiffAfterMoveForRPBug030910();
  void testComputeDiffAfterTakeBack();
  void testConsistentUpdate();
  void testConsistentExpect();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PieceEvalTest);

using namespace osl;
using namespace osl::eval;

const int Gold = PtypeEvalTraits<GOLD>::val;
const int Rook = PtypeEvalTraits<ROOK>::val;
const int Knight = PtypeEvalTraits<KNIGHT>::val;
const int Pawn = PtypeEvalTraits<PAWN>::val;
const int Pknight = PtypeEvalTraits<PKNIGHT>::val;
const int Bishop = PtypeEvalTraits<BISHOP>::val;
const int Pbishop = PtypeEvalTraits<PBISHOP>::val;

void PieceEvalTest::testSoma()
{
  const NumEffectState state(CsaString(
			    "P1-KY-KE-GI-KI *  * -HI-KE * \n"
			    "P2 * -HI *  * -KA-OU *  *  * \n"
			    "P3-FU * -FU-FU-FU * -GI-FU-KY\n"
			    "P4 * +KE *  *  * -FU *  * -FU\n"
			    "P5 * -FU+KE+KA *  *  *  * +FU\n"
			    "P6 *  * +FU *  *  * +FU *  * \n"
			    "P7+FU+FU * +FU+FU+FU *  *  * \n"
			    "P8 * +GI+KI *  *  * +GI *  * \n"
			    "P9+KY *  *  * +OU+KI *  * +KY\n"
			    "P+00FU\n"
			    "P+00FU\n"
			    "P+00KI\n"
			    "+\n").getInitialState());
  const PieceEval ev(state);
  {
    int ret=ev.computeDiffAfterMove
      (state, Move(Square(1,5),Square(1,4),PAWN,PAWN,false,BLACK));
    /** 相手が取り替えさないのが最善 */
    CPPUNIT_ASSERT_EQUAL(+Pawn*2, ret);
  }
  {
    int ret=ev.computeDiffAfterMove
      (state, Move(Square(6,5),Square(9,2),PBISHOP,PTYPE_EMPTY,true,BLACK));
    /** 香車取り返し only */
    CPPUNIT_ASSERT_EQUAL(-Bishop*2, ret);
  }
  {
    int ret=ev.computeDiffAfterMove
      (state, Move(Square(6,5),Square(8,3),PBISHOP,PTYPE_EMPTY,true,BLACK));
    /** 成り得 */
    CPPUNIT_ASSERT_EQUAL(+Pbishop-Bishop, ret);
  }
  {
    int ret=ev.computeDiffAfterMove
      (state, Move(Square(6,5),Square(2,1),PBISHOP,KNIGHT,true,BLACK));
    CPPUNIT_ASSERT_EQUAL(+Knight*2-Bishop*2, ret);
  }
  {
    int ret=ev.computeDiffAfterMove
      (state, Move(Square(8,4),Square(7,2),PKNIGHT,PTYPE_EMPTY,true,BLACK));
    CPPUNIT_ASSERT_EQUAL(-Knight*2, ret);
  }
  {
    int ret1=ev.computeDiffAfterMove
      (state, Move(Square(6,5),Square(5,4),BISHOP,PTYPE_EMPTY,false,BLACK));
    CPPUNIT_ASSERT_EQUAL(-Bishop*2, ret1);
  }
}

void PieceEvalTest::testBug()
{
  const NumEffectState state=CsaString(
    "P1-KY-KE * -KI *  *  * -KE-KY\n"
    "P2 * -OU-GI * -KI * -HI *  * \n"
    "P3 * -FU *  * -FU-GI-KA * -FU\n"
    "P4 *  * -FU-FU * -FU-FU-FU * \n"
    "P5-FU *  *  *  *  *  *  *  * \n"
    "P6 *  * +FU * +FU+GI+FU * +FU\n"
    "P7 * +FU * +FU * +FU *  *  * \n"
    "P8 * +KA+OU * +KI+GI * +HI * \n"
    "P9+KY+KE * +KI *  *  * +KE+KY\n"
    "P-00FU00FU\n"
    "+\n").getInitialState();
  const PieceEval ev(state);
  int ret=ev.computeDiffAfterMove<BLACK>
    (state, Move(Square(5,6),Square(5,5),PAWN,PTYPE_EMPTY,false,BLACK));
  CPPUNIT_ASSERT_EQUAL(0, ret);
}

void PieceEvalTest::testCannotTake()
{
  CPPUNIT_ASSERT_EQUAL(0, Ptype_Eval_Table.value(PTYPE_EMPTY));
  NumEffectState state(CsaString(
			 "P1 *  *  *  *  *  *  *  *  * \n"
			 "P2 *  * +RY-KI-OU *  *  *  * \n"
			 "P3 *  *  *  *  *  *  *  *  * \n"
			 "P4 *  *  *  *  *  *  *  *  * \n"
			 "P5 *  *  *  *  *  *  *  *  * \n"
			 "P6 *  *  *  *  *  *  *  *  * \n"
			 "P7 *  *  *  *  *  *  *  *  * \n"
			 "P8 *  *  *  *  *  *  *  *  * \n"
			 "P9+OU *  *  *  *  *  *  *  * \n"
			 "P+00KI\n"
			 "P-00AL\n"
			 "+\n").getInitialState());
  {
    Move m = Move(Square(6,3),GOLD,BLACK);
    const int ret
      = PieceEval::computeDiffAfterMoveForRP(state, m);
    // 相手は取れない
    CPPUNIT_ASSERT_EQUAL(0, ret);
  }
}

void PieceEvalTest::testComputeDiffAfterMoveForRP()
{
  NumEffectState state(CsaString(
			 "P1-KY-KE-GI-KI *  * -HI-KE * \n"
			 "P2 * -HI *  * -KA-OU *  *  * \n"
			 "P3-FU * -FU-FU-FU * -GI-FU-KY\n"
			 "P4 * +KE *  *  * -FU *  * -FU\n"
			 "P5 * -FU+KE+KA *  *  *  * +FU\n"
			 "P6 *  * +FU *  *  * +FU *  * \n"
			 "P7+FU+FU * +FU+FU+FU *  *  * \n"
			 "P8 * +GI+KI *  *  * +GI *  * \n"
			 "P9+KY *  *  * +OU+KI *  * +KY\n"
			 "P+00FU\n"
			 "P+00FU\n"
			 "P+00KI\n"
			 "+\n").getInitialState());
  {
    Move m = Move(Square(1,5),Square(1,4),PAWN,PAWN,false,BLACK);
    const int ret
      = PieceEval::computeDiffAfterMoveForRP(state, m);
    // 相手が取り替えさないのが最善
    CPPUNIT_ASSERT_EQUAL(Pawn*2, ret);
  }
  {
    Move m = Move(Square(8,4),Square(7,2),PKNIGHT,PTYPE_EMPTY,true,BLACK);
    const int ret
      = PieceEval::computeDiffAfterMoveForRP(state, m);
    // ただ捨て
    CPPUNIT_ASSERT_EQUAL(-Knight*2, ret);
  }
  state.changeTurn();
  {
    Move m = Move(Square(8,5),Square(8,6),PAWN,PTYPE_EMPTY,false,WHITE);
    const int ret
      = PieceEval::computeDiffAfterMoveForRP(state, m);
    // ただ捨て
    CPPUNIT_ASSERT_EQUAL(-Pawn*2, ret);
  }
  {
    Move m = Move(Square(8,2),Square(8,4),ROOK,KNIGHT,false,WHITE);
    const int ret
      = PieceEval::computeDiffAfterMoveForRP(state, m);
    // ただ取り
    CPPUNIT_ASSERT_EQUAL(Knight*2, ret);
  }
}

void PieceEvalTest::testComputeDiffAfterMoveForRPBug030910()
{
  NumEffectState state(CsaString(
			 "P1 *  *  *  *  *  * -HI-KE-KY\n"
			 "P2 *  *  * +RY+UM-GI-KI-OU * \n"
			 "P3+TO *  *  * -FU * -GI *  * \n"
			 "P4-FU *  * -FU * -GI+KY-FU * \n"
			 "P5 * -FU+FU-KI * -FU-FU *  * \n"
			 "P6+FU+FU *  * +KY * +FU * -FU\n"
			 "P7 *  *  *  * +FU+FU * +FU * \n"
			 "P8 *  *  *  * +KI * +GI+OU+FU\n"
			 "P9 *  *  *  *  * +KI * +KE+KY\n"
			 "P-00FU\n"
			 "P-00KE\n"
			 "P-00KE\n"
			 "P+00KA\n"
			 "-\n").getInitialState());
  {
    const Square p86 = Square(8,6);
    Move m = Move(Square(8,5),p86,PAWN,PAWN,false,WHITE);
    // こちらは通る
    const int ret = PieceEval::computeDiffAfterMoveForRP(state, m);
    CPPUNIT_ASSERT_EQUAL(Pawn*2, ret);
    const int rets = 
      PieceEval::computeDiffAfterMoveForRP(state, m);
    CPPUNIT_ASSERT_EQUAL(Pawn*2, rets);

  }
  {
    const Square p36 = Square(3,6);
    Move m = Move(Square(3,5),p36,PAWN,PAWN,false,WHITE);
    // EffectUtil::showEffect(state, p36, std::cerr);
    CPPUNIT_ASSERT_EQUAL(-Pawn*2, 
			 PieceEval::diffWithMove(state, m));

    const int ret = PieceEval::computeDiffAfterMoveForRP(state, m);
    // ただ取りなので得するはず
    CPPUNIT_ASSERT_EQUAL(Pawn*2, ret);
    const int rets = 
      PieceEval::computeDiffAfterMoveForRP(state, m);
    CPPUNIT_ASSERT_EQUAL(Pawn*2, rets);
  }
}

void PieceEvalTest::testConsistentUpdate()
{
  consistencyTestUpdate<PieceEval>();
}

void PieceEvalTest::testConsistentExpect()
{
  consistencyTestExpect<PieceEval>();
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
