#include "osl/record/kisen.h"
#include "osl/record/ki2.h"
#include "osl/stl/vector.h"
#include "osl/oslConfig.h"
#include <boost/filesystem.hpp>

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

#include <iostream>
#include <iterator>

class KisenTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( KisenTest );
  CPPUNIT_TEST( testShow );
  CPPUNIT_TEST( testConsistent );
  CPPUNIT_TEST( testGetPlayer );
  CPPUNIT_TEST( testGetRating );
  CPPUNIT_TEST( testGetStartDate );
  CPPUNIT_TEST( testResult );
  CPPUNIT_TEST( testGetIpxFileName );
  CPPUNIT_TEST( testSave );
  CPPUNIT_TEST_SUITE_END();
 private:
  osl::Record rec;
  osl::SimpleState state;
  osl::vector<osl::Move> moves;
 public:
  void setUp();
  void testShow();
  void testConsistent();
  void testGetPlayer();
  void testGetRating();
  void testGetStartDate();
  void testResult();
  void testGetIpxFileName();
  void testSave();
};

CPPUNIT_TEST_SUITE_REGISTRATION( KisenTest );

using namespace osl;

void KisenTest::setUp(){
}

const char *filename = OslConfig::testPublicFile("floodgate2010/gps_l-vs-Bonanza.kif");
const char *ipxFilename = OslConfig::testPublicFile("floodgate2010/gps_l-vs-Bonanza.ipx");

void KisenTest::testShow(){
  KisenFile kisenFile(filename);
  moves = kisenFile.getMoves(0);  
  CPPUNIT_ASSERT( !moves.empty() );
  if (OslConfig::verbose())
      copy(moves.begin(), moves.end(), 
	   std::ostream_iterator<Move>( std::cout , "\n" ));
}

void KisenTest::testConsistent(){
  KisenFile kisenFile(filename);
  const size_t count=OslConfig::inUnitTestShort()
    ? (size_t)100u : std::min((size_t)10000u, kisenFile.size());
  for (size_t i=0;i<count;i++) {
    if (i==172518) 
      continue;
    NumEffectState state=kisenFile.getInitialState();
    vector<Move> moves=kisenFile.getMoves(i);
    for (size_t j=0;j<moves.size();j++) {
      state.makeMove(moves[j]);
      CPPUNIT_ASSERT(state.isConsistent(true));
    }
  }
}

void KisenTest::testGetPlayer(){
  KisenIpxFile kisenIpxFile(ipxFilename);
  CPPUNIT_ASSERT("Bonanza"==kisenIpxFile.getPlayer(2,BLACK) ||
		 (std::cerr <<kisenIpxFile.getPlayer(2,BLACK) << std::endl,0)
		 );
  CPPUNIT_ASSERT("gps_l"==kisenIpxFile.getPlayer(2,WHITE) ||
		 (std::cerr <<kisenIpxFile.getPlayer(2,WHITE) << std::endl,0)
		 );
}

void KisenTest::testGetRating(){
  KisenIpxFile kisenIpxFile(ipxFilename);
  CPPUNIT_ASSERT_EQUAL(777u,kisenIpxFile.getRating(6,BLACK));
  CPPUNIT_ASSERT_EQUAL(777u,kisenIpxFile.getRating(6,WHITE));
}

void KisenTest::testGetStartDate(){
  using namespace boost::gregorian;
  KisenIpxFile kisenIpxFile(ipxFilename);
  const date answer(2010, Jan, 03);
  CPPUNIT_ASSERT_EQUAL(answer.year(),  kisenIpxFile.getStartDate(6).year());
  CPPUNIT_ASSERT_EQUAL(answer.month(), kisenIpxFile.getStartDate(6).month());
  CPPUNIT_ASSERT_EQUAL(answer.day(),   kisenIpxFile.getStartDate(6).day());
}

void KisenTest::testResult(){
  KisenIpxFile kisenIpxFile(ipxFilename);
  KisenFile kisenFile(filename);
  const size_t count=OslConfig::inUnitTestShort
    ? (size_t)100u : std::min((size_t)5000u, kisenFile.size());
  for(size_t i=0;i<count;i++){
    moves = kisenFile.getMoves(i);  
    if (moves.size() >= 256) continue;
    unsigned int result=kisenIpxFile.getResult(i);
    if(result==KisenIpxFile::BLACK_WIN || result==KisenIpxFile::BLACK_WIN_256){
      CPPUNIT_ASSERT((moves.size()%2)==1);
    }
    else if(result==KisenIpxFile::WHITE_WIN || result==KisenIpxFile::WHITE_WIN_256){
      CPPUNIT_ASSERT((moves.size()%2)==0);
    }
  }
}

void KisenTest::testGetIpxFileName() {
  KisenFile kisenFile(filename);
  CPPUNIT_ASSERT_EQUAL(std::string(ipxFilename), kisenFile.ipxFileName());
}

void KisenTest::testSave() {
  using namespace boost::gregorian;
  record::ki2::Ki2File ki2("record/ki2_test1.ki2", false);
  {
    std::ofstream ofs("record/test_kisen_t_temp.ipx");
    record::KisenIpxWriter writer(ofs);
    writer.save(ki2.getRecord(), 100, 1100, "black", "white");
  }
  record::KisenIpxFile ipx("record/test_kisen_t_temp.ipx");
  CPPUNIT_ASSERT(date(2010, Nov, 13) == ipx.getStartDate(0));
  CPPUNIT_ASSERT_EQUAL(std::string("black"), ipx.getTitle(0, BLACK));
  CPPUNIT_ASSERT_EQUAL(std::string("white"), ipx.getTitle(0, WHITE));
  CPPUNIT_ASSERT_EQUAL(100u, ipx.getRating(0, BLACK));
  CPPUNIT_ASSERT_EQUAL(1100u, ipx.getRating(0, WHITE));
  {
    using namespace boost::filesystem;
    path ipx("record/test_kisen_t_temp.ipx");
    if (exists(ipx))
      remove(ipx);
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; coding:utf-8
// ;;; End:
