/*
*   pam_abl - a PAM module and program for automatic blacklisting of hosts and users
*
*   Copyright (C) 2005 Andy Armstrong andy@hexten.net
*   Copyright (C) 2009 Chris Tasma pam-abl@deksai.com
*
*   This program is free software: you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation, either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "pam_abl.h"


static int record_host(const abl_args *args, time_t tm) {
    if (NULL != args->host_db) {
        const char *rhost;
        int err;

        if (err = pam_get_item(args->pamh, PAM_RHOST, (const void **) &rhost), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_RHOST");
            return err;
        }
        if (NULL != rhost) {
            return record(args, args->host_db, rhost, tm, args->host_purge);
        } else {
            log_debug(args, "PAM_RHOST is NULL");
            return 0;
        }
    } else {
        return 0;
    }
}

static int record_user(const abl_args *args, time_t tm) {
    if (NULL != args->user_db) {
        const char *user;
        int err;
        if (err = pam_get_item(args->pamh, PAM_USER, (const void **) &user), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_USER");
            return err;
        }
        if (NULL != user) {
            return record(args, args->user_db, user, tm, args->user_purge);
        } else {
            log_debug(args, "PAM_USER is NULL");
            return 0;
        }
    } else {
        return 0;
    }
}

static int record_attempt(const abl_args *args) {
    int err;
    time_t tm = time(NULL);

    log_debug(args, "Recording failed attempt");

    if (err = record_host(args, tm), 0 != err) {
        return err;
    }

    if (err = record_user(args, tm), 0 != err) {
        return err;
    }

    return 0;
}

static int check_attempt(const abl_args *args, abl_info *info) {
    int err;
    time_t tm = time(NULL);

    if (info->user != NULL && info->service != NULL && info->host != NULL) {
        info->subject = HOST;
        err = check_host(args, info, tm);
        if (err) log_warning(args,"Failed to check host.");
        update_status(args, info);
        if (info->state_change) {
            /*Do nothing with error, since it is logged, and there's nothing else to do*/
            run_command(args,info);
            /*No point in continuing since they've failed.*/
            return 0;
        }

        info->subject = USER;
        err = check_user(args, info, tm);
        if (err) log_warning(args,"Failed to check user.");
        update_status(args, info);
        if (info->state_change) run_command(args,info);
    }
    return 0;
}


static void cleanup(pam_handle_t *pamh, void *data, int err) {
    if (NULL != data) {
        abl_args *args = data;
        log_debug(args, "In cleanup, err is %08x", err);

        if (err && (err & PAM_DATA_REPLACE) == 0) {
            record_attempt(args);
        }
        config_free(args);
        free(args);
    }
}
/* Authentication management functions */

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc, const char **argv) {
    abl_args *args;
    abl_info info;
    char ipstring[64];
    int err = PAM_SUCCESS;

    /*log_debug(args, "pam_sm_authenticate(), flags=%08x", flags);*/

    memset(&info,0,sizeof(info));

    if (args = malloc(sizeof(abl_args)), NULL == args) {
        return PAM_BUF_ERR;
    }

    if (err = config_parse_args(pamh, argc, argv, args), PAM_SUCCESS == err) {
        if (err = pam_set_data(pamh, DATA_NAME, args, cleanup), PAM_SUCCESS != err) {
            log_pam_error(args,err,"setting PAM data");
            goto psa_fail;
        }
        if (err = pam_get_item(args->pamh, PAM_USER, (const void **) &info.user), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_USER");
            goto psa_fail;
        }
        if (err = pam_get_item(args->pamh, PAM_SERVICE, (const void **) &info.service), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_SERVICE");
            goto psa_fail;
        }
        if (err = pam_get_item(args->pamh, PAM_RHOST, (const void **) &info.host), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_RHOST");
            goto psa_fail;
        }

        check_attempt(args, &info);
        if (info.state == BLOCKED) {
            log_info("Blocking access from %s to service %s, user %s", info.host, info.service, info.user);
            return PAM_AUTH_ERR;
        }
        else {
            return PAM_SUCCESS;
        }
    } 

psa_fail:
    config_free(args);
    free(args);
    return err;
}

PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, const char **argv) {
    return pam_set_data(pamh, DATA_NAME, NULL, cleanup);
}

/* Init structure for static modules */
#ifdef PAM_STATIC
struct pam_module _pam_abl_modstruct = {
    MODULE_NAME,
    pam_sm_authenticate,
    pam_sm_setcred,
    NULL,
    NULL,
    NULL,
    NULL
};
#endif

