/*
 * shm_askpass: SSH-ASKPASS helper using shared POSIX shared memory.
 * Copyright (C) 2011 Scott Balneaves <sbalneav@ltsp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <pwd.h>
#include <syslog.h>

#define DEFAULTPATH "/shm_askpass"
#define PWSIZE BUFSIZ

void
cleanup (void)
{
  closelog();
}

void
shm_read (char *path)
{
  int fd;
  char *ptr;

  if ((fd = shm_open (path, O_RDONLY, S_IREAD)) < 0)
    {
      int errsv = errno;
      syslog (LOG_ERR, "shm_open() of %s failed: %s", path, strerror (errsv));
      exit (1);
    }

  if ((ptr = (char *) mmap (NULL, PWSIZE, PROT_READ,
                            MAP_SHARED, fd, 0)) == MAP_FAILED)
    {
      int errsv = errno;
      syslog (LOG_ERR, "mmap of %s failed: %s", path, strerror (errsv));
      exit (1);
    }

  printf ("%s\n", ptr);

  close (fd);

  if (getuid () != 0)
    {
      if (shm_unlink (path) < 0)
        {
          int errsv = errno;
          syslog (LOG_ERR, "shm_unlink of of %s failed: %s", path, strerror (errsv));
          exit (1);
        }
    }
}

void
shm_write (char *path)
{
  char *ptr;
  struct passwd *pwd;
  char *uname;
  int fd;

  if ((uname = getenv ("PAM_USER")) == NULL)
    {
      syslog (LOG_ERR, "No PAM_USER environment variable");
      exit (1);
    }

  if ((pwd = getpwnam (uname)) == NULL)
    {
      int errsv = errno;
      syslog (LOG_ERR, "No user %s found: %s", uname, strerror (errsv));
      exit (1);
    }

  if ((fd = shm_open (path, (O_CREAT | O_EXCL | O_RDWR),
                      (S_IREAD | S_IWRITE))) < 0)
    {
      int errsv = errno;
      syslog (LOG_ERR, "shm_open() of %s failed: %s", path, strerror (errsv));
      exit (1);
    }

  if (fchown (fd, pwd->pw_uid, pwd->pw_gid) < 0)
    {
      int errsv = errno;
      syslog (LOG_ERR, "fchown failed: %s", strerror (errsv));
      exit (1);
    }

  if (ftruncate (fd, PWSIZE) < 0)
    {
      int errsv = errno;
      syslog (LOG_ERR, "ftruncate failed: %s", strerror (errsv));
      exit (1);
    }

  if ((ptr = (char *) mmap (0, PWSIZE, (PROT_READ | PROT_WRITE),
                            MAP_SHARED, fd, 0)) == MAP_FAILED)
    {
      int errsv = errno;
      syslog (LOG_ERR, "mmap of %s failed: %s", path, strerror (errsv));
      exit (1);
    }

  bzero (ptr, PWSIZE);

  if (read (STDIN_FILENO, ptr, (PWSIZE - 1)) < 0)
    {
      int errsv = errno;
      syslog (LOG_ERR, "read from stdin failed: %s", strerror (errsv));
      exit (1);
    }

  close (fd);
}

void
shm_delete (char *path)
{
  if (shm_unlink (path) != 0)
    {
      int errsv = errno;
      syslog (LOG_ERR, "shm_unlink of of %s failed: %s", path, strerror (errsv));
      exit (1);
    }
}

void
usage ()
{
  fprintf (stderr,
"shm_askpass - An SSH-ASKPASS compliant helper using POSIX shared memory.\n"
"              Requires that the PAM_USER environment variable be set,\n"
"              see manpage for details.\n"
"\n"
"USAGE:\n"
"shm_askpass --write  - Reads password from stdin, stores as user ${PAM_USER}\n"
"            --delete - Deletes the shared memory segment\n"
"            --help   - This message\n"
"                     - with no options returns the stored password.  If run\n"
"                       as root, password is kept.  If run as a non-root id,\n"
"                       password is wiped from memory and removed.\n"
  );
  exit (0);
}

int
main (int argc, char **argv)
{
  char *path = getenv ("SHM_ASKPASS_PATH");

  /*
   * Open log
   */

  openlog (argv[0], LOG_NDELAY | LOG_PID, LOG_AUTHPRIV);

  /*
   * Register exit funtion to close log on exit.
   */

  if (atexit (cleanup) != 0)
    {
      fprintf (stderr, "Couldn't register exit handler\n");
      exit (1);
    }

  /*
   * If a path has been set, make sure it starts with a '/'
   */

  if (path && *path != '/')
    {
      syslog (LOG_ERR, "SHM_ASKPASS_PATH must start with a '/'");
      exit (1);
    }

  /*
   * handle command line
   */

  while (1)
    {
      static struct option long_options[] = {
        {
         "help", no_argument, NULL, 'h'},
        {
         "write", no_argument, NULL, 'w'},
        {
         "delete", no_argument, NULL, 'd'},
        {
         NULL, 0, NULL, 0}
      };
      /* getopt_long stores the option
       * index here. */
      int option_index = 0;
      int c;

      c = getopt_long (argc, argv, "hwd", long_options, &option_index);

      if (c == -1)
        {
          break;
        }

      switch (c)
        {
        case 'h':
          usage ();
          exit (0);

        case 'w':
          shm_write (path ? path : DEFAULTPATH);
          exit (0);

        case 'd':
          shm_delete (path ? path : DEFAULTPATH);
          exit (0);

        case '?':
          usage ();
          exit (0);

        default:
          usage ();
          exit (0);
        }
    }

  shm_read (path ? path : DEFAULTPATH);
  exit (0);
}
