/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "libpolyxmass-note.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmNote*
libpolyxmass_note_new (void)
{
  PxmNote *note = g_malloc0 (sizeof (PxmNote));

  note->notevalGPA = g_ptr_array_new ();

  return note;
}

PxmNoteVal*
libpolyxmass_noteval_new (void)
{
  PxmNoteVal *noteval = g_malloc0 (sizeof (PxmNoteVal));

  return noteval;
}


PxmProp*
libpolyxmass_note_prop_new (void)
{
  PxmProp *prop = libpolyxmass_prop_new ();
  
  PxmNote *note = libpolyxmass_note_new ();

  libpolyxmass_prop_set_name (prop, "NOTE");
  
  prop->data = (gpointer) note;

  prop->custom_dup = libpolyxmass_note_prop_dup;
  prop->custom_cmp = libpolyxmass_note_prop_cmp;
  prop->custom_free = libpolyxmass_note_prop_free;
  
  return prop;
}


PxmNoteVal*
libpolyxmass_noteval_dup (PxmNoteVal *noteval, gint how_dup)
{
  PxmNoteVal *noteval_new = NULL;
  PxmNoteValCmp how = (PxmNoteValCmp) how_dup;
  

  g_assert (noteval != NULL);
  
  noteval_new = libpolyxmass_noteval_new ();
 
  if (how & PXM_NOTEVAL_DUP_TYPE)
    {
      noteval_new->type = noteval->type;
    }
  
  if (how & PXM_NOTEVAL_DUP_VALUE)
    {
      g_assert (noteval->value != NULL);
      
      if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
	noteval_new->value = 
	  (gpointer) g_strdup ((gchar *) noteval->value);
      else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
	{
	  noteval_new->value = g_malloc0 (sizeof (gint));
	  (*(gint *) noteval_new->value = *(gint *) noteval->value);
	}
      else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
	{
	  noteval_new->value = g_malloc0 (sizeof (gdouble));
	  (*(gdouble *) noteval_new->value = *(gdouble *) noteval->value);
	}
      else
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	       _("%s@%d: the noteval->type is not valid\n"),
	       __FILE__, __LINE__);
    }
  
  return noteval_new;
}


PxmNote*
libpolyxmass_note_dup (PxmNote *note, gint how_dup)
{
  PxmNote *note_new = NULL;
  PxmNoteVal *noteval = NULL;
  PxmNoteVal *noteval_new = NULL;
  
  PxmNoteDup how = (PxmNoteDup) how_dup;
  
  gint iter = 0;
  
  g_assert (note != NULL);
  
  note_new = libpolyxmass_note_new ();
  
  g_assert (note->name != NULL);
  note_new->name = g_strdup (note->name);
  
  g_assert (note->notevalGPA != NULL);

  if (how == PXM_NOTE_DUP_DEEP_YES)
    {
      for (iter = 0; iter < note->notevalGPA->len ; iter ++)
	{
	  noteval = g_ptr_array_index (note->notevalGPA, iter);
	  g_assert (noteval != NULL);
	  
	  noteval_new = libpolyxmass_noteval_dup (noteval, 
						  PXM_NOTEVAL_CMP_BOTH);
	  
	  g_ptr_array_add (note_new->notevalGPA, noteval_new);
	}
    }
  
  return note_new;
}


PxmProp*
libpolyxmass_note_prop_dup (PxmProp *prop, gint how_dup)
{
  PxmProp *prop_new = NULL;
  PxmNote *note_new = NULL;
  
  g_assert (prop != NULL);
  
  g_assert (prop->name != NULL && 0 == strcmp ("NOTE", prop->name));
  g_assert (prop->data != NULL);
  

  prop_new = libpolyxmass_prop_new ();
  
  libpolyxmass_prop_set_name (prop_new, "NOTE");
  
  note_new = libpolyxmass_note_dup ((PxmNote *) prop->data, how_dup);

  prop_new->data = (gpointer) note_new;

  g_assert (prop->custom_dup != NULL);
  prop_new->custom_dup = prop->custom_dup;

  g_assert (prop->custom_cmp != NULL);
  
  prop_new->custom_cmp = prop->custom_cmp;

  g_assert (prop->custom_free != NULL);
  prop_new->custom_free = prop->custom_free;
    
  return prop_new;
}


/*  LOCATING FUNCTIONS
 */
gint
libpolyxmass_noteval_get_index_by_noteval (PxmNoteVal *noteval, 
					   GPtrArray *GPA)
{
  gint iter = 0;

  PxmNoteVal *noteval_iter = NULL;
  
  /* We have to localize a noteval object in GPA that is identical to
     the one passed as parameter (not by pointer value, of course,
     but by contents of the noteval (both the type and the value of the 
     noteval object).
  */
  g_assert (noteval != NULL);
  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      noteval_iter = g_ptr_array_index (GPA, iter);
      g_assert (noteval_iter != NULL);

      if (0 == libpolyxmass_noteval_cmp (noteval, noteval_iter, 
				     PXM_NOTEVAL_CMP_BOTH))
	return iter;
    }
  
  return -1;
}



/* DATA MODIFYING FUNCTIONS
 */
gchar* 
libpolyxmass_note_set_name (PxmNote *note, gchar *name)
{
  g_assert (note != NULL && name != NULL);

  /* If name already allocated, free it before.
   */
  if (note->name != NULL)
    g_free (note->name);

  note->name = g_strdup (name);

  return note->name;
}



/* COMPARISON FUNCTIONS
 */
gint
libpolyxmass_noteval_cmp (PxmNoteVal *noteval1,
		      PxmNoteVal *noteval2,
		      gint how_cmp)
{
  gint result = 0;
  PxmNoteValCmp what = (PxmNoteValCmp) how_cmp;

  g_assert (noteval1 != NULL && noteval2 != NULL);
  
  /* On the ground that other similar functions return 0 if the
     operands are identical, we will do the same. Return > 0 if the
     operands are different.
  */

  if (what & PXM_NOTEVAL_CMP_TYPE)
    {
      result += (noteval1->type != noteval2->type);
    }

  if (what & PXM_NOTEVAL_CMP_VALUE)
    {
      if (noteval1->value == NULL && noteval2->value == NULL)
	return result;
      
      if (noteval1->value != NULL && noteval2->value == NULL)
	return ++result;
      
      if (noteval1->value == NULL && noteval2->value != NULL)
	return ++result;
      
      if (noteval1->type == PXM_NOTE_VAL_TYPE_STR)
	result += strcmp ((gchar *) noteval1->value, 
			  (gchar *) noteval2->value);
      
      else if (noteval1->type == PXM_NOTE_VAL_TYPE_INT)
	result += (*(gint *) noteval1->value != *(gint *) noteval2->value);
      
      else if (noteval1->type == PXM_NOTE_VAL_TYPE_DBL)
	result += (*(gdouble *) noteval1->value 
		   != *(gdouble *) noteval2->value);
      else
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	       _("%s@%d: the noteval->type is not valid\n"),
	       __FILE__, __LINE__);
    }

  return result;
}


gint
libpolyxmass_note_cmp (PxmNote *note1,
		   PxmNote *note2,
		   gint how_cmp)
{
  gint result = 0;
  gint iter = 0;
  gint jter = 0;

  PxmNoteCmp how = (PxmNoteCmp) how_cmp;

  PxmNoteVal *noteval1 = NULL;
  PxmNoteVal *noteval2 = NULL;

  gboolean found_noteval = FALSE;
 

  g_assert (note1 != NULL);
  g_assert (note2 != NULL);


  g_assert (how & PXM_NOTE_CMP_NONE
	    || how & PXM_NOTE_CMP_NAME
	    || how & PXM_NOTE_CMP_NOTEVAL_ARRAY_SUBSET1
	    || how & PXM_NOTE_CMP_NOTEVAL_ARRAY_SUBSET2);
  
  if (how & PXM_NOTE_CMP_NONE)
    {
      return -1;
    }
  
  if (how & PXM_NOTE_CMP_NAME)
    {
      result += strcmp (note1->name, note2->name);
    }
  
  if (how & PXM_NOTE_CMP_NOTEVAL_ARRAY_SUBSET1)
    {
      /* It is admitted that the two notes are not absolutely
	 identical, only if all the noteval objects in note1 are in
	 note2 (ie note1 is a subset of note2). Thus, a very first
	 easy check is to ensure that the number of objects in
	 note1->notevalGPA is lower or equal to note2->notevalGPA.
      */
      if (note1->notevalGPA->len > note2->notevalGPA->len)
	{
	  return ++result;
	}

      for (iter = 0; iter < note1->notevalGPA->len; iter++)
	{
	  found_noteval = FALSE;
	  
	  noteval1 = g_ptr_array_index (note1->notevalGPA, iter);
	  g_assert (noteval1 != NULL);
	  
	  for (jter = 0; jter < note2->notevalGPA->len; jter++) 
	    {
	      noteval2 = g_ptr_array_index (note2->notevalGPA, jter);
	      g_assert (noteval2 != NULL);
	      
	      if (0 == libpolyxmass_noteval_cmp (noteval1, noteval2,
					     PXM_NOTEVAL_CMP_BOTH))
		{
		  found_noteval = TRUE;
		  
		  break;
		}
	      else
		continue;
	    }
	  
	  if (found_noteval == FALSE)
	    {
	      return ++result;
	    }
	}
    }

  if (how & PXM_NOTE_CMP_NOTEVAL_ARRAY_SUBSET2)
    {
      /* It is admitted that the two notes are not absolutely
	 identical, only if all the noteval objects in note2 are in
	 note1 (ie note2 is a subset of note1). Thus, a very first
	 easy check is to ensure that the number of objects in
	 note2->notevalGPA is lower or equal to note1->notevalGPA.
      */
      if (note2->notevalGPA->len > note1->notevalGPA->len)
	{
	  return ++result;
	}

      for (iter = 0; iter < note2->notevalGPA->len; iter++)
	{
	  found_noteval = FALSE;
	  
	  noteval2 = g_ptr_array_index (note2->notevalGPA, iter);
	  g_assert (noteval2 != NULL);
	  
	  for (jter = 0; jter < note1->notevalGPA->len; jter++) 
	    {
	      noteval1 = g_ptr_array_index (note1->notevalGPA, jter);
	      g_assert (noteval1 != NULL);
	      
	      if (0 == libpolyxmass_noteval_cmp (noteval2, noteval1,
					     PXM_NOTEVAL_CMP_BOTH))
		{
		  found_noteval = TRUE;
		  
		  break;
		}
	      else
		continue;
	    }
	  
	  if (found_noteval == FALSE)
	    {
	      return ++result;
	    }
	}
    }

  return result;
}


gint
libpolyxmass_note_prop_cmp (PxmProp *prop1,
			PxmProp *prop2,
			gint how_cmp)
{
  /* This function is called when two PxmProp instances named
     "NOTE" must be compared.
  */
  PxmNote *note1 = NULL;
  PxmNote *note2 = NULL;
  
  gint result = 0;

  g_assert (prop1 != NULL && prop2 != NULL);

  note1 = (PxmNote *) prop1->data;
  note2 = (PxmNote *) prop2->data;

  g_assert (note1 != NULL);
  g_assert (note2 != NULL);
  
  /* Remember that all the prop_cmp functions can be called with a
     number of default parameters:
     
     how_cmp & PXM_DEFAULT_PROP_CMP_NONE
     
     how_cmp & PXM_DEFAULT_PROP_CMP_NAME

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA

     how_cmp & PXM_DEFAULT_PROP_CMP_DEEP

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2

  */
  if (how_cmp & PXM_DEFAULT_PROP_CMP_NONE
      || how_cmp & PXM_NOTE_PROP_CMP_NONE)
    {
      return -1;
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_NAME
      || how_cmp & PXM_NOTE_PROP_CMP_NAME)
    {
      g_assert (prop1->name != NULL && prop2->name != NULL );

      result += strcmp (prop1->name, prop2->name);
      
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA
      || how_cmp & PXM_NOTE_PROP_CMP_DATA)
    {
      result += libpolyxmass_note_cmp (note1, note2, PXM_NOTE_CMP_NAME);
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_DEEP
      || how_cmp & PXM_NOTE_PROP_CMP_DEEP)
    {
      if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
	  || how_cmp & PXM_NOTE_PROP_CMP_PROP_ARRAY_SUBSET1)
	result += libpolyxmass_note_cmp (note1, note2, 
				     PXM_NOTE_CMP_NOTEVAL_ARRAY_SUBSET1);
      
      if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2
	  || how_cmp & PXM_NOTE_PROP_CMP_PROP_ARRAY_SUBSET2)
	result += libpolyxmass_note_cmp (note1, note2, 
				     PXM_NOTE_CMP_NOTEVAL_ARRAY_SUBSET2);
    }

  return result;
}


/* RENDERING FUNCTIONS - FORMATTER FUNCTIONS
 */
gchar*
libpolyxmass_note_prop_format_xml_note (PxmProp *prop,
					gchar *indent,
					gint offset,
					gpointer user_data)
{
  /* This formatter is used when a polymer sequence has property
     object (type PxmProp) of name "NOTE" (for which the data
     points to a PxmNote object). If such polymer sequence has to be
     saved to a file, under format xml, then when the saving function
     encounter a "NOTE"-named PxmProp object, it calls this
     saving function which is responsible for creating a character
     string to be sent to the xml file.

     The name of a PxmNote is a string of character, which is saved
     very easily in the xml file, under element <name>. The PxmNote
     object contains an array of PxmNoteval objects, which have to be
     deconstructed one by one (see array-iterating loop below) into a
     <data> element with an attribute that can either be "str" or
     "int" or "dbl", respectively to indicate that the data in the
     PxmNoteval object is text or numerical gint or gdouble.
  */
  PxmNote *note = NULL;
  
  GString *gs = NULL;
  
  gchar *lead = NULL;
  gchar *help = NULL;
  PxmNoteVal *noteval = NULL;
  
  gint new_offset = 0;
  gint iter = 0;
  
  g_assert (prop != NULL);
  g_assert (prop->name != NULL && prop->data != NULL);
  
  /* OK, the property to be formatter as XML text is not NULL and none
     of its member either. Let's get to the PxmNote object that is
     pointed by the prop->data member.
  */
  note = (PxmNote *) prop->data;
  g_assert (note->name != NULL && note->notevalGPA != NULL);
  
  /* OK, both the name and value members of the PxmNote object
     are non-NULL. We can start doing our work.
  */
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create a SINGLE <note> node that should look 
     like this:

     <prop>
     <name>NOTE</name>
     <data type="str" "WHATEVER_STRING: the note's name</data>
     <data type="int" WHATEVER_STRING: the note's name</data>
     <data type="dbl" WHATEVER_STRING: the note's value</data>
     </prop>
     
  */

  new_offset = offset;

  /* Open the <prop> element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);

  g_string_append_printf (gs, "%s<prop>\n", lead);

  g_free (lead);
  new_offset++;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  /* The prop name member.
   */
  g_string_append_printf (gs, "%s<name>%s</name>\n",
			  lead, prop->name);
  
  /* The note name member.
   */
  g_string_append_printf (gs, "%s<data>%s</data>\n",
			  lead, (gchar*) note->name);
  
  /* We now have to iterate in the note->valGPA and for each string
     found in the array we have to make a <data> element.
  */
  for (iter = 0 ; iter < note->notevalGPA->len ; iter++)
    {
      noteval = g_ptr_array_index (note->notevalGPA, iter);
      g_assert (noteval != NULL);
      
      /* Make the string to append to the gs xml string. The way the string
	 is made depends on the type of the value:
      */
      if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
	g_string_append_printf (gs, 
				"%s<data %s>%s</data>\n",
				lead, 
				"type=\"str\"",
				(gchar*) noteval->value);
      else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
	g_string_append_printf (gs, 
				"%s<data %s>%d</data>\n",
				lead, 
				"type=\"int\"",
				*(gint *) noteval->value);
      else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
	g_string_append_printf (gs, 
				"%s<data %s>%.9f</data>\n",
				lead, 
				"type=\"dbl\"",
				*(gdouble *) noteval->value);
      else g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
		  _("%s@%d: the noteval->type is not valid\n"),
		  __FILE__, __LINE__);
    }
  
  g_free (lead);
  new_offset--;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  /* Close the note element.
   */
  g_string_append_printf (gs, "%s</prop>\n", lead);
  
  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


gchar*
libpolyxmass_note_prop_format_txt_note (PxmProp *prop,
					gpointer user_data)
{
  /* This formatter is used when a polymer sequence has property
     object (type PxmProp) of name "NOTE" (for which the data points
     to a PxmNote object). If such polymer sequence has to be saved to
     a file, under format txt, then when the saving function encounter
     a "NOTE"-named PxmProp object, it calls this saving function
     which is responsible for creating a character string to be sent
     to the txt file (or txt string for clipboard operations).

     The name of a PxmNote is a string of character, which is saved
     very easily in the txt file, under heading "name:". The PxmNote
     object contains an array of PxmNoteval objects, which have to be
     deconstructed one by one (see array-iterating loop below) under a
     "data:" header, with an attribute that can either be "str" or
     "int" or "dbl", respectively to indicate that the data in the
     PxmNoteval object is text or numerical gint or gdouble.
  */
  PxmNote *note = NULL;
  
  GString *gs = NULL;
  
  gchar *help = NULL;
  PxmNoteVal *noteval = NULL;
  
  gint iter = 0;
  
  g_assert (prop != NULL);
  g_assert (prop->name != NULL && prop->data != NULL);
  
  /* OK, the property to be formatter as XML text is not NULL and none
     of its member either. Let's get to the PxmNote object that is
     pointed by the prop->data member.
  */
  note = (PxmNote *) prop->data;
  g_assert (note->name != NULL && note->notevalGPA != NULL);
  
  /* OK, both the name and value members of the PxmNote object
     are non-NULL. We can start doing our work.
  */
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create a SINGLE "note": text paragraph that
     should look like this:

     prop:
     -----
     name: NOTE
     data: NOTE_NAME
     data type="str" "WHATEVER_STRING: the note's value
     data type="int" WHATEVER_STRING: the note's value
     data type="dbl" WHATEVER_STRING: the note's value
  */

  g_string_append_printf (gs, "prop:\n"
                               "-----\n");
  
  /* The prop name member.
   */
  g_string_append_printf (gs, "name: '%s'\n", prop->name);
  
  /* The note name member.
   */
  g_string_append_printf (gs, "data: '%s'\n", (gchar*) note->name);
  
  /* We now have to iterate in the note->valGPA and for each string
     found in the array we have to make a <data> element.
  */
  for (iter = 0 ; iter < note->notevalGPA->len ; iter++)
    {
      noteval = g_ptr_array_index (note->notevalGPA, iter);
      g_assert (noteval != NULL);
      
      /* Make the string to append to the gs xml string. The way the string
	 is made depends on the type of the value:
      */
      if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
	g_string_append_printf (gs, 
				"data type=\"str\" %s\n", 
				(gchar*) noteval->value);
      else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
	g_string_append_printf (gs, 
				"data type=\"int\" %d\n", 
				*(gint *) noteval->value);
      else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
	g_string_append_printf (gs, 
				"data type=\"dbl\" %.9f\n",
				*(gdouble *) noteval->value);
      else g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
		  _("%s@%d: the noteval->type is not valid\n"),
		  __FILE__, __LINE__);
    }
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


PxmProp*
libpolyxmass_note_prop_render_xml_note (xmlDocPtr xml_doc,
					xmlNodePtr xml_node,
					gpointer user_data)
{
  /* This xml renderer is responsible for rendering a xml note that
     corresponds to a <prop> element of which the <name> element has
     the contents "NOTE". Thus, the property that is being
     created has a name "NOTE", and its pointer is pointing to
     an allocated PxmNote object.
  */

  /* The xml node we are in is structured this way:

  <prop>
  <name>NOTE</name>
  <data>WHATEVER_STRING: the note's name</data>
  <data type="str">WHATEVER_STRING: the 1st note's value</data>
  <data type="dbl">WHATEVER_STRING: the 2nd note's value</data>
  ... n-1
  <data type="str">WHATEVER_STRING: the nst note's value</data>
  </prop>

  The xml_node param should point to the xml element <prop> which
  is the entry node to the xml data describing the prop object.
   
  That is, xml_node->name == "prop", as exemplified below:
   
  <prop>
  ^
  |
  +------- here we are right now.
  <name>NOTE</name>
    
  Which means that xml_node->name == "prop" and that following the
  call to xml_node->children we should be pointing to the <name>
  child node of the <prop> node, thus:
  xml_node->xmlChildrenNode->content == "NOTE".
  */
  PxmProp *prop = NULL;
  PxmNote *note = NULL;
  PxmNoteVal *noteval = NULL;

  gchar *value = NULL;
  gchar *type = NULL;

  g_assert (xml_node != NULL);
  g_assert (xml_doc != NULL);

  g_assert (0 == strcmp ((gchar *) xml_node->name, "prop"));
  
  /* Allocate the PxmProp object, so that we'll fill it with read
     data.
  */
  prop = libpolyxmass_prop_new ();
  
  /* Go to the <name> node, which is the first <prop> element's child:
   */
  xml_node = xml_node->children;  

  /* From a rigourous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank) as
     long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode(xml_node))
    xml_node = xml_node->next;

  /* We have encountered the next children's element: it should be
     <name>. That <name> element should contain the "NOTE" string.
  */
  g_assert (0 == strcmp ((gchar *) xml_node->name, 
			 "name"));

  /* Get the name of the property so that we can start filling the
   * prop instance.
   */
  prop->name = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (prop->name != NULL);
  g_assert (0 == strcmp ("NOTE", prop->name));

  /* We are now certain that we are effectively dealing with a NOTE 
     prop object, so create such PxmNote object now.
  */
  note = libpolyxmass_note_new ();

  /* Go to next child of the prop element, which, according to DTD
     should be one (or more) <data> element(s). To go to the <data>
     element just go next. In this xml renderer there are two <data>
     elements that contain #PCDATA, which means that we just allocate
     strings.
  */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode(xml_node))
    xml_node = xml_node->next;

  /* We got to the first <data> element, which is going to be the 
     PxmNote note->name member.
  */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "data"));

  note->name = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (note->name != NULL);
  
  /* At this point go to the element(s) that is (are) to become the
     note value(s). There may be more than one value, and we have to
     parse all the available values as strings. Each time we find a string
     we must add it to the note->valGPA array of value strings.
  */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode(xml_node))
    xml_node = xml_node->next;

  while (xml_node != NULL)
    {
      /* If there is a node, it cannot be of any name other than "data".
       */
      g_assert (0 == strcmp ((gchar *) xml_node->name, "data"));

      /* Collect the value that is associated with the "data"
	 element. That string is to go to the note->valGPA array of
	 value strings. That string must not be NULL, otherwise the
	 existence _per se_ of the "data" element is questionable !
      */
      value = 
	(gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
      g_assert (value != NULL);

      /* We know that we are going to need a PxmNoteVal object. So just
	 create it;
      */
      noteval = libpolyxmass_noteval_new ();
      
      /* Get to know if there is an attribute (type) --we know it is
	 optional, if it is absent, then that equals to a string
	 type.
      */
      type = (gchar *) xmlGetProp (xml_node, (guchar*) "type");

      /*
	debug_printf (("type is now %s\n", type));
      */

      if (type == NULL)
	{
	  /* The value is by default of type string, so we just use
	     the value string obtained above to make the PxmNoteVal
	     object.
	  */
	  noteval->type = PXM_NOTE_VAL_TYPE_STR;
	  noteval->value = (gpointer) value;
	}
      else
	{
	  if (0 == strcmp ("str", type))
	    {
	      noteval->type = PXM_NOTE_VAL_TYPE_STR;
	      noteval->value = (gpointer) g_strdup (value);
	    }
	  else if (0 == strcmp ("int", type))
	    {
	      noteval->type = PXM_NOTE_VAL_TYPE_INT;
	      noteval->value = g_malloc0 (sizeof (gint));
	      if (FALSE == 
		  libpolyxmass_globals_strtoi (value,
					   (gint *) noteval->value,
					   10))
		g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
		       _("%s@%d: failed to convert from '%s' to gint\n"),
		       __FILE__, __LINE__, value);
	    }
	  else if (0 == strcmp ("dbl", type))
	    {
	      noteval->type = PXM_NOTE_VAL_TYPE_DBL;
	      noteval->value = g_malloc0 (sizeof (gdouble));
	      if (FALSE == 
		  libpolyxmass_globals_strtod (value,
					   (gdouble *) noteval->value))
		g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
		       _("%s@%d: failed to convert from '%s' to gdouble\n"),
		       __FILE__, __LINE__, value);
	    }
	  else
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
		   _("%s@%d: file badly formed.\n"),
		   __FILE__, __LINE__);

	  /* At this point we do not need the attribute anymore.
	   */
	  xmlFree (type);
      	}
      
      /* OK, now that we have our noteval object, we can just append
	 it to the note->notevalGPA:
      */
      g_ptr_array_add (note->notevalGPA, noteval);
      
      /* At this point, go the next "data" element, if any.
       */
      xml_node = xml_node->next;

      /* From a rigorous XML parsing point of view, the blanks found in
         the XML document are considered to be nodes, and we have to detect
         these and take proper action: go next sibling (next blank) as long
         as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }

  /* At this point, the PxmNote object 'note' has been filled OK, and
     now we have to set its pointer to the data member of the PxmProp
     object 'prop'.
  */
  prop->data = (gpointer) note;
  
  /* At this point we have parsed the <prop> node, of name "NOTE"
     into a prop object of name "NOTE" and with its data pointing
     to a PxmNote object.
  */

  /* Do not forget that when this property is used, it might be
     either compared or duplicated or freed, and we must set
     to it the pointers to the functions that can accomplish those
     tasks properly.
  */

  prop->custom_dup = libpolyxmass_note_prop_dup;
  prop->custom_cmp = libpolyxmass_note_prop_cmp;
  prop->custom_free = libpolyxmass_note_prop_free;

  return prop;
}


/* FREE'ING FUNCTIONS
 */
gint 
libpolyxmass_note_free (PxmNote *note)
{
  PxmNoteVal *noteval = NULL;
  
  g_assert (note != NULL);

  if (note->name != NULL)
    g_free (note->name);
  
  /* Free the array of value strings. The array cannot be NULL, even
     if it might be empty.
  */
  g_assert (note != NULL);
  
  while (note->notevalGPA->len > 0)
    {
      noteval = g_ptr_array_remove_index (note->notevalGPA, 0);
      g_assert (noteval != NULL);
      
      libpolyxmass_noteval_free (noteval);
    }
  
  g_ptr_array_free (note->notevalGPA, TRUE);
  
  g_free (note);
  
  return 1;
}


gint 
libpolyxmass_noteval_free (PxmNoteVal *noteval)
{
  g_assert (noteval != NULL);
  
  /* Whatever the type of value, it has to be freed using g_free,
     because it was allocated using g_malloc0.
  */
  if (noteval->value != NULL)
    g_free (noteval->value);

  g_free (noteval);
  
  return 1;
}


gint 
libpolyxmass_noteval_GPA_empty (GPtrArray *GPA)
{
  gint count = 0;
  PxmNoteVal *noteval = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      noteval = g_ptr_array_remove_index (GPA, 0);
      g_assert (noteval != NULL);
      
      libpolyxmass_noteval_free (noteval);

      count++;
    }

  return count;  
}


gint 
libpolyxmass_note_prop_free (PxmProp *prop)
{
  /* This function is called when a prop named "NOTE" must be
     freed.
  */
  g_assert (prop != NULL);
  
  if (prop->name != NULL)
    g_free (prop->name);
  
  if (prop->data != NULL)
    libpolyxmass_note_free ((PxmNote *) prop->data);
  
  g_free (prop);
    
  return 1;
}


gint 
libpolyxmass_note_prop_free_but_not_data (PxmProp *prop)
{
  /* This function is called when a prop named "NOTE" must be
     freed, but without freeing the data that were allocated in other
     places, and that are pointed to by the prop->data member.
  */
  g_assert (prop != NULL);
  
  if (prop->name != NULL)
    g_free (prop->name);
  
  g_free (prop);
    
  return 1;
}





