/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "pxmchem-mnmcompcalc.h"
#include "pxmchem-monomer.h"


PxmCompcalcRes
pxmchem_mnmcompcalc_polymer (PxmPolymer *polymer, GPtrArray *GPA,
			     PxmPolchemdef *polchemdef,
			     gint start_idx, gint end_idx)
{
  PxmCompcalcRes compcalcres = PXM_COMPCALC_SUCCESS;

  gint len = 0;
  gint iter = 0;
  gint jter = 0;

  gboolean processed = FALSE;
  
  PxmMonomer *monomer = NULL;
  PxmMonomer *monomer_fake = NULL;

  PxmProp *prop = NULL;
  

  /* We get a pointer to a polymer of which a composition (of
     monomers) is to be calculated either totally or only partially
     (depending on the values of start_idx and end_idx).
     
     We will iterate in the polymer->monomerGPA monomer array and for 
     each monomer we'll check if we already have encountered it or not. There
     will be two cases:

     1. If the monomer is encountered for the first time, a "fake"
     monomer object is constructed with the name of the iterated
     one. A prop named "COUNT" is created and the value of this prop
     is an integer which is set to 1. The "COUNT"-named prop object is
     set to the "fake" monomer, which is itself added to 'GPA'.
     
     2. If the monomer was already encoutered, the corresponding
     "fake" monomer is looked up in 'GPA' and when found the data in
     its "COUNT"-named prop object is incremented by 1.
  */

  g_assert (GPA != NULL);
  g_assert (polchemdef != NULL);

  g_assert (polymer != NULL);
  g_assert (polymer->monomerGPA != NULL);
  len = polymer->monomerGPA->len;


  if (len <= 0)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: the polymer sequence is empty.\n"),
	     __FILE__, __LINE__);
      
      return TRUE;
    }
  
  /* This function may be called to calculate the monomeric composition 
     of not a whole polymer sequence, but of an oligomer.
  */
  if (start_idx > len)
    start_idx = len;
  
  /* Special value: if -1, then that means the whole sequence should be
     taken into account. Thus set start_idx to 0.
  */
  if (start_idx == -1)
    start_idx = 0;

  /* And now make sure that start_idx does not have silly values apart
     from this special-meaning -1 value.
   */
  g_assert (start_idx >= 0);
  

  /* Right end, second.
   */
  g_assert (end_idx <= len);

  /* Special value: if -1, then that means the whole sequence should
     be taken into account. Thus set end_idx to len (because the for
     loop later will squeeze stop at iter == (len - 1).
  */
  if (end_idx == -1)
    end_idx = len;

  /* And now make sure that end_idx does not have silly values apart
     from this special-meaning -1 value.
   */
  g_assert (end_idx >= 0);
  
  for (iter = start_idx; iter < end_idx; iter++)
    {
      /* (Re)-initialize the processed variable, so that we'll later
	 effectively know if the currently iterated monomer was 
	 processed indeed or not.
      */
      processed = FALSE;

      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      /* We now should check if this monomer was found already.
       */
      for (jter = 0; jter < GPA->len; jter++)
	{
	  monomer_fake = g_ptr_array_index (GPA, jter);
	  g_assert (monomer_fake != NULL);
	  
	  if (0 == strcmp (monomer->name, monomer_fake->name))
	    {
	      /* The monomer we are iterating in the polymer sequence 
		 had already been encountered at least once. Increment
		 the "COUNT" property's data by one.
	      */
	      prop = libpolyxmass_prop_find_prop (monomer_fake->propGPA,
					      NULL,
					      NULL,
					      "COUNT",
					      NULL,
					      PXM_UNDEF_PROP_CMP);
	      /* It is not possible that a "COUNT"-name prop not be 
		 found, otherwise the function we are in is buggy.
	      */
	      g_assert (prop != NULL);
	      
	      /* Increment by one the value of the prop name "COUNT".
	       */
	      (*((gint *) prop->data))++;

	      /* Attention: If the monomer that we are iterating is 
		 modified, we want to acknowledge this by simply duplicating
		 the prop object and preprend it to the propGPA of the "fake"
		 monomer.
	      */
	      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
					      NULL,
					      NULL,
					      "MODIF",
					      NULL,
					      PXM_UNDEF_PROP_CMP);

	      if (prop != NULL)
		{
		  /* The monomer that we are iterating through is modified,
		     so we duplicate the modif prop.
		  */
		  prop = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		  
		  g_ptr_array_add (monomer_fake->propGPA, prop);
		}
	      
	      /* We should break now, but let the outer loop know that the
		 monomer was successfully processed.
	      */
	      processed = TRUE;
	      
	      break;
	    }
	}
      /* end of 
      for (jter 0; jter < GPA->len; jter++)
      */

      /* At this point we are here either because the monomer that is
	 being iterated in the polymer sequence was not found in the
	 array of "fake" composition monomers or because it was indeed
	 found.
      */

      /*
	 If it was indeed found, and thus was successfully processed,
	 all we do is continue iterating in the polymer sequence array
	 of monomers.
      */
      if (processed == TRUE)
	continue; 

      /*
	 If it was not found, then that means that it is the first
	 time that this monomer is encountered in the polymer
	 sequence, and thus a "fake" composition monomer has to be
	 created and appended to the array of "fake" monomers (GPA).
      */
      monomer_fake = pxmchem_monomer_new_by_code (monomer->code,
						  polchemdef->monomerGPA);
      if (monomer_fake == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to create new monomer: '%s'\n"),
		 __FILE__, __LINE__, monomer->name);
	  
	  return PXM_COMPCALC_FAILURE;
	}
      
      /* We have to construct the prop, named "COUNT" in which we'll
	 store the count for the monomers by the monomer->name that
	 are encountered in the polymer sequence.
      */
      prop = libpolyxmass_prop_new ();
      libpolyxmass_prop_set_name (prop, "COUNT");
      
      prop->data = g_malloc0 (sizeof (gint));
      
      (*((gint *) prop->data)) = 1;

      /* The default freeing function is OK because an allocated gint 
	 is freed the same as an allocated string.
      */

      /* Set this prop to the propGPA of the "fake" monomer.
       */
      g_ptr_array_add (monomer_fake->propGPA, prop);
      
      /* Attention: If the monomer that we are iterating is 
	 modified, we want to acknowledge this by simply duplicating
	 the prop object and preprend it to the propGPA of the "fake"
	 monomer.
      */
      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				      NULL,
				      NULL,
				      "MODIF",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
      
      if (prop != NULL)
	{
	  /* The monomer that we are iterating through is modified,
	     so we duplicate the modif prop.
	  */
	  prop = 
	    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
	  
	  g_ptr_array_add (monomer_fake->propGPA, prop);
	}
      
      /* Finally append this monomer object to the GPtrArray of "fake"
	 composition monomers passed as parameter.
      */
      g_ptr_array_add (GPA, monomer_fake);
    }
  /* end of
  for (iter = start_idx; iter < end_idx; iter++)
  */

  /* We have finished iterating the polymer sequence according to the
     start_idx/end_idx specifications passed as parameter.
  */     

  return compcalcres; 
}

  
PxmCompcalcRes
pxmchem_mnmcompcalc_oligomer (PxmOligomer *oligomer, GPtrArray *GPA,
			      PxmPolchemdef *polchemdef)
{
  g_assert (GPA != NULL);
  g_assert (polchemdef != NULL);
  
  g_assert (oligomer != NULL);
  g_assert (oligomer->polymer != NULL);
  
  return pxmchem_mnmcompcalc_polymer (oligomer->polymer, GPA,
				      polchemdef,
				      oligomer->start_idx, oligomer->end_idx);
}

  


