/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifndef PXMCHEM_POLYMER_H
#define PXMCHEM_POLYMER_H

#include "libpolyxmass-globals.h"
#include "libpolyxmass-masspair.h"
#include "pxmchem-modif.h"
#include "libpolyxmass-reportopt.h"


#define POLYMER_SEQUENCE_XML_VERSION  "0.1"

typedef struct _PxmPlminfo PxmPlminfo;
typedef struct _PxmPolymer PxmPolymer;


/* The PxmPlminfo struct contains informations about a given
 * polymer instance, in particular elements of its 
 * identity, like its filename, its date of creation, 
 * of modification, its author...
 * 
 * This struct does not contain any information about the 
 * polymer sequence itself nor about chemical composition or
 * about masses... 
 *
 * This struct is included in the polymer struct to help fully 
 * characterize a polymer instance.
 */
struct _PxmPlminfo
{
  /* The type of the polymer is the name by which 
   * a polymer knows of what type it is. For example, a polymer of 
   * type "protein" is a different polymer than one of type "dna".
   */
  gchar *type;

  /* The name of a polymer is the name by which the
   * user will want to identify his polymer sequence. For example,
   * this could be "apomyoglobin" or "PhiX DNA" or "FIX-glycochain".
   */
  gchar *name;

  /* The code of a given polymer instance is a code that
   * the user might want to use to identify a polymer sequence. For
   * example this code could be for a protein the SwissProt code, or
   * for a gene, the GeneBank code.
   */
  gchar *code;

  /* Username of user having last modified (or having created) the 
   * polymer sequence.
   */
  gchar *author;

  gchar *date_year;
  gchar *date_month;
  gchar *date_day;
  
  /* The name of the polymer sequence file as it was opened by
   * the program.
   */
  gchar *file;
};


/* The PxmPolymer struct is the object that contains the polymer 
 * sequence and related data. 
 * 
 * Into this struct are a number of data including 
 * GPtrArray's, a string, a PxmPlminfo struct...
 */
struct _PxmPolymer
{
  /* version identifies the version of the polymer sequence file
     format. This string, which can be transformed to a float for
     comparisons, is used to ensure that the file format of a polymer
     sequence is readable by the library. When a polymer sequence file
     is saved, it will be saved with version string defined in #define
     POLYMER_SEQUENCE_XML_VERSION.
   */
  gchar *version;
  
  /* Al the info data pertaining to the polymer object 
   * and polymer file loaded by the user are stored in this PxmPlminfo
   * struct member.
   */
  PxmPlminfo *plminfo;

  /* The masses of the polymer, with no specific way to know how
     these were calculated. So use prop objects to add informations if
     necessary.
  */
  PxmMasspair *masspair_seq; /* whole sequence */
  PxmMasspair *masspair_sel; /* selection only */
  
  /* A boolean value that tells if the polymer sequence has been
   * modified or not. This is a convenience for the user of the 
   * PxmPolymer object.
   */
  gboolean modified;
  
  /* This GPtrArray contains the polymer sequence under the form 
   * of monomer instances, which -in turn- are amenable to 
   * further characterization by the prop object mechanism.
   * The GPtrArray of PxmMonomer instances is thus the best way to 
   * characterize the fine structure of a polymer, much more fine
   * grained than the sequence simple string made of monomer codes.
   */
  GPtrArray *monomerGPA;

  /* A polymer sequence might well be modified in some manner.
   * For example, it might be modified on its left cap. This can
   * be modelled by using the prop object mechanism, which is why 
   * the polymer struct has a GPtrArray of prop instances. As for
   * other structures (monomer, modif...), the number of prop instances
   * that can be added to this array is unlimited.
   */
  GPtrArray *propGPA;
};


/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS, INITING FUNCTIONS ...
 */
PxmPolymer *
pxmchem_polymer_new (void);


PxmPlminfo *
pxmchem_plminfo_new (void);

gboolean
pxmchem_polymer_set_version (PxmPolymer *polymer, gchar *version);

gboolean
pxmchem_polymer_set_modified (PxmPolymer *polymer, gboolean modified);

gboolean
pxmchem_polymer_invert_modified (PxmPolymer *polymer);

gboolean
pxmchem_polymer_get_modified (PxmPolymer *polymer);


gboolean
pxmchem_polymer_plminfo_set_type (PxmPlminfo *plminfo, gchar *type);

gboolean
pxmchem_polymer_plminfo_set_name (PxmPlminfo *plminfo, gchar *name);

gboolean
pxmchem_polymer_plminfo_set_code (PxmPlminfo *plminfo, gchar *code);

gboolean
pxmchem_polymer_plminfo_set_author (PxmPlminfo *plminfo, gchar *author);

gboolean
pxmchem_polymer_plminfo_set_date_year (PxmPlminfo *plminfo, gchar *year);

gboolean
pxmchem_polymer_plminfo_set_date_month (PxmPlminfo *plminfo, gchar *month);

gboolean
pxmchem_polymer_plminfo_set_date_day (PxmPlminfo *plminfo, gchar *day);

gboolean
pxmchem_polymer_plminfo_set_file (PxmPlminfo *plminfo, gchar *file);


/* INTEGRITY CHECKING FUNCTIONS
 */



/*  LOCATING FUNCTIONS
 */
gchar *
pxmchem_polymer_get_left_end_modif_name (PxmPolymer *polymer);

gchar *
pxmchem_polymer_get_right_end_modif_name (PxmPolymer *polymer);

gint
pxmchem_polymer_find_sequence_motif (PxmPolymer *polymer, 
				     gchar *motif,
				     GPtrArray *motifGPA,
				     gint how_cmp,
				     gint codelen, 
				     GPtrArray *refGPA, 
				     GPtrArray *propGPA,
				     GPtrArray *fillGPA);

gint
pxmchem_polymer_find_prop_in_sequence (PxmPolymer *polymer, 
				       gint how_cmp,
				       PxmProp *prop,
				       GPtrArray *fillGPA);

/* UTILITY FUNCTIONS
 */
gchar *
pxmchem_polymer_get_current_xml_version (void);


/* MODIFYING FUNCTIONS
 */
PxmEnd
pxmchem_polymer_modify (PxmPolymer *polymer, gchar *modif, PxmEnd end);

gboolean
pxmchem_polymer_LE_modify (PxmPolymer *polymer, gchar *modif);

gboolean
pxmchem_polymer_RE_modify (PxmPolymer *polymer, gchar *modif);

PxmEnd
pxmchem_polymer_un_modify (PxmPolymer *polymer, gchar *modif, PxmEnd end);

gint
pxmchem_polymer_LE_un_modify (PxmPolymer *polymer, gchar *modif);

gint
pxmchem_polymer_RE_un_modify (PxmPolymer *polymer, gchar *modif);





/* UTILITY FUNCTIONS
 */
gint
pxmchem_polymer_composition (PxmPolymer *polymer,
			     GPtrArray *mnm_refGPA,
			     GPtrArray *GPA,
			     gint start_idx, gint end_idx);


gchar *
pxmchem_polymer_make_codes_string (PxmPolymer *polymer,
				   gint start_idx, gint end_idx);


gint
pxmchem_polymer_calculate_length_from_string (gchar* seq,
					    gboolean check_codes,
					    gint codelen,
					    GPtrArray *mnm_refGPA);

gchar *
pxmchem_polymer_get_current_xml_version (void);



/* Text-format TRANSACTIONS
 */

/***** XML-format *****/
gchar *
pxmchem_polymer_format_xml_string_DTD (void);


gchar *
pxmchem_polymer_format_xml_string_polseqdata (PxmPolymer *polymer, 
					   gchar *indent, gint offset);

gchar *
pxmchem_polymer_format_xml_string_polseqinfo (PxmPlminfo *plminfo, 
					      gchar *indent, gint offset);

gchar *
pxmchem_polymer_format_xml_string_polseq_with_mnm_GPA (GPtrArray *GPA, 
						       gchar *indent, 
						       gint offset);
gint
pxmchem_polymer_write_xml_file (PxmPolymer *polymer, gchar *file,
				gpointer user_data);

/* Fake sequence: plminfo only, empty sequence.
 */
gint
pxmchem_plminfo_write_xml_file (PxmPlminfo *plminfo, gchar *file, 
				gpointer user_data);




gchar *
pxmchem_polymer_get_type_from_xml_file (gchar *file);


PxmPolymer *
pxmchem_polymer_render_xml_file (gchar *file,
				 GPtrArray *mnm_refGPA,
				 gint codelen,
				 gpointer user_data);

PxmPlminfo *
pxmchem_polymer_render_xml_node_polseqinfo (xmlDocPtr xml_doc,
					xmlNodePtr xml_node,
					gpointer user_data);

PxmPolymer *
pxmchem_polymer_render_xml_node_polseqdata (xmlDocPtr xml_doc,
					    xmlNodePtr xml_node,
					    GPtrArray *mnm_refGPA,
					    gint codelen,
					    gpointer user_data);

gint
pxmchem_polymer_render_xml_node_polseq (xmlDocPtr xml_doc,
					xmlNodePtr xml_node,
					GPtrArray *fillGPA,
					GPtrArray *mnm_refGPA,
					gint codelen,
					gpointer user_data);

gchar *
pxmchem_polymer_get_xml_file_version (gchar *file);



/***** TEXT-format *****/
GString *
pxmchem_polymer_format_txt_string_polseqinfo (PxmPlminfo *plminfo,
					      PxmReportOpt *reportopt,
					      GString *gs);

GString *
pxmchem_polymer_format_txt_string_polseq_with_mnm_GPA (GPtrArray *GPA,
						       PxmReportOpt *reportopt,
						       GString *gs);

GString *
pxmchem_polymer_format_txt_string_polymer_prop (PxmPolymer *polymer,
						PxmReportOpt *reportopt,
						GString *gs);








/* FREE'ING FUNCTIONS
 */
gboolean
pxmchem_polymer_free (PxmPolymer *polymer);


gboolean
pxmchem_plminfo_free (PxmPlminfo *plminfo);




/* GPtrArray-RELATED FUNCTIONS
 */
gint
pxmchem_polymer_GPA_free (GPtrArray *GPA);



#endif
