/*****************************************************************

  codec.c

  Copyright (c) 2002 by Burkhard Plaum - plaum@ipf.uni-stuttgart.de

  http://libquicktime.sourceforge.net

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

*****************************************************************/

#include <quicktime/colormodels.h>
#include <quicktime.h>
#include <funcprotos.h>
#include <string.h>
#include "svq1.h"

typedef struct
  {
  svq1_t svq;
  char * buffer;
  int buffer_size;
  
  unsigned char * temp_frame;
  unsigned char * temp_rows[3];

  int initialized;
  
  } quicktime_svq_codec_t;

static int delete_codec(quicktime_video_map_t *vtrack)
  {
  quicktime_svq_codec_t *codec = ((quicktime_codec_t*)vtrack->codec)->priv;
  
  if(codec->buffer)
    free(codec->buffer);

  if(codec->temp_frame)
    free(codec->temp_frame);

  if(codec->svq.current)
    free(codec->svq.current);
  
  if(codec->svq.previous)
    free(codec->svq.previous);
  
  if(codec->svq.motion)
    free(codec->svq.motion);
  
  free(codec);
  return 0;
  }

static int decode(quicktime_t *file, unsigned char **row_pointers, int track)
  {
  int result = 0;
  int64_t i;
  quicktime_video_map_t *vtrack = &(file->vtracks[track]);
  quicktime_trak_t *trak = vtrack->track;
  int height = trak->tkhd.track_height;
  int width = trak->tkhd.track_width;
  quicktime_svq_codec_t *codec =
    ((quicktime_codec_t*)vtrack->codec)->priv;

  int frame_size;
  
  int use_temp = (BC_YUV420P != file->color_model ||
                  file->in_x != 0 ||
                  file->in_y != 0 ||
                  file->in_w != width ||
                  file->in_h != height ||
                  file->out_w != width ||
                  file->out_h != height);

  if(!codec->initialized)
    {
    codec->initialized = 1;
    }
  
  if(use_temp && !codec->temp_frame)
    {
    codec->temp_frame = malloc(width*height+(width*height)/2);
    codec->temp_rows[0] = codec->temp_frame;
    codec->temp_rows[1] = codec->temp_frame + width*height;
    codec->temp_rows[2] = codec->temp_frame + width*height + (width*height)/4;
    }
  
  quicktime_set_video_position(file, vtrack->current_position, track);
  frame_size = quicktime_frame_size(file, vtrack->current_position, track);

  if(codec->buffer_size < frame_size)
    {
    codec->buffer_size = frame_size;
    codec->buffer = realloc(codec->buffer, frame_size);
    }
  
  result = !quicktime_read_data(file, codec->buffer, codec->buffer_size);
  
  if(!result)
    {
    result = svq1_decode_frame(&(codec->svq), codec->buffer);
    }
  
  if(use_temp)
    {
    svq1_copy_frame(&(codec->svq), codec->temp_rows, width);
    
    cmodel_transfer(row_pointers, 
                    codec->temp_rows,
                    row_pointers[0],
                    row_pointers[1],
                    row_pointers[2],
                    codec->temp_rows[0],
                    codec->temp_rows[1],
                    codec->temp_rows[2],
                    file->in_x, 
                    file->in_y, 
                    file->in_w, 
                    file->in_h,
                    0, 
                    0, 
                    file->out_w, 
                    file->out_h,
                    BC_YUV420P,
                    file->color_model,
                    0,
                    width,
                    file->out_w);
    }
  else
    {
    svq1_copy_frame(&(codec->svq), row_pointers, width);
    }
  return result;
  }

static int reads_colormodel(quicktime_t *file, 
                            int colormodel, 
                            int track)
  {
  return (colormodel == BC_YUV420P);
  }

void quicktime_init_codec_svq1(quicktime_video_map_t *vtrack)
  {
  quicktime_svq_codec_t *codec;
  
  /* Init public items */
  ((quicktime_codec_t*)vtrack->codec)->priv = calloc(1, sizeof(quicktime_svq_codec_t));
  ((quicktime_codec_t*)vtrack->codec)->delete_vcodec = delete_codec;
  ((quicktime_codec_t*)vtrack->codec)->decode_video = decode;
    
  ((quicktime_codec_t*)vtrack->codec)->reads_colormodel = reads_colormodel;
  }
