////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////
#ifndef S11N_NS_LIST_H_INCLUDED
#define S11N_NS_LIST_H_INCLUDED 1

#include <list>
#include <iterator> // insert_iterator
#include <algorithm> // for_each()
#include <stdio.h> // snprintf()

#include <S11N_NS/functor.h> // copy_if, same_name

#include <S11N_NS/data_node_serialize.h> // core serialize funcs
#include <S11N_NS/data_node_functor.h> // data_node_child_serializer functor
#include <S11N_NS/data_node_algo.h> // dump_node_debug()
#include <S11N_NS/abstract_creator.h> // abstract_creator class

namespace S11N_NS {
/**
   The S11N_NS::list namespace defines functors and algorithms for
   working with std::list/vector-style containers.
*/
namespace list {


        /**
           Deprecated. See serialize_list().

           Serializes all Serializable objects in the given list.

           ListType must contain ([const] SomeSerializableType *).
        
           True is returned on success.

           False is returned if any child Serializable in src fails to
           serialize.  If false is returned then dest is in an
           undefined state: it may have been partially serialized, and
           certainly wasn't *completely* serialized.

           Note that a subnodename is required because it is
           patentedly a Bad Idea to store more than one Serializable
           in the same node, so this function creates subnodes in
           dest.
        */
        template <typename NodeT, typename ListType>
        bool serialize_pointer_list( NodeT & dest,
                             const std::string & subnodename,
                             const ListType & src )
        {
                return std::for_each( src.begin(),
                                      src.end(),
                                      S11N_NS::data_node_child_serializer<NodeT>( dest, subnodename )
                                      ).result;
        }


        /**
           Deprecated. See deserialize_list().

           The complement to serialize_pointer_list().
        
           Looks for child nodes of src named subnodename. For each
           one that it finds it tries to deserialize it to a
           (ListType::value_type) and insert it into dest.

           Returns false only if children are found but their
           deserialization fails. If that happens, dest is in an
           undefined state: any other children (not the one which
           failed deserialization) might still be in the list, and
           there is know way for the client to know which child failed
           or why it failed.
        */
        template <typename NodeT, typename ListType>
        bool deserialize_pointer_list( const NodeT & src,
                             const std::string & subnodename,
                             ListType & dest )
        {
                //CERR << "deserialize_pointer_list("<<src.name()<<" , "<<subnodename<<", destlist )\n";
                typedef std::list<const NodeT *> MatchList;
                MatchList li;
                S11N_NS::copy_if( src.children().begin(),
                                  src.children().end(),
                                  std::insert_iterator<MatchList>( li, li.begin() ),
                                  S11N_NS::same_name<NodeT>( subnodename )
                                  );
                //CERR << "found " << li.size() << " match(es)\n";
                if( ! li.size() ) return false;
                bool b = std::for_each( li.begin(),
                                        li.end(),
                                        S11N_NS::data_node_child_deserializer<ListType>( dest )
                                      ).result;
                //CERR << "for_each() returned " << b << ", and result list size=" << dest.size() << "\n";
                return b;
        }

        /**
           serialize_list() supports list/vector-like types containing
           any Serializable type. Serializes (src.begin(),src.end()]
           to dest. Each item has a node name of that given to the
           ctor or an unspecified dummy value.
           
           Returns true on success. If false is returned then dest is
           in an undefined state: some number of serializations may
           have succeeded before the failure. This operation stops
           processing at the first serialization error.


           Compatible ListTypes must support:
           
           - a value_type typedef describing the type of it's contents.

           - push_back( value_type )

           - const_iterator typedefs.

           Restrictions on value_type's type:

           - Must be a Serializable. This includes any i/o streamable
           type which has a proxy installed by s11n (includes all PODs
           and std::string by default).

           - may be a pointer type.

           Some s11n-side requirements:

           - ListType must be registered with the ListType classloader.

           ACHTUNG: never pass the same destination container to 
           this function more than once or you will get duplicate and/or
           incorrect data.
        */
        template <typename NodeType, typename SerType>
        bool serialize_list( NodeType & dest, const SerType & src )
        {
                dest.impl_class( class_name<SerType>::name() );
                typename SerType::const_iterator it = src.begin();
                // reminder: (*it) may be a pointer- or a value-type,
                // as long as it's a Serializable.
                //dest.set( "s11n_metadata", "serialized-by:S11N_NS::list::serialize_list" );
                for( ; src.end() != it; ++it )
                {
                        NodeType * ch = new NodeType;
                        if( ! serialize( *ch, *it ) )
                        {
                                CERR << "serialize_list: a child failed to serialize: " << ch->name() << " @ " << std::hex << ch << "\n";
                                S11N_NS::dump_node_debug( *ch, std::cerr );
                                delete( ch );
                                return false;
                        }
                        dest.children().push_back( ch );
                }
                return true;
        }

        /**
           Identical to the two-argument form of serialize_list(), but
           serializes src into a subnode of dest, named subnodename.
        */
        template <typename NodeType, typename SerType>
        bool serialize_list( NodeType & dest,
                             const std::string & subnodename,
                             const SerType & src )
        {
                NodeType * ch = new NodeType;
                if( ! serialize_list<NodeType,SerType>( *ch, src ) )
                {
                        delete( ch );
                        return false;
                }
                dest.children().push_back( ch );
                return true;
        }

        /**
           For each (src.children().begin(),end()] an object
           of type SerType::value_type is created, deserialized,
           and is added to dest via push_back( item ).

           See serialize_list() for the list type requirements.

           If SerType::value_type is a pointer type then dest owns any
           newly-created pointers, and it's owner (or the container
           itself) is responsible for cleaning them up. (Non-pointer
           types need no explicit clean-up, of course.)

           Returns true if all deserializations succeed.  Stops
           processing and returns false at the first error, in which
           case dest is in an undefined state: some children may or
           may not have been successfully deserialized into it. The
           immediate implication of this is that it will fail if src
           contains any other children than the type which dest
           expects!
        */
        template <typename NodeType, typename SerType>
        bool deserialize_list( const NodeType & src, SerType & dest )
        {
                // note: some of the odd code here is to allow this
                // func to support both value- and pointer-types.
                typedef typename SerType::value_type VT;
                typedef S11N_NS::abstract_creator<VT> ACVT;
                typedef typename NodeType::child_list_type::const_iterator CHIT;
                const NodeType * nch = 0;
                CHIT it = src.children().begin();
                CHIT et = src.children().end();
                S11N_NS::object_reference_wrapper<SerType> dwrap(dest);
                VT ser; // reminder: might be a pointer type
                for( ; et != it; ++it )
                {
                        nch = *it;
                        if( ! ACVT::create( ser,nch->impl_class() ) )
                        {
                                CERR << "Internal error: abstract_creator<> "
                                     << "could not create a new object of type '"
                                     << nch->impl_class()<<"'!\n";
                                return false;
                        }
                        if( ! deserialize( *nch, ser ) )
                        {
                                CERR << "deserialize_list(): deser of a child failed!\n";
                                CERR << "name="<< nch->name()<< ". implclass="<< nch->impl_class()<<" @ " << std::hex<<nch <<"\n";
                                S11N_NS::dump_node_debug( *nch, std::cerr );
                                ACVT::release( ser );
                                return false;
                        }
                        dwrap().insert( dwrap().end(), ser );
                }
                return true;
        }

        /**
           Identical to the two-argument form of deserialize_list(), but
           deserializes a subnode of src, named subnodename. If no such
           child is found in src then false is returned.
        */
        template <typename NodeType, typename SerType>
        bool deserialize_list( const NodeType & src,
                               const std::string & subnodename,
                               SerType & dest )
        {
                const NodeType * ch = S11N_NS::find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize_list<NodeType,SerType>( *ch, dest );
        }        

        /**
           serialize_streamable_list serializes objects of type
           <tt>std::list&lt;X&gt;</tt> (and compatible list types,
           such as std::vector). It stores them in such a way that
           they can be loaded into any compatible container via
           deserialize_streamable_list().

           Conventions:

           - NodeType must support a generic set(Key,Val) function, as
           implemented by the S11N_NS::data_node interface.

           - ListType must conform to std::list conventions and it's
           value_type must be a non-pointer type which is
           i/ostreamable (this includes all PODs and
           std::string). Pointers are not supported by
           this function.

           ACHTUNG:

           - Never call this on a node for which you store other
           properties, as deserialize_streamable_list() will consume them
           if you use that function.

           - This function sets dummy property keys, both to please
           the conventions of keys having non-empty values and to keep
           the list in the proper order. It uses keys which should be
           portable to, e.g., standard XML.

           Returns the number of items serialized.
        */
        template <typename NodeType, typename ListType>
        size_t serialize_streamable_list( NodeType & dest, const ListType & src )
        {
                typedef typename ListType::const_iterator CIT;
                CIT it = src.begin();
                size_t i = 0;
                // We zero-pad all keys to be the same length, so that
                // they will be kept in the proper order in the
                // target's property list (annoying, but necessary).
                // We prepend a non-numeric char to make this output
                // compatible with standard XML
                static const int bsize = 10;
                char num[bsize];
                char fmt[bsize];
                size_t sz = src.size();
                int places = 1; // # of digits to use
                for( ; sz >= 0x0f; sz = (size_t)(sz/0x0f)) { ++places; }
                snprintf( fmt, bsize, "x%%0%dx", places );
                for( ; src.end() != it; ++it )
                {
                        snprintf( num, bsize, fmt, i );
                        ++i;
                        dest.set( num, (*it) );
                }
                return i;
        }


        /**
           Identical to serialize_streamable_list(dest,src), but
           creates a subnode in dest, named subnodename, where the
           data is stored.
        */
        template <typename NodeType, typename ListType>
        size_t serialize_streamable_list( NodeType & dest,
                                          const std::string & subnodename,
                                          const ListType & src )
        {
                NodeType & n = S11N_NS::create_child( dest, subnodename );
                return serialize_streamable_list<NodeType,ListType>( n, src );
        }


        /**
           Deserializes dest from src. It reads in all properties from
           src, ignoring their keys and taking only their values. This
           is suitable for use with the result of a
           serialize_streamable_list() call. See that function for more
           information, including the conventions which must be
           supported by NodeType and ListType.
           
           Returns the number of items deserialized.
        */
        template <typename NodeType, typename ListType>
        size_t deserialize_streamable_list( const NodeType & src, ListType & dest )
        {
                typedef typename ListType::value_type VT;
                typename NodeType::const_iterator it = src.begin();
                size_t i = 0;
                VT defaultval;
                for( ; src.end() != it; ++it )
                {
                        ++i;
                        dest.insert( dest.end(),
                                     S11N_NS::from_string<VT>( (*it).second, defaultval ) );
                }
                return i;
        }


        /**
           Identical to deserialize_streamable_list(), but looks for
           the data in a subnode of src named subnodename.

           Returns 0 if no child could be found.
        */
        template <typename NodeType, typename ListType>
        size_t deserialize_streamable_list( const NodeType & src,
                                     const std::string & subnodename,
                                     ListType & dest )
        {
                const NodeType * ch = S11N_NS::find_child_by_name( src, subnodename );
                if( ! ch ) return 0;
                return deserialize_streamable_list<NodeType,ListType>( *ch, dest );
        }



        /**
           list_serializer_proxy is a functor for de/serializing lists
           of Serializables.
        */
        class list_serializer_proxy
        {
        public:
                list_serializer_proxy()
                {}

                /**
                   see serialize_list().

                */
                template <typename NodeType, typename SerType>
                bool operator()( NodeType & dest, const SerType & src ) const
                {
                        return serialize_list( dest, src );
                }

                /** see deserialize_list(). */
                template <typename NodeType, typename SerType>
                bool operator()( const NodeType & src, SerType & dest ) const
                {
                        return deserialize_list( src, dest );
                }
        };



} } // namespace S11N_NS::list


#endif // S11N_NS_LIST_H_INCLUDED
