/* Author: Joshua Brindle <jbrindle@tresys.com>
 *         Jason Tang     <jtang@tresys.com>
 *         Ivan Gyurdiev  <ivg2@cornell.edu>
 *           
 * Copyright (C) 2005 Tresys Technology, LLC
 * Copyright (C) 2005 Red Hat Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _SEMANAGE_INTERNAL_HANDLE_H_
#define _SEMANAGE_INTERNAL_HANDLE_H_

#include <stddef.h>
#include "handle_internal.h"
#include <sepol/handle.h>
#include "modules.h"
#include "semanage_conf.h"
#include "database.h"
#include "direct_api.h"
#include "policy.h"

struct semanage_handle {
	int con_id;             /* Connection ID */
	int policy_serial;      /* Policy serial number at connect time */

	/* Error handling */
	int msg_level;
	const char* msg_channel;
	const char* msg_fname;
#ifdef __GNUC__
	__attribute__ ((format (printf, 3, 4)))
#endif
	void (*msg_callback) (
		void* varg,
		semanage_handle_t* handle,
		const char* fmt,
		...);
	void* msg_callback_arg;

	/* Direct vs Server specific handle */
	union {
		struct semanage_direct_handle direct;
	} u;

	/* Libsepol handle */
	sepol_handle_t* sepolh;

	semanage_conf_t *conf;
	int is_connected;
	int is_in_transaction;
	int do_reload;		/* whether to reload policy after commit */
	int modules_modified;

	/* This timeout is used for transactions and waiting for lock
	   -1 means wait indefinetely
	    0 means return immediately
	    >0 means wait that many seconds */
	int timeout;

	/* these function pointers will point to the appropriate
	 * routine given the connection type.  think of these as
	 * simulating polymorphism for non-OO languages. */
	struct semanage_policy_table* funcs;

	/* Object databases */
#define DBASE_COUNT      9

#define DBASE_LOCAL_USERS       0
#define DBASE_LOCAL_PORTS       1
#define DBASE_LOCAL_INTERFACES  2
#define DBASE_LOCAL_BOOLEANS    3
#define DBASE_SEUSERS           4

#define DBASE_POLICY_USERS      5
#define DBASE_POLICY_PORTS      6
#define DBASE_POLICY_INTERFACES 7
#define DBASE_POLICY_BOOLEANS   8
	dbase_config_t dbase[DBASE_COUNT];
};

static inline
dbase_config_t* semanage_user_dbase_local(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_LOCAL_USERS];
}

static inline 
dbase_config_t* semanage_port_dbase_local(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_LOCAL_PORTS];
}

static inline
dbase_config_t* semanage_iface_dbase_local(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_LOCAL_INTERFACES];
}

static inline
dbase_config_t* semanage_bool_dbase_local(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_LOCAL_BOOLEANS];
}

static inline
dbase_config_t* semanage_seuser_dbase(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_SEUSERS];
}

static inline
dbase_config_t* semanage_user_dbase_policy(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_POLICY_USERS];
}

static inline
dbase_config_t* semanage_port_dbase_policy(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_POLICY_PORTS];
}

static inline
dbase_config_t* semanage_iface_dbase_policy(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_POLICY_INTERFACES];
}

static inline
dbase_config_t* semanage_bool_dbase_policy(semanage_handle_t* handle) {
	return &handle->dbase[DBASE_POLICY_BOOLEANS];
}

#endif

