/* Copyright (C) 2005 Red Hat, Inc. */

#include <stdlib.h>
#include "policy.h"
#include "handle.h"
#include "database.h"
#include "modules.h"
#include "seusers.h"
#include "debug.h"

#define MODE_SET    1
#define MODE_MODIFY 2

static int clear_obsolete(
	semanage_handle_t* handle,
	dbase_config_t* src,
	dbase_config_t* dst) {

	record_key_t* key = NULL;
	record_t** records = NULL;
	size_t nrecords = 0;
	size_t i;

	dbase_table_t* src_dtable = src->dtable;
	dbase_table_t* dst_dtable = dst->dtable;
	record_table_t* rtable = src_dtable->get_rtable(src->dbase);	

	if (src_dtable->list(handle, src->dbase, &records, &nrecords) < 0)
		goto err;

	for (i = 0; i < nrecords; i++) {
		int exists;

		if (rtable->key_extract(handle, records[i], &key) < 0)
			goto err;

		if (dst_dtable->exists(handle, dst->dbase, key, &exists) < 0)
			goto err;

		if (!exists) {
			if (src_dtable->del(handle, src->dbase, key) < 0)
				goto err;

			/* FIXME: notice to user */	
                        /* INFO(handle, "boolean %s is obsolete, unsetting configured value..."); */
		}
	}

	for (i=0; i < nrecords; i++)	
		rtable->free(records[i]);
	free(records);
	rtable->key_free(key);
	return STATUS_SUCCESS;

	err:
	/* FIXME: handle error */
	for (i=0; i < nrecords; i++)
		rtable->free(records[i]);
	free(records);
	rtable->key_free(key);
	return STATUS_ERR;
}

typedef struct load_handler_arg {
	semanage_handle_t* handle;
	dbase_config_t* dconfig;
	int mode; 
} load_handler_arg_t;

static int load_handler(
	record_t* record,
	void* varg) {

	record_key_t* rkey = NULL;
	load_handler_arg_t* arg = 
		(load_handler_arg_t*) varg;

	semanage_handle_t* handle = arg->handle;
	dbase_t* dbase = arg->dconfig->dbase;
	dbase_table_t* dtable = arg->dconfig->dtable;
	record_table_t* rtable = dtable->get_rtable(dbase);	

	if (rtable->key_extract(handle, record, &rkey) < 0)
		goto err;
 
	switch (arg->mode) {
	
		case MODE_SET:
			if (dtable->set(handle, dbase, rkey, record) < 0) 
				goto err;
			break;
		
		default:
		case MODE_MODIFY:
			if (dtable->modify(handle, dbase, rkey, record) < 0) 
				goto err;
			break;

	}

	rtable->key_free(rkey);
	return 0;

	err:
	/* FIXME: handle error */
	rtable->key_free(rkey);
	return -1;
}


typedef struct load_table {
	dbase_config_t* from;
	dbase_config_t* to;
	int mode;
} load_table_t;

/* This function must be called AFTER all modules are loaded.
 * Modules could be represented as a database, in which case
 * they should be loaded at the beginning of this function */

int semanage_base_merge_components(
	semanage_handle_t* handle) {

	int i;
	load_table_t components[] = {

		{ semanage_user_dbase_local(handle),
		  semanage_user_dbase_policy(handle), MODE_MODIFY },
#if 0
		{ semanage_port_dbase_local(handle),
		  semanage_port_dbase_policy(handle), MODE_MODIFY },
#endif
		{ semanage_iface_dbase_local(handle),
		  semanage_iface_dbase_policy(handle), MODE_MODIFY },
	
		{ semanage_bool_dbase_local(handle),
		  semanage_bool_dbase_policy(handle), MODE_SET },
	};
	const int CCOUNT = sizeof(components)/sizeof(components[0]);

	load_handler_arg_t load_arg;
	load_arg.handle = handle;

	/* Merge components into policy (and validate) */
	for (i = 0; i < CCOUNT; i++) {
		dbase_config_t* from = components[i].from;
		dbase_config_t* to = components[i].to;
		load_arg.dconfig = to;
		load_arg.mode   = components[i].mode;

		/* Must invoke cache function first */
		if (from->dtable->cache(handle, from->dbase) < 0) 
			goto err;

		if (to->dtable->cache(handle, to->dbase) < 0) 
			goto err;

		/* Clear obsolete items for MODE_SET */
		if (components[i].mode == MODE_SET)
			if (clear_obsolete(handle, from, to) < 0)
				goto err;
		
		/* Now iterate */
		if (from->dtable->iterate(
			handle, from->dbase, load_handler, &load_arg) < 0) 
			goto err;
	}	

	/* Validate seusers against policy */
	if (semanage_seuser_validate(handle) < 0)
		goto err;
	
	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not merge local modifications into policy");
	return STATUS_ERR;
}

int semanage_commit_components(
	semanage_handle_t* handle) {

	int i;
	dbase_config_t* components[] = {
		/* semanage_modules_dbase(handle), */
		semanage_iface_dbase_local(handle),
		semanage_bool_dbase_local(handle),
		semanage_user_dbase_local(handle),
		/* semanage_port_dbase_local(handle), */
		semanage_seuser_dbase(handle)
	};
	const int CCOUNT = sizeof(components)/sizeof(components[0]);

	for (i = 0; i < CCOUNT; i++) {
		/* Flush to disk */
		if (components[i]->dtable->flush(
			handle, components[i]->dbase) < 0)
			goto err;
	}

	/* Drop cache, because we're leaving transaction soon */
	for (i=0; i < CCOUNT; i++)
		components[i]->dtable->drop_cache(components[i]->dbase);


	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not commit local modifications");

	for (i=0; i < CCOUNT; i++) 
		components[i]->dtable->drop_cache(components[i]->dbase);
	return STATUS_ERR;
}
