# $DUH: DelayToRcpt.pm,v 1.1 2002/12/13 01:43:37 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::Callbacks::DelayToRcpt - collection of milters that does not error until RCPT TO:

=head1 SYNOPSIS

    use PMilter::Callbacks::DelayToRcpt;

    my $cb = PMilter::Callbacks::DelayToRcpt->new(..);

=head1 DESCRIPTION

Functions identically to PMilter::Callbacks, except that errors returned
during the "connect", "helo", and "envfrom" callbacks are delayed until the
"envrcpt" callback.

Many broken client mailers exist in the real world and will do such things
as instantaneously reconnect when receiving an error at the MAIL FROM:
stage.  This wrapper ensures that errors are never propagated back to the
MTA until at least the RCPT TO: phase.

=cut

package PMilter::Callbacks::DelayToRcpt;
use base PMilter::Callbacks;

use strict;
use warnings;

use PMilter::Callbacks qw(:all);

sub new {
	my $list = shift;

	unless (ref($list) eq 'ARRAY') {
		unshift(@_, $list);
		$list = [];
	}

	PMilter::Callbacks::new([ 'envrcpt', @$list ], @_);
}

sub call {
	my $this = shift;
	my $what = shift;
	my $ctx = shift;
	my $data = $ctx->getglobal($this, []);

	# [0] stores message-specific status, reset on envfrom.
	# [1] stores connection-global status.

	$data->[0] = $data->[1] if ($what eq 'envfrom'); # message context

	# SMFIS_CONTINUE is 0, so the conditional here works.
	my $rc = $data->[0] || $this->SUPER::call($what, $ctx, @_);

	if ($rc == SMFIS_REJECT || $rc == SMFIS_TEMPFAIL) {
		if ($what eq 'connect' || $what eq 'helo') {
			$data->[0] = $data->[1] = $rc;
			$rc = SMFIS_CONTINUE;
		} elsif ($what eq 'envfrom') {
			$data->[0] = $rc;
			$rc = SMFIS_CONTINUE;
		}
	}

	$rc;
}

1;
