# $DUH: Milter.pm,v 1.5 2002/12/10 19:42:44 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

Sendmail::Milter - Perl binding of Sendmail Milter protocol

=head1 SYNOPSIS

    use Sendmail::Milter;
    Sendmail::Milter::auto_setconn(NAME);
    Sendmail::Milter::register(NAME, { CALLBACKS }, FLAGS);
    Sendmail::Milter::main();

=head1 DESCRIPTION

All constants exported by L<PMilter::Callbacks> with export tag ':all' are
exported by default in Sendmail::Milter.

Most of PMilter's Sendmail::Milter interface is a clone of the frontend
functions in L<PMilter::Server>.  However, this compatibility package also
includes some methods specific to the Sendmail MTA, which are deliberately
not included in PMilter::Server.

=head1 METHODS

=over 4

=cut

package Sendmail::Milter;
use base Exporter;

use Carp;
use PMilter::Callbacks qw(:all);
use PMilter::Server;

# The specific version of the "real" Sendmail::Milter that we emulate.

our $VERSION = '0.18';

# Sendmail exports all these by default.

our @EXPORT_OK = @PMilter::Callbacks::EXPORT_OK;
our @EXPORT = @EXPORT_OK;
our %EXPORT_TAGS = ( all => \@EXPORT_OK );

# Accessible globals

*DEFAULT_CALLBACKS = *PMilter::Callbacks::DEFAULT_CALLBACKS;

foreach my $sym (qw(main register setconn)) {
	*{"Sendmail::Milter::$sym"} = *{"PMilter::Server::$sym"};
}

foreach my $sym (@EXPORT_OK) {
	*{"Sendmail::Milter::$sym"} = *{"PMilter::Callbacks::$sym"};
}

=pod

=item auto_getconn(NAME[, CONFIG])

Returns the connection descriptor for milter NAME in Sendmail configuration
file CONFIG (default "/etc/mail/sendmail.cf").  This can then be passed to
setconn(), below.

Returns a true value on success, undef on failure.

This is fully compatible with Sendmail::Milter.

=cut

sub auto_getconn {
	my $milter = shift;
	my $cf = shift || '/etc/mail/sendmail.cf';
	local *CF;

	open(CF, '<'.$cf) || confess "open $cf: $!";

	while (<CF>) {
		s/\s+$//; # also trims newlines

		s/^X([^,\s]+),\s*// || next;
		($milter eq $1) || next;

		while (s/^(.)=([^,\s]+)(,\s*|\Z)//) {
			if ($1 eq 'S') {
				close(CF);
				return $2;
			}
		}
	}

	close(CF);
	undef;
}

=pod

=item auto_setconn(NAME[, CONFIG])

Creates the server connection socket for milter NAME in Sendmail
configuration file CONFIG.

Essentially, does:

    setconn(auto_getconn(NAME, CONFIG))

Returns a true value on success, undef on failure.

This is fully compatible with Sendmail::Milter.

=cut

sub auto_setconn {
	my $conn = auto_getconn(@_);

	defined($conn) ? setconn($conn) : undef;
}

=pod

=item main([MAXCONN[, MAXREQ]])

See L<PMilter::Server> for a description of this method.

=item register(NAME, CALLBACKS[, FLAGS])

See L<PMilter::Server> for a description of this method.

=item sendmail_class(CLASS[, CONFIG])

Returns a list containing all members of the Sendmail class CLASS, in
Sendmail configuration file CONFIG (default "/etc/mail/sendmail.cf").  
Typically this is used to look up the entries in class "w", the local
hostnames class.

This is a PMilter extension.

=cut

sub sendmail_class {
	my $class = shift;
	my $cf = shift || '/etc/mail/sendmail.cf';
	my %entries;
	local *CF;

	open(CF, '<'.$cf) || confess "open $cf: $!";

	while (<CF>) {
		s/\s+$//; # also trims newlines

		if (s/^C\s*$class\s*//) {
			foreach (split(/\s+/)) {
				$entries{$_} = 1;
			}
		} elsif (s/^F\s*$class\s*(-o)?\s*//) {
			my $required = !defined($1);
			local *I;

			confess "sendmail class $class lookup resulted in pipe: $_" if (/^\|/);

			if (open(I, '<'.$_)) {
				while (<I>) {
					s/#.*$//;
					s/\s+$//;
					next if /^$/;
					$entries{$_} = 1;
				}
				close(I);
			} elsif ($required) {
				confess "sendmail class $class lookup: $_: $!";
			}
		}
	}

	close(CF);
	keys %entries;
}

=pod

=item setconn(DESC)

See L<PMilter::Server> for a description of this method.

=cut

1;

__END__

=back

=head1 SEE ALSO

See http://sendmail-milter.sourceforge.net/ for a description of
how to use the Sendmail::Milter compatibility interface.  This is
the basis of PMilter's functionality, and is the standard for
writing Perl milter modules for cross-implementation compatibility.
