#!/usr/local/bin/perl
#
# $DUH: duh.org.pl,v 1.21 2002/12/19 15:27:49 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

# This is the actual milter frontend in production use on duh.org.
# Several PMilter concepts are demonstrated here, though this file
# is more of a "working configuration" example.

use strict;
use warnings;

use Carp;
use Data::Dumper;
use PMilter::Server qw(:all);
use PMilter::Session;
use PMilter::Callbacks;
use PMilter::Callbacks::Leaf;
use PMilter::Callbacks::DelayToRcpt;
use PMilter::DNSBL::List;
use PMilter::Modules qw(globrx);
use Sendmail::Milter;
use Socket;

my $urlmsg = 'Please see http://www.duh.org/spamblock/ for assistance.';
my $debug = 1;

########## DNSBLs

### List of blocked country codes.

my @ccs = qw(AE AR BR CL CN CO CU EG ID IR JO KR MY NG PK SG TG TH TM TW);
my %ccs = map { $_ => 1 } @ccs;

### zz.countries.nerd.dk transform

my @zzccs;

open(CC, '</usr/share/misc/country') || die $!;
while (<CC>) {
	s/#.*$//;
	s/\s+$//; # also strips newlines

	my @entry = split(/\t/);
	next unless @entry;

	if ($ccs{$entry[1]}) {
		$entry[3] =~ s/^0+//;
		push(@zzccs, inet_ntoa(pack('N', 0x7f000000 + $entry[3])));
	}
}
close(CC);

### Full country blocks.

my $ccmsg = "Access denied to %1: Due to excessive spam, we do not normally accept mail from your country - $urlmsg";

my $cc_bls = PMilter::DNSBL::List->new(
	'countries.spamhosts.duh.org',
	[ 'zz.countries.nerd.dk', \@zzccs ],
)->setdefault($ccmsg);

### Single-stage open relay inputs.

my $relaymsg = "Access denied to %1: This address is vulnerable to open-relay/open-proxy attacks (listed in %2) - $urlmsg";

my $relayinput_bls = PMilter::DNSBL::List->new(
	[ 'spamhosts.duh.org', '127.0.0.7' ],
	'list.dsbl.org',
	'relays.ordb.org',
	[ 'relays.osirusoft.com', [qw(127.0.0.2 127.0.0.9)] ],
	'relays.visi.com',
	'opm.blitzed.org',
	'proxies.relays.monkeys.com',
)->setdefault($relaymsg);

### Generic blacklists, including the above.

my $genmsg = "Access denied to %1: This address is listed by %2 as a spam source - $urlmsg";

my $generic_bls = PMilter::DNSBL::List->new(
	[ 'spamhosts.duh.org', [map "127.0.0.$_", (2,4,5,6,8,100)], 'Access denied to %1: Address disallowed by local rule - '.$urlmsg ],
	'dnsbl.njabl.org',
	'sbl.spamhaus.org',
	[ 'blackholes.five-ten-sg.com', [qw(127.0.0.2 127.0.0.4)] ],
)->setdefault($genmsg);

### Dynamic pools, plus all the above.

my $dynmsg = "Access denied to %1: This is a dynamic pool address; you must use your Internet provider\'s SMTP server for sending outbound mail - $urlmsg";

my $dynamic_bls = PMilter::DNSBL::List->new(
	[ 'spamhosts.duh.org', '127.0.0.3' ],
	[ 'dnsbl.njabl.org', '127.0.0.3' ],
	'dynablock.wirehub.net',
)->setdefault($dynmsg);

my $dynamic_rdns_bls = PMilter::DNSBL::List->new(
	[ 'rdns.spamhosts.duh.org', '127.0.0.3' ],
)->setdefault($dynmsg);

### Reusable groups.

my $allbutrelay_bls = PMilter::DNSBL::List->new($cc_bls, $dynamic_bls, $generic_bls);
my $all_bls = PMilter::DNSBL::List->new($cc_bls, $dynamic_bls, $generic_bls, $relayinput_bls);

########## Message data checks

# Strict-RFC violations, or other things that "shouldn't be there".
my $bad_headers = [
	# foreign encoded from/to/cc does not belong in message/rfc822
	'^(From|To|Cc): =\?[^\@]*\?=$',

	# these don't belong in transit
	'^X-UIDL: ',
];

# Known spam.
my $spam_headers = [
	# known spamware
	'^X-(AD2000-Serial|Advertisement):',
	'^X-Mailer: (Mail Bomber|Accucast)',

	# older Pegasus does this, but *lots* of spamware does too
	'^Comments: Authenticated sender is',

	# the law says you must tag, and my sanity says I must block
	'^Subject: ADV ?:',
];

########## Local callbacks

sub connect_callback {
	my $ctx = shift;
	my $host = shift;

	$ctx->setpriv({});
	return SMFIS_CONTINUE;
}

# Actual delivery boxes for remailing.
sub received_remailer_filter {
	my $ctx = shift;
	local $_ = $_[0];

	# Everything designated as a final delivery machine; doesn't
	# include SMTP submission (which could have dynamic IPs).

	($ctx->gethost =~ /pobox\.com$/) &&
		/by (cali-[123]|dolly1|granite|icicle|kumquat|nellie|wormwood)\.pobox\.com/i &&
		!/^from \S+ \(\S+\.pobox\.com /i && # skip internal rerouting
		return &{$_[1]}($ctx, @_);

	();
}

########## Callback tree

my $cb = PMilter::Callbacks::DelayToRcpt->new(
	PMilter::Callbacks::Leaf->new(
		# this group is excluded for local hosts and my mail forwarders
		[ 'connect_regex', [
				'^(?:\[(?:10\.|127\.0\.0\.1)|localhost\z)',
				'\.(?:pobox\.com|afn\.org|zoneedit\.com|dnsvr\.com|directnic\.com)$'
			], 'BREAK'
		],

		[ 'connect_dnsbl_rdns', $dynamic_rdns_bls ],
		[ 'connect_dnsbl_ip', $all_bls ],

		[ 'helo_rawliteral', undef, $urlmsg ],
		[ 'helo_unqualified', undef, $urlmsg ],
		[ 'helo_ipmismatch', undef, $urlmsg ],
		[ 'helo_regex', [ globrx(Sendmail::Milter::sendmail_class('w')) ],
			'Host "%1" is me; I do not talk to myself', $urlmsg ],
	),

	{ FLAGS => SMFIF_ADDHDRS, PACKAGE => __PACKAGE__ },
	[ 'header_nofrom', undef, $urlmsg ],
	[ 'header_regex', $bad_headers ],
	[ 'header_regex', $spam_headers, undef, '"NO UCE" means NO SPAM!' ],

	[ 'received_dnsbl_ip', $allbutrelay_bls, \&received_remailer_filter ],
	[ 'received_dnsbl_ip', $relayinput_bls ],
);

########## Setup, open logs, and fork

Sendmail::Milter::auto_setconn('pmilter') || die;
Sendmail::Milter::register('pmilter', $cb) || die;

# fork after above, so perl errors happen in the foreground
my $pid = fork;
die "fork: $!" if ($pid < 0);
exit 0 if $pid;

open(O, '>pid'); print O "$$\n"; close(O);

setsyslog('local2');
setloglevel(LOG_DEBUG);
printlog(LOG_NOTICE, 'master process starting');

local $PMilter::Session::DebugLevel = 0;
Sendmail::Milter::main;
