////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/Drawable.hpp>
#include <SFML/Graphics/OpenGL.hpp>


namespace sf
{
////////////////////////////////////////////////////////////
/// Default constructor
////////////////////////////////////////////////////////////
Drawable::Drawable(float Left, float Top, float ScaleX, float ScaleY, float Rotation, const Color& Col) :
myLeft     (Left),
myTop      (Top),
myScaleX   (ScaleX),
myScaleY   (ScaleY),
myRotation (Rotation),
myCenterX  (0.f),
myCenterY  (0.f),
myColor    (Col),
myBlendMode(Blend::Alpha)
{

}


////////////////////////////////////////////////////////////
/// Virtual destructor
////////////////////////////////////////////////////////////
Drawable::~Drawable()
{
    // Nothing to do
}


////////////////////////////////////////////////////////////
/// Set the left position of the object
////////////////////////////////////////////////////////////
void Drawable::SetLeft(float Left)
{
    myLeft = Left;
}


////////////////////////////////////////////////////////////
/// Set the top position of the object
////////////////////////////////////////////////////////////
void Drawable::SetTop(float Top)
{
    myTop = Top;
}


////////////////////////////////////////////////////////////
/// Set the position of the object
////////////////////////////////////////////////////////////
void Drawable::SetPosition(float Left, float Top)
{
    myLeft = Left;
    myTop  = Top;
}


////////////////////////////////////////////////////////////
/// Set the horizontal scale of the object
////////////////////////////////////////////////////////////
void Drawable::SetScaleX(float Scale)
{
    if (Scale > 0.f)
        myScaleX = Scale;
}


////////////////////////////////////////////////////////////
/// Set the vertical scale of the object
////////////////////////////////////////////////////////////
void Drawable::SetScaleY(float Scale)
{
    if (Scale > 0.f)
        myScaleY = Scale;
}


////////////////////////////////////////////////////////////
/// Set the scale of the object
////////////////////////////////////////////////////////////
void Drawable::SetScale(float ScaleX, float ScaleY)
{
    if (ScaleX > 0.f) myScaleX = ScaleX;
    if (ScaleY > 0.f) myScaleY = ScaleY;
}


////////////////////////////////////////////////////////////
/// Set the orientation of the object
////////////////////////////////////////////////////////////
void Drawable::SetRotation(float Rotation)
{
    myRotation = Rotation;
}


////////////////////////////////////////////////////////////
/// Set the center of rotation, in coordinates relative to the
/// object
////////////////////////////////////////////////////////////
void Drawable::SetRotationCenter(float X, float Y)
{
    myCenterX = X;
    myCenterY = Y;
}


////////////////////////////////////////////////////////////
/// Set the color of the object
////////////////////////////////////////////////////////////
void Drawable::SetColor(const Color& Col)
{
    myColor = Col;
}


////////////////////////////////////////////////////////////
/// Set the blending mode for the object
////////////////////////////////////////////////////////////
void Drawable::SetBlendMode(Blend::Mode Mode)
{
    myBlendMode = Mode;
}


////////////////////////////////////////////////////////////
/// Get the left position of the object
////////////////////////////////////////////////////////////
float Drawable::GetLeft() const
{
    return myLeft;
}


////////////////////////////////////////////////////////////
/// Get the top position of the object
////////////////////////////////////////////////////////////
float Drawable::GetTop() const
{
    return myTop;
}


////////////////////////////////////////////////////////////
/// Get the horizontal scale of the object
////////////////////////////////////////////////////////////
float Drawable::GetScaleX() const
{
    return myScaleX;
}


////////////////////////////////////////////////////////////
/// Get the vertical scale of the object
////////////////////////////////////////////////////////////
float Drawable::GetScaleY() const
{
    return myScaleY;
}


////////////////////////////////////////////////////////////
/// Get the orientation of the object
////////////////////////////////////////////////////////////
float Drawable::GetRotation() const
{
    return myRotation;
}


////////////////////////////////////////////////////////////
/// Get the color of the object
////////////////////////////////////////////////////////////
const Color& Drawable::GetColor() const
{
    return myColor;
}


////////////////////////////////////////////////////////////
/// Move the object
////////////////////////////////////////////////////////////
void Drawable::Move(float OffsetX, float OffsetY)
{
    myLeft += OffsetX;
    myTop  += OffsetY;
}


////////////////////////////////////////////////////////////
/// Scale the object
////////////////////////////////////////////////////////////
void Drawable::Scale(float FactorX, float FactorY)
{
    if (FactorX > 0.f) myScaleX *= FactorX;
    if (FactorY > 0.f) myScaleY *= FactorY;
}


////////////////////////////////////////////////////////////
/// Rotate the object
////////////////////////////////////////////////////////////
void Drawable::Rotate(float Angle)
{
    myRotation += Angle;
}


////////////////////////////////////////////////////////////
/// Get the current blending mode
////////////////////////////////////////////////////////////
Blend::Mode Drawable::GetBlendMode() const
{
    return myBlendMode;
}


////////////////////////////////////////////////////////////
/// Draw the object into the specified window
////////////////////////////////////////////////////////////
void Drawable::Draw(const RenderWindow& Window) const
{
    // Save the current modelview matrix and reset it
    GLCheck(glMatrixMode(GL_MODELVIEW));
    GLCheck(glPushMatrix());
    GLCheck(glLoadIdentity());

    // Set transformations
    GLCheck(glTranslatef(myLeft + myCenterX, myTop + myCenterY, 0));
    GLCheck(glRotatef(-myRotation, 0, 0, 1));
    GLCheck(glTranslatef(-myCenterX, -myCenterY, 0));
    GLCheck(glScalef(myScaleX, myScaleY, 1));

    // Setup alpha-blending
    if (myBlendMode == Blend::None)
    {
        GLCheck(glDisable(GL_BLEND));
        GLCheck(glDisable(GL_ALPHA_TEST));
    }
    else
    {
        GLCheck(glEnable(GL_BLEND));
        GLCheck(glEnable(GL_ALPHA_TEST));
        switch (myBlendMode)
        {
            case Blend::Alpha :    GLCheck(glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)); break;
            case Blend::Add :      GLCheck(glBlendFunc(GL_ONE, GL_ONE));                       break;
            case Blend::Multiply : GLCheck(glBlendFunc(GL_DST_COLOR, GL_ZERO));                break;
            default : break;
        }
    }

    // Set color
    GLCheck(glColor4ub(myColor.r, myColor.g, myColor.b, myColor.a));

    // Let the derived class render the object geometry
    Render(Window);

    // Restore the previous modelview matrix
    GLCheck(glMatrixMode(GL_MODELVIEW));
    GLCheck(glPopMatrix());
}

} // namespace sf
