////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Window/WindowImpl.hpp>
#include <SFML/Window/Event.hpp>
#include <SFML/Window/OpenGLCaps.hpp>
#include <SFML/Window/WindowListener.hpp>
#include <algorithm>
#include <cmath>

#if defined(SFML_SYSTEM_WINDOWS)

    #include <SFML/Window/Win32/WindowImplWin32.hpp>

#elif defined(SFML_SYSTEM_LINUX)

    #include <SFML/Window/Linux/WindowImplX11.hpp>

#elif defined(SFML_SYSTEM_MACOS)

    #include <SFML/Window/OSX/WindowImplCarbon.hpp>

#endif


namespace sf
{
namespace priv
{
////////////////////////////////////////////////////////////
// Static member data
////////////////////////////////////////////////////////////
const WindowImpl* WindowImpl::ourCurrent = NULL;


////////////////////////////////////////////////////////////
/// Create a new window depending on the current OS
////////////////////////////////////////////////////////////
WindowImpl* WindowImpl::New()
{
    WindowImpl* Window = NULL;

    #if defined(SFML_SYSTEM_WINDOWS)

        // Win32 window
        Window = new WindowImplWin32();

    #elif defined(SFML_SYSTEM_LINUX)

        // Unix window
        Window = new WindowImplX11();

    #elif defined(SFML_SYSTEM_MACOS)

        // Carbon window
        Window = new WindowImplCarbon();

    #endif

    // This is supposed to be the first window created,
    // so once it is done we can initialize the OpenGL caps
    OpenGLCaps::Initialize();

    return Window;
}


////////////////////////////////////////////////////////////
/// Create a new window depending on the current OS
////////////////////////////////////////////////////////////
WindowImpl* WindowImpl::New(VideoMode Mode, const std::string& Title, unsigned long WindowStyle, int AntialiasingLevel)
{
    #if defined(SFML_SYSTEM_WINDOWS)

        // Win32 window
        return new WindowImplWin32(Mode, Title, WindowStyle, AntialiasingLevel);

    #elif defined(SFML_SYSTEM_LINUX)

        // Unix window
        return new WindowImplX11(Mode, Title, WindowStyle, AntialiasingLevel);

    #elif defined(SFML_SYSTEM_MACOS)

        // Carbon window
        return new WindowImplCarbon(Mode, Title, WindowStyle, AntialiasingLevel);

    #endif
}


////////////////////////////////////////////////////////////
/// Create a new window depending on the current OS
////////////////////////////////////////////////////////////
WindowImpl* WindowImpl::New(WindowHandle Handle, int AntialiasingLevel)
{
    #if defined(SFML_SYSTEM_WINDOWS)

        // Win32 window
        return new WindowImplWin32(Handle, AntialiasingLevel);

    #elif defined(SFML_SYSTEM_LINUX)

        // Unix window
        return new WindowImplX11(Handle, AntialiasingLevel);

    #elif defined(SFML_SYSTEM_MACOS)

        // Carbon window
        return new WindowImplCarbon(Handle, AntialiasingLevel);

    #endif
}


////////////////////////////////////////////////////////////
/// Default constructor
////////////////////////////////////////////////////////////
WindowImpl::WindowImpl() :
myWidth       (0),
myHeight      (0),
myDepthBits   (0),
myStencilBits (0),
myJoyThreshold(0.1f)
{
    // Initialize the joysticks
    for (unsigned int i = 0; i < JoysticksCount; ++i)
    {
        myJoysticks[i].Initialize(i);
        myJoyStates[i] = myJoysticks[i].UpdateState();
    }
}


////////////////////////////////////////////////////////////
/// Destructor
////////////////////////////////////////////////////////////
WindowImpl::~WindowImpl()
{
    // Nothing to do
}


////////////////////////////////////////////////////////////
/// Add a listener to the window
////////////////////////////////////////////////////////////
void WindowImpl::AddListener(WindowListener* Listener)
{
    if (Listener)
        myListeners.insert(Listener);
}


////////////////////////////////////////////////////////////
/// Remove a listener from the window
////////////////////////////////////////////////////////////
void WindowImpl::RemoveListener(WindowListener* Listener)
{
    myListeners.erase(Listener);
}


////////////////////////////////////////////////////////////
/// Get the client width of the window
////////////////////////////////////////////////////////////
unsigned int WindowImpl::GetWidth() const
{
    return myWidth;
}


////////////////////////////////////////////////////////////
/// Get the client height of the window
////////////////////////////////////////////////////////////
unsigned int WindowImpl::GetHeight() const
{
    return myHeight;
}


////////////////////////////////////////////////////////////
/// Get the depth buffer bits
////////////////////////////////////////////////////////////
unsigned int WindowImpl::GetDepthBits() const
{
    return myDepthBits;
}


////////////////////////////////////////////////////////////
/// Get the stencil buffer bits
////////////////////////////////////////////////////////////
unsigned int WindowImpl::GetStencilBits() const
{
    return myStencilBits;
}


////////////////////////////////////////////////////////////
/// Set the window as the current target for rendering
////////////////////////////////////////////////////////////
void WindowImpl::SetCurrent() const
{
    if (ourCurrent != this)
    {
        MakeCurrent();
        ourCurrent = this;
    }
}


////////////////////////////////////////////////////////////
/// Change the joystick threshold, ie. the value below which
/// no move event will be generated
////////////////////////////////////////////////////////////
void WindowImpl::SetJoystickThreshold(float Threshold)
{
    myJoyThreshold = Threshold;
}


////////////////////////////////////////////////////////////
/// Process incoming events from operating system
////////////////////////////////////////////////////////////
void WindowImpl::DoEvents()
{
    // Read the joysticks state and generate the appropriate events
    ProcessJoystickEvents();

    // Let the derived class process other events
    ProcessEvents();
}


////////////////////////////////////////////////////////////
/// Send an event to listeners
////////////////////////////////////////////////////////////
void WindowImpl::SendEvent(const Event& EventToSend)
{
    for (std::set<WindowListener*>::iterator i = myListeners.begin(); i != myListeners.end(); ++i)
    {
        (*i)->OnEvent(EventToSend);
    }
}


////////////////////////////////////////////////////////////
/// Read the joysticks state and generate the appropriate events
////////////////////////////////////////////////////////////
void WindowImpl::ProcessJoystickEvents()
{
    for (unsigned int i = 0; i < JoysticksCount; ++i)
    {
        // Copy the previous state of the joystick and get the new one
        JoystickState PreviousState = myJoyStates[i];
        myJoyStates[i] = myJoysticks[i].UpdateState();

        // Axis
        for (unsigned int j = 0; j < myJoysticks[i].GetAxesCount(); ++j)
        {
            float PrevPos = PreviousState.Axis[j];
            float CurrPos = myJoyStates[i].Axis[j];
            if (fabs(CurrPos - PrevPos) >= myJoyThreshold)
            {
                Event Event;
                Event.Type               = Event::JoyMoved;
                Event.JoyMove.JoystickId = i;
                Event.JoyMove.Axis       = static_cast<Joy::Axis>(j);
                Event.JoyMove.Position   = CurrPos;
                SendEvent(Event);
            }
        }

        // Buttons
        for (unsigned int j = 0; j < myJoysticks[i].GetButtonsCount(); ++j)
        {
            bool PrevPressed = PreviousState.Buttons[j];
            bool CurrPressed = myJoyStates[i].Buttons[j];

            if ((!PrevPressed && CurrPressed) || (PrevPressed && !CurrPressed))
            {
                Event Event;
                Event.Type                 = CurrPressed ? Event::JoyButtonPressed : Event::JoyButtonReleased;
                Event.JoyButton.JoystickId = i;
                Event.JoyButton.Button     = j;
                SendEvent(Event);
            }
        }
    }
}


} // namespace priv

} // namespace sf
