/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim:expandtab:autoindent:tabstop=4:shiftwidth=4:filetype=c:cindent:textwidth=0:
 *
 * Copyright (C) 2005 Dell Inc.
 *  by Michael Brown <Michael_E_Brown@dell.com>
 * Licensed under the Open Software License version 2.1 
 * 
 * Alternatively, you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, 
 * or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU General Public License for more details.
 */

// for windows auto-lib selection.
// tells compiler to include smbiosxml libs.
// also does additional checks to ensure runtime lib in compiler config 
// is compatible with xerces
#define LIBSMBIOS_NEED_SMBIOSXML

// compat header should always be first header if including system headers
#include "smbios/compat.h"

#include <iomanip>
#include <fstream>

#include "testSmbiosXml.h"
#include "smbios/SmbiosDefs.h"

// specific to unit tests. Users do not need to include this,
// so it is not in testSmbiosXml.h
#include "smbios/IMemory.h"
#include "smbios/ISmi.h"
#include "smbios/IObserver.h"

#include "smbios/version.h"

using namespace std;

extern string global_programDirname;
extern string global_writeDirectory;

// Note:
//      Except for , there are no "using namespace XXXX;" statements
//      here... on purpose. We want to ensure that while reading this code that
//      it is extremely obvious where each function is coming from.
//
//      This leads to verbose code in some instances, but that is fine for
//      these purposes.

// Register the test
CPPUNIT_TEST_SUITE_REGISTRATION (testSmbiosXml);

void copyFile( string dstFile, string srcFile )
{
    ifstream src(srcFile.c_str(), ios_base::binary);
    ofstream dst(dstFile.c_str(), ios_base::out | ios_base::binary | ios_base::trunc);

    char ch;
    while( src.get(ch)) dst.put(ch);

    if( !src.eof() || !dst ) throw exception();
}

bool fileExists(string fileName)
{
    FILE *fh=0;
    fh=fopen(fileName.c_str(), "rb");
    if(!fh)
        return false;

    fclose(fh);
    return true;
}

void testSmbiosXml::setUp()
{
    string xmlFile = global_programDirname + getXmlFile();
    string testInput = global_programDirname + getTestDirectory() + "/testInput.xml";
    if(!fileExists(testInput))
        testInput = getTestDirectory() + "/testInput.xml"; 

    // copy the memdump.dat file. We do not write to it, but rw open will fail
    // if we do not copy it
    string memdumpOrigFile = global_programDirname + getTestDirectory() + "/memdump.dat";
    if(!fileExists(memdumpOrigFile))
        memdumpOrigFile = getTestDirectory() + "/memdump.dat";
    string memdumpCopyFile = global_writeDirectory + "/memdump-copy.dat";
    copyFile( memdumpCopyFile, memdumpOrigFile );

    // copy the CMOS file. We are going to write to it and do not wan to mess up
    // the pristine unit test version
    string cmosOrigFile = global_programDirname + getTestDirectory() + "/cmos.dat";
    if(!fileExists(cmosOrigFile))
        cmosOrigFile = getTestDirectory() + "/cmos.dat";
    string cmosCopyFile = global_writeDirectory + "/cmos-copy.dat";
    copyFile( cmosCopyFile, cmosOrigFile );

    // Smi output file.
    string smiOutput = global_writeDirectory + "/smi-output.dat";

    // set up XML factory. from here on, we can just say SmbiosFactory.
    smbios::SmbiosXmlFactory::getFactory();

    // normal users of the smbios classes need not
    // set the four parameters below. They should all be set inside the factory
    // properly by default. We override stuff here to have
    // the smbios, cmos, etc classes use file dumps instead of
    // real memory/cmos/etc.
    smbios::SmbiosFactory::getFactory()->setParameter("memFile", memdumpCopyFile);
    smbios::SmbiosFactory::getFactory()->setParameter("offset", 0);
    smbios::SmbiosFactory::getFactory()->setMode(smbios::SmbiosFactory::UnitTestMode);

    cmos::  CmosRWFactory::getFactory()->setParameter("cmosMapFile", cmosCopyFile);
    cmos::  CmosRWFactory::getFactory()->setMode( factory::IFactory::UnitTestMode );

    memory::MemoryFactory::getFactory()->setParameter("memFile", memdumpCopyFile);
    memory::MemoryFactory::getFactory()->setMode( memory::MemoryFactory::UnitTestMode );

    smi::SmiFactory::getFactory()->setParameter("smiFile", smiOutput);
    smi::SmiFactory::getFactory()->setMode( smi::SmiFactory::UnitTestMode );

    // The parameter below will normally need to be set by the client code.
    smbios::SmbiosFactory::getFactory()->setParameter("xmlFile", xmlFile);

    parser = 0;
    doc = 0;
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER XMLPlatformUtils::Initialize();
        parser = xmlutils::getParser();
        doc = parser->parseURI( testInput.c_str() );
    }
    catch ( ... )
    {}

}

void testSmbiosXml::resetFactoryToBuiltinXml()
{
    smbios::SmbiosFactory::getFactory()->setParameter("xmlFile", "");
}

void testSmbiosXml::tearDown()
{
    // the factory is static. If we do not reset the factory, the next
    // unit test may accidentally get the wrong objects.
    // Lifetime rules: CmosTokenTable cannot live longer than the ISmbiosTable
    // object used in its construction.
    smbios::TokenTableFactory::getFactory()->reset();

    smbios::SmbiosFactory::getFactory()->reset();

    memory::MemoryFactory::getFactory()->reset();

    cmos::CmosRWFactory::getFactory()->reset();

    smi::SmiFactory::getFactory()->reset();

    parser->resetDocumentPool();
    parser->release();
    parser = 0;
    doc = 0;
    XERCES_CPP_NAMESPACE_QUALIFIER XMLPlatformUtils::Terminate();
}

// checkSkipTest for Skipping known BIOS Bugs.
void 
testSmbiosXml::checkSkipTest(string testName)
{
    if(!doc)
        return;

    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *testsToSkip = xmlutils::findElement(doc->getDocumentElement(),"testsToSkip","","");
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *test = xmlutils::findElement(testsToSkip,"test","name",testName);

        if(test)
            throw skip_test();
    }
    catch (const skip_test &)
    {
        throw;
    }
    catch (const exception &)
    {
        //Do Nothing
    }
}


//
//
// TABLE tests
//
//

void testSmbiosXml::testTable_Subscript()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    // PURPOSE:
    //      The purpose of this test is to test the subscript operator [].
    //      It tests these operators using string and int args and tests
    //      it outside of a loop.

    // table should not be deleted when we are finished. It is managed by the
    // factory. Factory will delete it for us when ->reset() is called.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item1;
    smbios::ISmbiosTable::iterator item2;

    item1 = (*table)["BIOS Information"];
    item2 = (*table)[smbios::BIOS_Information];

    // Use CPPUNIT_ASSERT_EQUAL when testing equality.
    // It gives better diagnostics on failure.
    CPPUNIT_ASSERT_EQUAL( item1->getHandle(), item2->getHandle() );
    CPPUNIT_ASSERT_EQUAL( item1->getLength(), item2->getLength() );
    CPPUNIT_ASSERT_EQUAL( item1->getType()  , item2->getType()   );

    CPPUNIT_ASSERT_EQUAL( (*table)[smbios::BIOS_Information]->getType(), item1->getType() );
    CPPUNIT_ASSERT_EQUAL( (*table)["BIOS Information"]->getType(), item1->getType() );

    item1 = (*table)["System Information"];
    item2 = (*table)[smbios::System_Information];

    CPPUNIT_ASSERT_EQUAL( item1->getHandle(), item2->getHandle() );
    CPPUNIT_ASSERT_EQUAL( item1->getLength(), item2->getLength() );
    CPPUNIT_ASSERT_EQUAL( item1->getType()  , item2->getType()   );

    CPPUNIT_ASSERT_EQUAL( (*table)[smbios::System_Information]->getType(), item1->getType() );
    CPPUNIT_ASSERT_EQUAL( (*table)["System Information"]->getType(), item1->getType() );


    STD_TEST_END();
}

void testSmbiosXml::testTable_Subscript_builtinXml()
{
    resetFactoryToBuiltinXml();
    testTable_Subscript();
}

void
testSmbiosXml::testEntryCount ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    // test streamify() while we are at it.
    ostringstream ost;
    int tableEntriesCounted = 0;
    smbios::ISmbiosTable::iterator item = table->begin();
    while( item != table->end() )
    {
        tableEntriesCounted++;
        ost << *item << endl;
        ++item;
    }

    CPPUNIT_ASSERT_EQUAL( tableEntriesCounted, table->getNumberOfEntries() );
    STD_TEST_END();
}

void testSmbiosXml::testEntryCount_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testEntryCount();
}

void
testSmbiosXml::testConstIterator ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    const smbios::ISmbiosTable *constTable = &*table;

    int tableEntriesCounted = 0;
    smbios::ISmbiosTable::const_iterator item = constTable->begin();
    while( item != constTable->end() )
    {
        tableEntriesCounted++;
        (void) *item;  // do this to sniff out possible errors with deref iterator.

        ++item;
    }
    CPPUNIT_ASSERT_EQUAL( tableEntriesCounted, constTable->getNumberOfEntries() );
    STD_TEST_END();
}

void testSmbiosXml::testConstIterator_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testConstIterator();
}

void
testSmbiosXml::testSubscriptOperator1 ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    int tableEntriesCounted = 0;
    // table[-1] is special, it returns all objects.
    // This test should be identical to testConstIterator (both walk all entries)
    for( smbios::ISmbiosTable::iterator item = (*table)[-1] ; item != table->end(); ++item)
    {
        tableEntriesCounted++;
        (void) *item;  // do this to sniff out possible errors with deref iterator.
    }
    CPPUNIT_ASSERT_EQUAL( table->getNumberOfEntries(), tableEntriesCounted );
    STD_TEST_END();
}

void testSmbiosXml::testSubscriptOperator1_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSubscriptOperator1();
}

void
testSmbiosXml::testSubscriptOperator2 ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

// it turns out that 8450 has 3 BIOS Information blocks.

//    // do not delete. Factory manages lifetime.
//    smbios::ISmbiosTable *table =
//        smbios::SmbiosFactory::getFactory()->getSingleton();
//
//    // There should probably be only one "BIOS Information" block. Check.
//    //  update this test if it turns out that there are actual systems with >1 Bios Info block
//    //
//    int tableEntriesCounted = 0;
//    for( smbios::ISmbiosTable::iterator item = (*table)[0] ; item != table->end(); ++item)
//    {
//        (void) *item;  // do this to sniff out possible errors with deref iterator.
//        tableEntriesCounted++;
//    }
//    CPPUNIT_ASSERT_EQUAL( 1, tableEntriesCounted );

    STD_TEST_END();
}

void testSmbiosXml::testSubscriptOperator2_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSubscriptOperator2();
}

void
testSmbiosXml::testSubscriptOperator3 ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    // There should normally be more than one "Port Connector" block. Check.
    //                               "Port Connector" block is type == 8
    //   memdump-opti.txt     has 12
    //   memdump-PAweb110.txt has 17
    //   memdump-PAweb110.txt has 9  *unverified...
    //  update this test if it turns out that there are actual systems with <2 Port Connector blocks
    //
    int tableEntriesCounted = 0;
    for( smbios::ISmbiosTable::iterator item = (*table)[8] ; item != table->end(); ++item)
    {
        (void) *item;  // do this to sniff out possible errors with deref iterator.
        tableEntriesCounted++;
    }
    CPPUNIT_ASSERT( 1 < tableEntriesCounted );
    STD_TEST_END();
}

void testSmbiosXml::testSubscriptOperator3_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSubscriptOperator3();
}



//
//
// ITEM tests
//
//

void testSmbiosXml::testStreamify()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // BEGIN EXAMPLE iterator
    // table should not be deleted when we are finished. It is managed by the
    // factory. Factory will delete it for us when ->reset() is called.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    ostringstream ost;
    ost << *table << endl;

    // iterate over all PORT INFORMATION BLOCKS
    for( smbios::ISmbiosTable::iterator item = (*table)[8] ; item != table->end(); ++item)
    {
        ost << *item << endl;
    }
    // END EXAMPLE iterator

    STD_TEST_END();
}

void
testSmbiosXml::testItemIdentity ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    // test that when we grab things out of the
    // table, we get copies of the same thing
    // when we ask for the same thing, rather than
    // separate items with the same data.
    //
    // we use references below to make the CPPUNIT_ASSERT
    // a bit easier to read.

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    // grab the first bios block
    smbios::ISmbiosTable::iterator position1 = (*table)[smbios::BIOS_Information];
    smbios::ISmbiosItem &item1 = *position1; //reference

    // use another iterator and grab another
    smbios::ISmbiosTable::iterator position2 = (*table)[smbios::BIOS_Information];
    smbios::ISmbiosItem &item2 = *position2; //reference

    // Check that they both gave the same thing
    // The address of each should be equal
    CPPUNIT_ASSERT_EQUAL(  &item1, &item2 );  // easiest to read
    CPPUNIT_ASSERT_EQUAL(  &(*position1), &item2 ); // same test, written differently
    CPPUNIT_ASSERT_EQUAL(  &item1       , &(*position2) ); // same test, written differently
    CPPUNIT_ASSERT_EQUAL(  &(*position1), &(*position2) ); // same test, written differently

    // use another iterator and grab another _different_ pointer.
    smbios::ISmbiosTable::iterator position3 = (*table)[smbios::System_Information];
    smbios::ISmbiosItem &item3 = *position3; //reference

    // Check that these are different...
    CPPUNIT_ASSERT( &item1 != &item3 );
    CPPUNIT_ASSERT( &item2 != &item3 );

    CPPUNIT_ASSERT_EQUAL( item1.getType(), static_cast<u8>(smbios::BIOS_Information) );
    CPPUNIT_ASSERT_EQUAL( item2.getType(), static_cast<u8>(smbios::BIOS_Information) );
    CPPUNIT_ASSERT_EQUAL( item3.getType(), static_cast<u8>(smbios::System_Information) );

    STD_TEST_END();
}

void testSmbiosXml::testItemIdentity_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testItemIdentity();
}

void
testSmbiosXml::testEachItemAccessors ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    // test ->getU8 and ->getU16 methods
    // no way to test in generic table the ->getU32
    // or ->getU64
    //
    // primarily to ensure that getUx() has the endianness correct

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item = table->begin();
    while( item != table->end() )
    {
        u8 type1 = item->getType();
        u8 type2 = item->getU8(0);
        CPPUNIT_ASSERT_EQUAL( type1, type2 );

        u8 len1 = item->getLength();
        u8 len2 = item->getU8(1);
        CPPUNIT_ASSERT_EQUAL( len1, len2 );

        u16 handle1 = item->getHandle();
        u16 handle2 = item->getU16(2);
        CPPUNIT_ASSERT_EQUAL( handle1, handle2 );

        ++item;
    }

    STD_TEST_END();
}

void testSmbiosXml::testEachItemAccessors_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSmbiosXml::testEachItemAccessors();
}

void testSmbiosXml::testItem_GetBiosInfo()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    // Purpose:
    //      The purpose of this test is to exercise all of the getXX()
    //      functions that are avalable using std SMBIOS tables that are
    //      available on all dumps. Do not rely on any specific value in any
    //      table in these tests as this test will be run against many
    //      tables.
    //

    // BEGIN EXAMPLE factory
    // table should not be deleted when we are finished. It is managed by the
    // factory. Factory will delete it for us when ->reset() is called.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item1 = (*table)["BIOS Information"];

    //
    // here is an example XML for BIOS Information Block, which we test below.
    //
    //<FIELD offset="0h" name="Type" length="BYTE" usage="STRUCTURE_TYPE"/>
    //<FIELD offset="1h" name="Length" length="BYTE" usage="SIZE"/>
    //<FIELD offset="2h" name="Handle" length="WORD" usage="HANDLE"/>
    //<FIELD offset="4h" name="Vendor" length="BYTE" usage="STRING"/>
    //<FIELD offset="5h" name="BIOS Version" length="BYTE" usage="STRING"/>
    //<FIELD offset="6h" name="BIOS Starting Address Segment" length="WORD" usage="ADDRESS"/>
    //<FIELD offset="8h" name="BIOS Release Date" length="BYTE" usage="STRING"/>
    //<FIELD offset="9h" name="BIOS ROM Size" length="BYTE" usage="SIZE"/>
    //<FIELD offset="Ah" name="BIOS Characteristics" length="QWORD" usage="BITFIELD">
    //

    u64 ll1, ll2;
    int int1, int2;
    string str1, str2;

    // Test Philosophy:
    //      There are multiple ways to access any particular item in the table.
    //      If you have an XML smbios file, you can access by string name and
    //      the code will look the name up in the XML.
    //
    //      We play the different access modes off each other below to ensure
    //      that each access mode returns the exact same data.
    int1 = item1->getU8("Type");
    int2 = item1->getType();
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    int1 = item1->getU8("Length");
    int2 = item1->getLength();
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    int1 = item1->getU16("Handle");
    int2 = item1->getHandle();
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    str1 = item1->getString("Vendor") ;
    str2 = item1->getString( 0x4 ) ;
    CPPUNIT_ASSERT_EQUAL (str1, str2);

    str1 = item1->getString("BIOS Version") ;
    str2 = item1->getString( 0x5 ) ;
    CPPUNIT_ASSERT_EQUAL (str1, str2);

    int1 = item1->getU16("BIOS Starting Address Segment");
    int2 = item1->getU16( 0x6 );
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    str1 = item1->getString("BIOS Release Date");
    str2 = item1->getString( 0x8 );
    CPPUNIT_ASSERT_EQUAL (str1, str2);

    int1 = item1->getU8("BIOS ROM Size");
    int2 = item1->getU8(0x9);
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    ll1 = item1->getU64("BIOS Characteristics");
    ll2 = item1->getU64(0xA);
    // will not compile on VC++
    //    CPPUNIT_ASSERT_EQUAL (ll1, ll2);

    //First get some bits from the BIOS Characteristics bitfield and compare to U64
    u32 bitfield = item1->getBitfield(0xA, smbios::ISmbiosItem::FIELD_LEN_QWORD, 0, 15);
    CPPUNIT_ASSERT_EQUAL (static_cast<u32>(ll2 & 0x000000000000FFFFL), bitfield);

    //Get some bits from the BIOS Characteristics bitfield using the other method
    //and compare it to the U64
    bitfield = 0;
    bitfield = item1->getBitfield("BIOS Characteristics", "Reserved0");
    bitfield |= item1->getBitfield("BIOS Characteristics", "Reserved1") << 1;
    bitfield |= item1->getBitfield("BIOS Characteristics", "Unknown") << 2;
    bitfield |= item1->getBitfield("BIOS Characteristics", "BIOS Characteristics Not Supported") << 3;
    bitfield |= item1->getBitfield("BIOS Characteristics", "ISA is supported") << 4;
    bitfield |= item1->getBitfield("BIOS Characteristics", "MCA is supported") << 5;
    bitfield |= item1->getBitfield("BIOS Characteristics", "EISA is supported") << 6;
    bitfield |= item1->getBitfield("BIOS Characteristics", "PCI is supported") << 7;
    bitfield |= item1->getBitfield("BIOS Characteristics", "PC Card (PCMCIA) is supported") << 8;
    bitfield |= item1->getBitfield("BIOS Characteristics", "Plug and Play is supported") << 9;
    bitfield |= item1->getBitfield("BIOS Characteristics", "APM is supported") << 0x0a;
    bitfield |= item1->getBitfield("BIOS Characteristics", "BIOS is Upgradeable (Flash)") << 0x0b;
    bitfield |= item1->getBitfield("BIOS Characteristics", "BIOS shadowing is allowed") << 0x0c;
    bitfield |= item1->getBitfield("BIOS Characteristics", "VL-VESA is supported") << 0x0d;
    bitfield |= item1->getBitfield("BIOS Characteristics", "ESCD support is available") << 0x0e;
    bitfield |= item1->getBitfield("BIOS Characteristics", "Boot from CD is supported") << 0x0f;
    //cout << "BIOS Characteristics Bitfield (0x" << hex << bitfield << ")" << dec << endl;
    //Compare the lower 15 bits with the bitfield we just retrieved
    CPPUNIT_ASSERT_EQUAL (static_cast<u32>(ll2 & 0x000000000000FFFFL), bitfield);

    STD_TEST_END();
}

void testSmbiosXml::testItem_GetBiosInfo_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testItem_GetBiosInfo();
}

void testSmbiosXml::testItem_GetSystemInfo()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    // PURPOSE:
    //      Same purpose as testGet_BiosInfo()

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item1 = ( *table )["System Information"];

    //
    // here is an example XML for System Information Block, which we test below.
    //
    // <FIELD offset="0h" name="Type" length="BYTE" usage="STRUCTURE_TYPE"/>
    // <FIELD offset="1h" name="Length" length="BYTE" usage="SIZE"/>
    // <FIELD offset="2h" name="Handle" length="WORD" usage="HANDLE"/>
    // <FIELD offset="4h" name="Manufacturer" length="BYTE" usage="STRING"/>
    // <FIELD offset="5h" name="Product Name" length="BYTE" usage="STRING"/>
    // <FIELD offset="6h" name="Version" length="BYTE" usage="STRING"/>
    // <FIELD offset="7h" name="Serial Number" length="BYTE" usage="STRING"/>
    // <FIELD offset="8h" version="2.1" name="UUID" length="BYTE" count="16" usage="NUMBER"/>
    // <FIELD offset="18h" version="2.1" name="Wake-up Type" length="BYTE" usage="ENUM">
    //
    //
    string str1, str2;
    int int1, int2;

    int1 = item1->getU8("Type");
    int2 = item1->getType();
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    int1 = item1->getU8("Length");
    int2 = item1->getLength();
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    int1 = item1->getU16("Handle");
    int2 = item1->getHandle();
    CPPUNIT_ASSERT_EQUAL (int1, int2);

    str1 = item1->getString("Manufacturer") ;
    str2 = item1->getString( 0x4 ) ;
    CPPUNIT_ASSERT_EQUAL (str1, str2);

    str1 = item1->getString("Product Name") ;
    str2 = item1->getString( 0x5 ) ;
    CPPUNIT_ASSERT_EQUAL (str1, str2);

#if 0
    //
    // This is not a good test case because several
    // of our BIOSs have a '\0' in the header for this
    // string, which means this string does not
    // exist. Lowlevel code will throw an exception.
    //
    str1 = item1->getString("Version") ;
    str2 = item1->getString( 0x6 ) ;
    CPPUNIT_ASSERT_EQUAL( str1, str2 );
#endif

    try
    {
        str1 = item1->getString("Serial Number") ;
        str2 = item1->getString( 0x7 ) ;
        CPPUNIT_ASSERT_EQUAL (str1, str2);
    }
    catch( const exception & )
    {
        // 4G systems do not support Serial Number.
    }

#if 0
    //
    // These are not good test cases because they are SMBIOS 2.3
    // additions and are not guaranteed to be present.
    //
    u8 val1, val2;
    val1 =  item1->getU8("UUID") ;
    val2 = item1->getU8( 0x8 ) ;
    CPPUNIT_ASSERT_EQUAL( val1, val2 );

    val1 = item1->getU8("Wake-up Type") ;
    val2 = item1->getU8( 0x9 ) ;
    CPPUNIT_ASSERT_EQUAL( val1, val2 );
#endif

    STD_TEST_END();
}

void testSmbiosXml::testItem_GetSystemInfo_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testItem_GetSystemInfo();
}


void testSmbiosXml::testTypeMismatch()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    // PURPOSE:
    //      The purpose of this test is to test all types of invalid item
    //      access. The getXX(string) methods all validate that the field
    //      passed to them is actually of type XX. If a mismatch is found, an
    //      exception is thrown.
    //
    //      Each test validates that an exception is thrown and the type.

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item1 = (*table)["BIOS Information"];

    //
    // refer to testGet_BiosInfo() for XML sample for BIOS INFORMATION BLOCK
    //

    // ASSERT_THROWS is not a CPPUNIT macro, it is our custom macro.
    ASSERT_THROWS( item1->getU8("Handle"), smbios::ParseException );
    ASSERT_THROWS( item1->getU16("Type"), smbios::ParseException );
    ASSERT_THROWS( item1->getU32("Type"), smbios::ParseException );
    ASSERT_THROWS( item1->getU64("Type"), smbios::ParseException );
    ASSERT_THROWS( item1->getString("Type"), smbios::ParseException );
    ASSERT_THROWS( item1->getBitfield("Type", "foo"), smbios::ParseException );

    STD_TEST_END();
}

void testSmbiosXml::testTypeMismatch_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSmbiosXml::testTypeMismatch();
}


void
testSmbiosXml::testGetBoundaries()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // do not delete. Factory manages lifetime.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    for( smbios::ISmbiosTable::iterator item = (*table)[-1] ; item != table->end(); ++item)
    {
        int len = item->getLength();

        // none of these should throw.
        (void) item->getU8(len - 1);
        (void) item->getU16(len - 2);
        (void) item->getU32(len - 4);

        // not all items are large enough. only attempt if item is at least 8 bytes
        if( len >= 8 )
            (void) item->getU64(len - 8);

        ASSERT_THROWS( item->getU8(len - 0), smbios::DataOutOfBounds);
        ASSERT_THROWS( item->getU16(len - 1), smbios::DataOutOfBounds);
        ASSERT_THROWS( item->getU32(len - 3), smbios::DataOutOfBounds);
        ASSERT_THROWS( item->getU64(len - 7), smbios::DataOutOfBounds);
    }

    STD_TEST_END();
}

void testSmbiosXml::testGetBoundaries_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSmbiosXml::testGetBoundaries();
}


//
// CMOS Token
//


void
testSmbiosXml::testCmosConstructor ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    smbios::TokenTableFactory *ttFactory;
    ttFactory = smbios::TokenTableFactory::getFactory() ;
    smbios::ITokenTable *tokenTable = ttFactory->getSingleton();

    ostringstream ost;

    smbios::ITokenTable::iterator token = tokenTable->begin();
    while( token != tokenTable->end() )
    {
        ost << *token << endl;
        ++token;
    }

    STD_TEST_END();
}

void
testSmbiosXml::testCmosChecksum ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    smbios::TokenTableFactory *ttFactory;
    ttFactory = smbios::TokenTableFactory::getFactory() ;
    smbios::ITokenTable *tokenTable = ttFactory->getSingleton();

    smbios::ITokenTable::iterator token = tokenTable->begin();
    while( token != tokenTable->end() )
    {
        (void) *token;
        ++token;
    }

    cmos::ICmosRW *cmos = cmos::CmosRWFactory::getFactory()->getSingleton();
    observer::IObservable *ob = dynamic_cast<observer::IObservable *>(cmos);
    bool doUpdate = false;
    if( ob )
        ob->notify(&doUpdate);

    STD_TEST_END();
}

void
testSmbiosXml::testCmosWriting ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    smbios::TokenTableFactory *ttFactory;
    ttFactory = smbios::TokenTableFactory::getFactory() ;
    smbios::ITokenTable *tokenTable = ttFactory->getSingleton();
    const smbios::ITokenTable *tokenTableC = ttFactory->getSingleton();

    ASSERT_THROWS( (*tokenTable) ["la la la"], smbios::NotImplemented );
    ASSERT_THROWS( (*tokenTableC)["la la la"], smbios::NotImplemented );

    // test [] on const table.
    (void) tokenTableC[0xFE];

    ostringstream ost;
    ost << *tokenTable << endl;
    ost << *tokenTableC << endl;

    // test const iterator
    smbios::ITokenTable::const_iterator tokenC = tokenTableC->begin();
    while( tokenC != tokenTableC->end() )
    {
        (void) *tokenC;
        (void) tokenC->isString();

        tokenC++;
    }

    // refuse to write to cmos unless the checksum is correct
    cmos::ICmosRW *cmos = cmos::CmosRWFactory::getFactory()->getSingleton();
    observer::IObservable *ob = dynamic_cast<observer::IObservable *>(cmos);
    bool doUpdate = false;
    if( ob )
        ob->notify(&doUpdate);

    smbios::ITokenTable::iterator token = tokenTable->begin();
    while( token != tokenTable->end() )
    {
        //cout << *token << endl;
        if( token->isString() )
        {
            const char *testStr = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890abcdefghijklmnop";
            const u8 *testStrU8 = reinterpret_cast<const u8*>(testStr);
            u8 *myStr=0;
            u8 *myStr1=0;
            try
            {
                // not really a valid test anymore since SMI tokens can be
                // accessed as bit or string.
                //ASSERT_THROWS( token->activate(), smbios::InvalidAccessMode );
                //ASSERT_THROWS( token->isActive(), smbios::InvalidAccessMode );

                unsigned int size = token->getStringLength() + 1;

                myStr1 = new u8[ size ];
                memset( myStr1, 0, size );

                try
                {
                    token->getString( myStr1, size );
                }
                catch(const smi::UnhandledSmi &)
                {   /* if token is a smi token, cannot unit test. */
                    delete [] myStr1;
                    goto next_token;
                }

                token->setString( testStrU8, strlen(testStr) + 1 );

                CPPUNIT_ASSERT( size <= strlen(testStr)+1 );

                myStr = new u8[ size ];
                memset( myStr, 0, size );
                token->getString( myStr, size );

                // return might be smaller, only compare up to what was stored.
                if( 0 != memcmp( testStr, reinterpret_cast<char*>(myStr), size - 1 ) )
                {
                    // FAILED
                    ostringstream ost;
                    ost << "String set on token failed." << endl;
                    ost << (*token) << endl;
                    ost << "Size of string to compare is: " << size-1 << endl;
                    ost << "Original data: (" << myStr1  << ")" << endl;
                    ost << "Wrote        : (" << testStr << ")" << endl;
                    ost << "Read back    : (" << myStr   << ")" << endl;
                    CPPUNIT_FAIL( ost.str().c_str() );
                }
            }
            catch(...)
            {
                delete [] myStr1;
                delete [] myStr;
                myStr1 = 0;
                myStr = 0;
                throw;
            }
            delete [] myStr1;
            delete [] myStr;
            myStr1 = 0;
            myStr = 0;
        }
        else
        {
            try
            {
                token->activate();
            }
            catch(const smi::UnhandledSmi &)
            {   /* if token is a smi token, cannot unit test. */
                goto next_token;
            }
            if( ! token->isActive() )
            {
                ostringstream ost;
                ost << "Failed to SET bit token. Token data: " << endl;
                ost << (*token);
                CPPUNIT_FAIL( ost.str().c_str() );
            }
            // not really a valid test anymore since SMI tokens can be
            // accessed as bit or string.
            //ASSERT_THROWS( token->setString(0, 0), smbios::InvalidAccessMode );
            //ASSERT_THROWS( token->getStringLength(), smbios::InvalidAccessMode );
        }

next_token:
        // test post-increment behaves properly
        smbios::ITokenTable::iterator before = token;
        smbios::ITokenTable::iterator after = token++;
        CPPUNIT_ASSERT( before == after );
    }

    // recheck the checksums.
    // ensure that we wrote correct checksums out
    if( ob )
        ob->notify(&doUpdate);

    STD_TEST_END();
}


//
// SMI Tests
//

void
testSmbiosXml::testSmi_callingInterface()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    std::auto_ptr<smi::ISmi> smi = smi::SmiFactory::getFactory()->makeNew(smi::SmiFactory::DELL_CALLING_INTERFACE_SMI);
    smi->setCommandIOMagic( 0x1234, 0x56 );

    smi::IDellCallingInterfaceSmi *ci = dynamic_cast<smi::IDellCallingInterfaceSmi *>(&*smi);
    ci->setClass( 0xAABB );
    ci->setSelect( 0xCCDD );
    ci->setArg( 0, 0xA1A2A3A4 );
    ci->setArg( 1, 0xB1B2B3B4 );
    ci->setArg( 2, 0xC1C2C3C4 );
    ci->setArg( 3, 0xD1D2D3D4 );
    try
    {   /* This is expected to fail in unit test, no good way to simulate them*/
        ci->execute();
    }
    catch( const smi::UnhandledSmi & ) {}

    STD_TEST_END();
}

void
testSmbiosXml::testSmi_callingInterface_physaddr ()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    std::auto_ptr<smi::ISmi> smi = smi::SmiFactory::getFactory()->makeNew(smi::SmiFactory::DELL_CALLING_INTERFACE_SMI);
    smi->setCommandIOMagic( 0x1234, 0x56 );

    smi::IDellCallingInterfaceSmi *ci = dynamic_cast<smi::IDellCallingInterfaceSmi *>(&*smi);
    ci->setClass( 0xAABB );
    ci->setSelect( 0xCCDD );
    ci->setArgAsPhysicalAddress(0, 0);
    ci->setArgAsPhysicalAddress(1, 1);
    ci->setArgAsPhysicalAddress(2, 2);
    ci->setArgAsPhysicalAddress(3, 3);
    try
    {   /* This is expected to fail in unit test, no good way to simulate them*/
        ci->execute();
    }
    catch( const smi::UnhandledSmi & ) {}

    STD_TEST_END();
}


//
// System Info
//


void
testSmbiosXml::testSystemInfo()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    int   systemId   = 0;
    const char *systemName = 0;
    const char *serviceTag = 0;
    const char *assetTag   = 0;
    const char *biosVersion   = 0;
    const char *vendorName    = 0;

    try
    {
        systemId   = SMBIOSGetDellSystemId();
        systemName = SMBIOSGetSystemName();
        serviceTag = SMBIOSGetServiceTag();
        assetTag   = SMBIOSGetAssetTag();
        biosVersion   = SMBIOSGetBiosVersion();
        vendorName    = SMBIOSGetVendorName();

        int   isDell        = SMBIOSIsDellSystem();

        (void) systemId; //avoid unused var warning
        (void) isDell; //avoid unused var warning

        //We should at least get an empty string from these
        //methods.  Never a null string.
        CPPUNIT_ASSERT(systemId != 0);
        CPPUNIT_ASSERT(systemName != 0);
        //CPPUNIT_ASSERT(serviceTag != 0); // svc tag can legitimately be 0
        //CPPUNIT_ASSERT(assetTag != 0); //This fails on latitude so we comment out for now.
        CPPUNIT_ASSERT(biosVersion != 0);
        CPPUNIT_ASSERT(vendorName != 0);
    }
    catch(...)
    {
        SMBIOSFreeMemory( systemName );
        SMBIOSFreeMemory( serviceTag );
        SMBIOSFreeMemory( assetTag );
        SMBIOSFreeMemory( biosVersion );
        SMBIOSFreeMemory( vendorName );

        throw;
    }

    SMBIOSFreeMemory( systemName );
    SMBIOSFreeMemory( serviceTag );
    SMBIOSFreeMemory( assetTag );
    SMBIOSFreeMemory( biosVersion );
    SMBIOSFreeMemory( vendorName );

    STD_TEST_END();
}

void testSmbiosXml::testSystemInfo_builtinXml ()
{
    resetFactoryToBuiltinXml();
    testSmbiosXml::testSystemInfo();
}


// testInput.xml tests


void
testSmbiosXml::testIdByte()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    int   systemId   = SMBIOSGetDellSystemId  ();

    if (!doc)
        throw skip_test();

    int id;
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *systeminfo = xmlutils::findElement( doc->getDocumentElement(), "systemInfo", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *idByte = xmlutils::findElement( systeminfo, "idByte", "", "" );
        string idStr = xmlutils::getNodeText( idByte );
        id = strtol( idStr.c_str(), 0, 0);
    }
    catch( const exception & )
    {
        throw skip_test();
    }

    CPPUNIT_ASSERT_EQUAL ( id, systemId );

    STD_TEST_END()
}

string testSmbiosXml::getTestInputString( string toFind )
{
    if (!doc)
        throw skip_test();

    string foundString = "";

    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *systeminfo = xmlutils::findElement( doc->getDocumentElement(), "systemInfo", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *sysName = xmlutils::findElement( systeminfo, toFind, "", "" );
        foundString = xmlutils::getNodeText( sysName );
    }
    catch( const exception & )
    {
        throw skip_test();
    }

    return foundString;
}

string safeConvertToString( const char *str )
{
    string fromSystem = "";
    if( 0 != str )
    {
        fromSystem = str;
    }
    SMBIOSFreeMemory(str);
    return fromSystem;
}

void
testSmbiosXml::testSystemName()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    string fromSystem = safeConvertToString( SMBIOSGetSystemName() );
    string testInput  = getTestInputString( "systemName" );

    CPPUNIT_ASSERT_EQUAL ( testInput, fromSystem );
    STD_TEST_END()
}

void
testSmbiosXml::testLibraryVersion()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    string fromSystem = SMBIOSGetLibraryVersionString();
    string testInput  = LIBSMBIOS_RELEASE_VERSION;

    CPPUNIT_ASSERT_EQUAL ( testInput, fromSystem );
    STD_TEST_END()
}

void
testSmbiosXml::testServiceTag()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    string fromSystem = safeConvertToString( SMBIOSGetServiceTag() );
    string testInput  = getTestInputString( "serviceTag" );

    CPPUNIT_ASSERT_EQUAL ( testInput, fromSystem );

    STD_TEST_END()
}

//  not part of public API, so just wing it here so that we can do the unit test.
extern char *getServiceTagFromCMOSToken();

void
testSmbiosXml::testServiceTagWriting()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    string fromSystem = safeConvertToString( SMBIOSGetServiceTag() );
    string testInput  = getTestInputString( "serviceTag" );

    CPPUNIT_ASSERT_EQUAL ( testInput, fromSystem );

    string rawCMOSOrig("");
    try
    {
        rawCMOSOrig = safeConvertToString( getServiceTagFromCMOSToken() );
    }
    catch(const exception &)
    {
        // if service tag is not in SMBIOS, we cannot do the
        // tests below
        throw skip_test();
    }

    CPPUNIT_ASSERT_EQUAL ( testInput, rawCMOSOrig );

    string shouldBe, rawCMOSNew;

    // test std 5-char svc tag
    shouldBe = "NEWTG";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // test std 7-char svc tag (alphabet 1/3)
    shouldBe = "BCDFGHJ";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // test std 7-char svc tag (alphabet 2/3)
    shouldBe = "KLMNPQR";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // test std 7-char svc tag (alphabet 3/3)
    shouldBe = "STVWXYZ";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (1)
    shouldBe = "A";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (2)
    shouldBe = "AB";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (3)
    shouldBe = "ABC";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (4)
    shouldBe = "ABCD";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (6)
    shouldBe = "BCDFGH";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    shouldBe = "BCDFGH0";  // invalid/missing chars for 7-char svc tag 
                    // converted to '0'
    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (7)
    shouldBe = "XGYZYYY";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // odd size (8)
    shouldBe = "MNPQMNPQ";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    shouldBe = "MNPQMNP"; // extra chars ignored
    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );

    // invalid chars in 7-char tag
    shouldBe = "ABEIOUD";
    SMBIOSSetServiceTag("", shouldBe.c_str(), strlen(shouldBe.c_str()));
    rawCMOSNew = safeConvertToString( getServiceTagFromCMOSToken() );

    shouldBe = "AB0000D"; // invalid chars turned into '0';
    CPPUNIT_ASSERT_EQUAL ( shouldBe, rawCMOSNew );


    STD_TEST_END()
}

void
testSmbiosXml::testAssetTag()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    string fromSystem = safeConvertToString( SMBIOSGetAssetTag() );
    string testInput  = getTestInputString( "assetTag" );

    CPPUNIT_ASSERT_EQUAL ( testInput, fromSystem );

    STD_TEST_END()
}

void
testSmbiosXml::testBiosVersion()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    string fromSystem = safeConvertToString( SMBIOSGetBiosVersion() );
    string testInput  = getTestInputString( "biosVersion" );

    CPPUNIT_ASSERT_EQUAL ( testInput, fromSystem );

    STD_TEST_END()
}

void
testSmbiosXml::testIsDell()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    int   isDell   = SMBIOSIsDellSystem  ();

    if (!doc)
        throw skip_test();

    int isDellExpected;
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *systeminfo = xmlutils::findElement( doc->getDocumentElement(), "systemInfo", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *val = xmlutils::findElement( systeminfo, "SMBIOSIsDellSystem", "", "" );
        string strval = xmlutils::getNodeText( val );
        isDellExpected = strtol( strval.c_str(), 0, 0);
    }
    catch( const exception & )
    {
        throw skip_test();
    }

    CPPUNIT_ASSERT_EQUAL ( isDell, isDellExpected );

    STD_TEST_END()
}

void testSmbiosXml::testVariousAccessors()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    // table should not be deleted when we are finished. It is managed by the
    // factory. Factory will delete it for us when ->reset() is called.
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item = (*table)["BIOS Information"] ;

    string vendorStr="";
    string versionStr="";
    string releaseStr="";

    if (!doc)
        throw skip_test();

    // pull info out of xml
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *smbios = xmlutils::findElement( doc->getDocumentElement(), "smbios", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *biosInfo = xmlutils::findElement( smbios, "biosInformation", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *vendor = xmlutils::findElement( biosInfo, "vendor", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *version = xmlutils::findElement( biosInfo, "version", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *release = xmlutils::findElement( biosInfo, "release", "", "" );
        vendorStr = xmlutils::getNodeText( vendor );
        versionStr = xmlutils::getNodeText( version );
        releaseStr = xmlutils::getNodeText( release );
    }
    catch( const exception & )
    {
        throw skip_test();
    }

    const string string1( item->getString(4) ); // BIOS VENDOR
    const string string2( item->getString(5) ); // BIOS VERSION
    const string string3( item->getString(8) ); // RELEASE DATE

    const string string4( item->getStringByStringNumber(1) ); //BIOS VENDOR
    const string string5( item->getStringByStringNumber(2) ); //BIOS VERSION
    const string string6( item->getStringByStringNumber(3) ); //RELEASE DATE

    CPPUNIT_ASSERT_EQUAL( vendorStr, string1 );
    CPPUNIT_ASSERT_EQUAL( versionStr, string2 );
    CPPUNIT_ASSERT_EQUAL( releaseStr, string3 );

    CPPUNIT_ASSERT_EQUAL( string1, string4 );
    CPPUNIT_ASSERT_EQUAL( string2, string5 );
    CPPUNIT_ASSERT_EQUAL( string3, string6 );

    STD_TEST_END();
}

void
testSmbiosXml::testStateBytes()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    if( ! SMBIOSHasNvramStateBytes() )
        throw skip_test();

    int testValue = 0;
    SMBIOSGetNvramStateBytes( 0x0000 );

    // test DSA mode
    testValue = 0x1234;
    SMBIOSSetNvramStateBytes( testValue, 0x0000 );
    CPPUNIT_ASSERT_EQUAL( testValue, SMBIOSGetNvramStateBytes( 0x0000 ) ); // DSA sees real value
    CPPUNIT_ASSERT_EQUAL( 0x0000,    SMBIOSGetNvramStateBytes( 0x8000 ) ); // toolkit should see 0
    CPPUNIT_ASSERT_EQUAL( 0x0000,    SMBIOSGetNvramStateBytes( 0xF100 ) ); // other should see 0

    // test DSA mode (proper mask off)
    testValue = 0x9234; // we should not be able to set topmost bit
    SMBIOSSetNvramStateBytes( testValue, 0x0000 );
    CPPUNIT_ASSERT_EQUAL( (testValue & ~0x8000), SMBIOSGetNvramStateBytes( 0x0000 ) ); // DSA sees real value
    CPPUNIT_ASSERT_EQUAL( 0x0000,                SMBIOSGetNvramStateBytes( 0x8000 ) ); // toolkit should see 0
    CPPUNIT_ASSERT_EQUAL( 0x0000,                SMBIOSGetNvramStateBytes( 0xF100 ) ); // other should see 0

    // test Toolkit mode
    testValue = 0x0234; // we should not be able to set topmost bit
    SMBIOSSetNvramStateBytes( testValue, 0x8000 );
    CPPUNIT_ASSERT_EQUAL( testValue, SMBIOSGetNvramStateBytes( 0x8000 ) ); //toolkit sees real value
    CPPUNIT_ASSERT_EQUAL( 0x0000,    SMBIOSGetNvramStateBytes( 0x0000 ) ); // DSA should see 0
    CPPUNIT_ASSERT_EQUAL( 0x0000,    SMBIOSGetNvramStateBytes( 0xF100 ) ); // other should see 0

    // test Toolkit mode (proper mask off)
    testValue = 0x7234; // we should not be able to set topmost nibble (4 bits)
    SMBIOSSetNvramStateBytes( testValue, 0x8000 );
    CPPUNIT_ASSERT_EQUAL( (testValue & ~0xF000), SMBIOSGetNvramStateBytes( 0x8000 ) ); //toolkit sees real value
    CPPUNIT_ASSERT_EQUAL( 0x0000,                SMBIOSGetNvramStateBytes( 0x0000 ) ); // DSA should see 0
    CPPUNIT_ASSERT_EQUAL( 0x0000,                SMBIOSGetNvramStateBytes( 0xF100 ) ); // other should see 0

    // test other mode
    testValue = 0x0034; // we should not be able to set topmost byte
    SMBIOSSetNvramStateBytes( testValue, 0xF100 );
    CPPUNIT_ASSERT_EQUAL( testValue, SMBIOSGetNvramStateBytes( 0xF100 ) ); // other sees real value
    CPPUNIT_ASSERT_EQUAL( 0x0000,    SMBIOSGetNvramStateBytes( 0x0000 ) ); // DSA should see 0
    CPPUNIT_ASSERT_EQUAL( 0x0000,    SMBIOSGetNvramStateBytes( 0x8000 ) ); // DSA should see 0

    // test other mode (proper mask off)
    testValue = 0x7234; // we should not be able to set topmost byte
    SMBIOSSetNvramStateBytes( testValue, 0xF100 );
    CPPUNIT_ASSERT_EQUAL( (testValue & ~0xFF00), SMBIOSGetNvramStateBytes( 0xF100 ) ); // other sees real value
    CPPUNIT_ASSERT_EQUAL( 0x0000,                SMBIOSGetNvramStateBytes( 0x0000 ) ); // DSA should see 0
    CPPUNIT_ASSERT_EQUAL( 0x0000,                SMBIOSGetNvramStateBytes( 0x8000 ) ); // DSA should see 0

    STD_TEST_END();
}

void
testSmbiosXml::testUpBoot()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    if( ! SMBIOSHasBootToUp() )
        throw skip_test();

    SMBIOSSetBootToUp(1);
    CPPUNIT_ASSERT_EQUAL( 1, SMBIOSGetBootToUp() );

    SMBIOSSetBootToUp(0);
    CPPUNIT_ASSERT_EQUAL( 0, SMBIOSGetBootToUp() );

    SMBIOSSetBootToUp(1);
    CPPUNIT_ASSERT_EQUAL( 1, SMBIOSGetBootToUp() );

    SMBIOSSetBootToUp(0);
    CPPUNIT_ASSERT_EQUAL( 0, SMBIOSGetBootToUp() );

    STD_TEST_END();
}


void
testSmbiosXml::testOutOfBounds()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    smbios::ISmbiosTable::iterator item = (*table)["BIOS Information"] ;

    // access string '0' should always throw. (string offset start at 1, per
    // spec)
    ASSERT_THROWS( item->getStringByStringNumber(0), smbios::StringUnavailable );

    if (!doc)
        throw skip_test();

    int numStrings = 0;
    // pull info out of xml
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *smbios = xmlutils::findElement( doc->getDocumentElement(), "smbios", "", "" );
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *biosInfo = xmlutils::findElement( smbios, "biosInformation", "", "" );
        numStrings = strtoul( xmlutils::safeGetAttribute( biosInfo, "numstrings" ).c_str(), 0, 0 );
    }
    catch( const exception & )
    {
        throw skip_test();
    }

    // Should not throw (cast to void to eliminate unused var warn)
    if( numStrings > 0 )
        (void) (item->getStringByStringNumber(numStrings));

    ASSERT_THROWS( item->getStringByStringNumber(numStrings + 1), smbios::StringUnavailable );
    ASSERT_THROWS( item->getStringByStringNumber(numStrings + 2), smbios::StringUnavailable );

    STD_TEST_END()
}


void
testSmbiosXml::testConstructionOffset()
{
    STD_TEST_START("%s  ", getTestName().c_str() );

    if (!doc)
        throw skip_test();

    u32 offset = 0;
    // pull info out of xml
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMElement *smbios = xmlutils::findElement( doc->getDocumentElement(), "smbios", "", "" );
        offset = strtoul( xmlutils::safeGetAttribute( smbios, "offset" ).c_str(), 0, 0 );
        if( 0 == offset )
        {
            throw skip_test();
        }
    }
    catch( const exception & )
    {
        throw skip_test();
    }

    smbios::SmbiosFactory::getFactory()->setParameter("offset", offset);
    smbios::ISmbiosTable *table =
        smbios::SmbiosFactory::getFactory()->getSingleton();

    int tableEntriesCounted = 0;
    smbios::ISmbiosTable::iterator item = table->begin();
    while( item != table->end() )
    {
        tableEntriesCounted++;
        (void) *item;  // do this to sniff out possible errors with deref iterator.
        ++item;
    }

    CPPUNIT_ASSERT( tableEntriesCounted == table->getNumberOfEntries() );

    //
    // TEST BAD FILENAMES
    // construct our table with various invalid file names
    //
    smbios::SmbiosFactory *factory = smbios::SmbiosFactory::getFactory();
    //memory::MemoryFactory *memfactory = memory::MemoryFactory::getFactory();

    //memfactory->setParameter("memFile", "");
    //ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::Exception );

    // Not portable to Windows.
    //factory->setParameter("memFile", "/dev/null");
    //ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::ParseException );

    // Not portable to Windows.
    //factory->setParameter("memFile", "/dev/zero");
    //ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::ParseException );

    factory->reset();

    //
    // TEST BAD OFFSETS
    //
    // This is now a NON-XML factory.
    // This call nails the _instance variable to a regular non-xml factory
    // instance. Subsequent setup calls in setUp ensure it is properly set
    // with the correct paths and whatnot.
    tearDown();
    factory = smbios::SmbiosFactory::getFactory();
    setUp();

    // Ok, none of these are related to construction offset, but it is good to
    // run these tests while we have a handy reference to a NON-XML factory.
    auto_ptr<smbios::ISmbiosTable>p(factory->makeNew());
    auto_ptr<const smbios::ISmbiosTable>q(factory->makeNew());
    ASSERT_THROWS( (*p)["BIOS Information"], smbios::NotImplemented );
    ASSERT_THROWS( (*q)["BIOS Information"], smbios::NotImplemented );

    smbios::ISmbiosTable::iterator item1 = (*p)[smbios::BIOS_Information];
    smbios::ISmbiosTable::const_iterator item2 = (*q)[smbios::BIOS_Information];

    // test streamify() for non-XML SmbiosItem class while we are here.
    ostringstream ost;
    ost << *item1 << endl;

    CPPUNIT_ASSERT_EQUAL( item1->getHandle(), item2->getHandle() );
    CPPUNIT_ASSERT_EQUAL( item1->getHandle(), item2->getU16(2) );
    CPPUNIT_ASSERT_EQUAL( item1->getLength(), item2->getLength() );
    CPPUNIT_ASSERT_EQUAL( item1->getLength(), item2->getU8(1) );
    CPPUNIT_ASSERT_EQUAL( item1->getType()  , item2->getType()   );
    CPPUNIT_ASSERT_EQUAL( item1->getType()  , item2->getU8(0)   );

    ASSERT_THROWS( item1->getU8("BIOS Version"), smbios::NotImplemented);
    ASSERT_THROWS( item1->getU16("BIOS Version"), smbios::NotImplemented);
    ASSERT_THROWS( item1->getU32("BIOS Version"), smbios::NotImplemented);
    ASSERT_THROWS( item1->getU64("BIOS Version"), smbios::NotImplemented);
    ASSERT_THROWS( item1->getString("BIOS Version"), smbios::NotImplemented);
    ASSERT_THROWS( item1->getBitfield("BIOS Version", "test bitfield"), smbios::NotImplemented);
    // End unrelated tests.

    // use auto_ptr so in case it does _not_ throw, we don't leak mem in the test.
    //  DO NOT USE ->getSingleton() here... we use ->makeNew() on purpose.
    //  This is an internal test and the parameters of this test mean we should _not_ use a singleton.
    factory->setParameter("offset", 1);
    ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::IException );

    factory->setParameter("offset", 1000);
    ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::IException );

    factory->setParameter("offset", 0xFFFFFUL ); // F_BLOCK_END (private definition)
    ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::IException );

    factory->setParameter("offset", 0xFFFFFUL - 1); // F_BLOCK_END (private definition)
    ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable>p(factory->makeNew()), smbios::IException );

    smbios::SmbiosFactory::getFactory()->setParameter("offset", offset + 1);
    ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable> table( factory->makeNew() ), smbios::IException );

    smbios::SmbiosFactory::getFactory()->setParameter("offset", offset - 1);
    ASSERT_THROWS( auto_ptr<smbios::ISmbiosTable> table( factory->makeNew() ), smbios::IException );

    // should not be able to use no-argument constructor...
    // UNCOMMENT TO CHECK.
    // THIS TEST DOES NOT COMPILE. IT SHOULD NOT COMPILE
    // DUE TO THE DEFINITION OF SmbiosTable.
    //ASSERT_THROWS( smbios::SmbiosTableFileIo myTable1, smbios::PermissionException);

    STD_TEST_END()
}


void
testSmbiosXml::testException()
{
    STD_TEST_START("%s  ", getTestName().c_str() );
    std::string actual = "";
    smbios::Exception<smbios::IException> foo;
    string source = "";
    string expected = "";
    foo.setParameter("foo", "happy");
    foo.setParameter("bar", 42);
    foo.setParameter("recursive", "%(foo)s");
    foo.setParameter("recursiverecursive", "%(recursive)s");

    source = "The %% cat is %(foo)s. The best number is %(bar)i. %";
    expected = "The % cat is happy. The best number is 42. %";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    // test copy constructor
    smbios::Exception<smbios::IException> bar = foo;
    actual = bar.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = "The %% cat is %(recursive)s. The best number is %(bar)i. %";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = "The %% cat is %(recursiverecursive)s. The best number is %(bar)i. %";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = "The %% cat %is %(recursiverecursive)s. The best number is %(bar)i. %";
    expected = "The % cat %is happy. The best number is 42. %";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = " %(a_really_long_variable_longer_than_32_characters)s";
    expected = " %(a_really_long_variable_longer_than_32_characters)s";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = " %(no_closing_paren";
    expected = " %(no_closing_paren";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = " %(a_var_with_no_type)";
    expected = " %(a_var_with_no_type)";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = " %(a_var_with_no_type)  ";
    expected = " %(a_var_with_no_type)  ";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    source = " %";
    expected = " %";
    foo.setMessageString( source );
    actual = foo.what();
    CPPUNIT_ASSERT_EQUAL( expected, actual );

    STD_TEST_END()
}

