/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim:expandtab:autoindent:tabstop=4:shiftwidth=4:filetype=c:cindent:textwidth=0:
 *
 * Copyright (C) 2005 Dell Inc.
 *  by Michael Brown <Michael_E_Brown@dell.com>
 * Licensed under the Open Software License version 2.1
 *
 * Alternatively, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 */

// compat header should always be first header if including system headers
#include "smbios/compat.h"

#include <iostream>
#include <sstream>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <unistd.h>

#include "RbuImpl.h"
#include "smbios/IToken.h"
#include "smbios/SystemInfo.h"

// always include last if included.
#include "smbios/message.h"  // not needed outside of this lib. (mainly for gettext i18n)

using namespace std;

namespace rbu
{
const char *rbu_v1_mono_data_file = "/sys/firmware/rbu/rbudata";
const char *rbu_v1_mono_size_file = "/sys/firmware/rbu/rbudatasize";
const char *rbu_v1_pkt_data_file = "/sys/firmware/rbu/packetdata";
const char *rbu_v1_pkt_size_file = "/sys/firmware/rbu/packetdatasize";

const char *rbu_v2_fw_data_file = "/sys/class/firmware/dell_rbu/data";
const char *rbu_v2_fw_load_file = "/sys/class/firmware/dell_rbu/loading";
const char *rbu_v2_drv_data_file = "";
const char *rbu_v2_img_type_file = "/sys/devices/platform/dell_rbu/image_type";
const char *rbu_v2_pkt_size_file = "/sys/devices/platform/dell_rbu/packet_size";

    packet_type getSupportedPacketType(void)
    {
        packet_type pt = pt_mono;
        try
        {
            smbios::SmbiosFactory *smbiosFactory = smbios::SmbiosFactory::getFactory();
            smbios::ISmbiosTable *table = smbiosFactory->getSingleton();
            smbios::ISmbiosItem &rbuStructure = *((*table)[RBU_SMBIOS_STRUCT]);
    
            u8 byte = rbuStructure.getU8(0x0F); // Characteristics field
            if( byte & 0x01 ) 
                pt = pt_packet;
        }
        catch(const smbios::DataOutOfBounds &)
        {
            // only happens when old-style struct present w/o characteristics field
            // this means only mono supported.
        }
        return pt;
    }
        
    void activateRbuToken()
    {
        smbios::TokenTableFactory *ttFactory = smbios::TokenTableFactory::getFactory() ;
        smbios::ITokenTable *tokenTable = ttFactory->getSingleton();
        (*tokenTable)[ RBU_ACTIVATE ]->activate();
    }
    
    void cancelRbuToken()
    {
        smbios::TokenTableFactory *ttFactory = smbios::TokenTableFactory::getFactory() ;
        smbios::ITokenTable *tokenTable = ttFactory->getSingleton();
        (*tokenTable)[ RBU_CANCEL ]->activate();
    }

    void checksumPacket(rbu_packet *pkt, size_t size)
    {
        u16 *buf = reinterpret_cast<u16 *>(pkt);
        pkt->pktChksum = 0;
    
        u16 csum = 0;
        for(size_t i=0; i<size/2; ++i)
            csum = csum + buf[i];
    
        pkt->pktChksum = -csum;
    }

    driver_type getDriverType()
    {
        if (!access(rbu_v1_mono_data_file, F_OK))
            return rbu_linux_v1;
        else if (!access(rbu_v2_fw_data_file, F_OK))
            return rbu_linux_v2;
        else
            return rbu_unsupported;
    }

    void createPacket(char *buffer, size_t bufSize, size_t imageSize)
    {
        // set up packet
        rbu_packet *pkt = reinterpret_cast<rbu_packet *>(buffer);
    
        pkt->pktId = 0x4B505224;  //2452504B;   // must be '$RPK'
        pkt->pktSize = bufSize / 1024;    // size of packet in KB
        pkt->reserved1 = 0;  //
        pkt->hdrSize = 2;    // size of packet header in paragraphs (16 byte chunks)
        pkt->reserved2 = 0;  //
        pkt->pktSetId = 0x12345678;   // unique id for packet set, can be anything
        pkt->pktNum = 0;     // sequential pkt number (only thing that changes)
        pkt->totPkts = (imageSize/bufSize) + ((imageSize % bufSize) ? 1:0) + 1;// total number of packets
        pkt->pktVer = 1;     // version == 1 for now
        pkt->pktChksum = 0;  // sum all bytes in pkt must be zero
    
        checksumPacket(pkt, bufSize);
    }

    static void writePacket(const char *fn, const char *buffer, size_t bufSize, bool openclose)
    {
        static FILE *data_fh = 0;
        if(!data_fh)
            data_fh = fopen(fn, "wb");

        if (!data_fh)
            throw RbuDriverIOErrorImpl(strerror(errno));
    
        try
        {
            fwrite(buffer, 1, bufSize, data_fh);
            if (ferror(data_fh))
                throw RbuDriverIOErrorImpl(strerror(errno));
    
            if(openclose)
            {
                fclose(data_fh);
                data_fh = 0;
            }
        }
        catch(...)
        {
            fclose(data_fh);
            throw;
        }

        fflush(NULL);
    }

    static void pktUpdateLoop(FILE *hdr_fh, const char *packetFilename, char *buffer, size_t bufSize, bool openclose)
    {
        cout << "Writing RBU data (4096k/dot): ";

        fseek(hdr_fh, 0, SEEK_END);
        size_t totalSizeBytes = ftell(hdr_fh);

        fseek(hdr_fh, 0, 0);
        // set up packet
        rbu_packet *pkt = reinterpret_cast<rbu_packet *>(buffer);
        createPacket(buffer, bufSize, totalSizeBytes);
    
        // TODO: password support.
        writePacket(packetFilename, buffer, bufSize, openclose);
        cout << ".";
    
        while(!feof(hdr_fh))
        {
            ++pkt->pktNum;
            memset(pkt->pktData, 0, bufSize - sizeof(rbu_packet));
            fread(pkt->pktData, 1, bufSize - sizeof(rbu_packet), hdr_fh);
            if (ferror(hdr_fh))
                throw HdrFileIOErrorImpl(strerror(errno));
    
            checksumPacket(pkt, bufSize);
            writePacket(packetFilename, buffer, bufSize, openclose);
    
            cout << ".";
        }
        cout << endl;
    
        cout << "Done writing packet data." << endl;
    }

    static void monoUpdateLoop(FILE *hdr_fh, FILE *data_fh)
    {
        cout << "Writing RBU data (4096k/dot): ";
        fseek(hdr_fh, 0, 0);
        const int bufSize = 4096;
        char *buffer[bufSize];
        while(!feof(hdr_fh))
        {
            memset(buffer, 0, bufSize);
            size_t readSz = fread(buffer, 1, bufSize, hdr_fh);
            if (ferror(hdr_fh))
                throw HdrFileIOErrorImpl(strerror(errno));
    
            fwrite(buffer, 1, readSz, data_fh);
            if (ferror(data_fh))
                throw RbuDriverIOErrorImpl(strerror(errno));
            cout << "." << flush;
        }
        cout << endl;
    }


/****************************************
   RBU Linux v1 functions (older driver)
****************************************/
    static void setSize(const char *fn, size_t sz)
    {
        FILE *size_fh = fopen(fn, "wb");
        if (!size_fh)
            throw RbuDriverIOErrorImpl(strerror(errno));

        ostringstream ost("");
        ost << sz;
        cout << "writing (" << sz << ") to file: " << fn << endl;
        fwrite(ost.str().c_str(), 1, ost.str().length(), size_fh);
        if (ferror(size_fh))
            throw RbuDriverIOErrorImpl(strerror(errno));
        fclose(size_fh);
        size_fh = 0;
    }

    static void doPacketUpdate_v1(FILE *hdr_fh) 
    {
        const size_t bufSize = 4096;
        char buffer[bufSize] = {0};
    
        // set packet size, reset mono handler
        setSize(rbu_v1_mono_size_file, 0);
        setSize(rbu_v1_pkt_size_file, bufSize);

        pktUpdateLoop(hdr_fh, rbu_v1_pkt_data_file, buffer, bufSize, true);
    }
    
    static void doMonoUpdate_v1(FILE *hdr_fh) 
    {
        cout << "Prep driver for data load." << endl;
    
        FILE *data_fh = fopen(rbu_v1_mono_data_file, "wb");
        if (!data_fh)
            throw RbuDriverIOErrorImpl(strerror(errno));
    
        fseek(hdr_fh, 0, SEEK_END);
        size_t totalSizeBytes = ftell(hdr_fh);

        // set mono data size, reset pkt handler
        setSize(rbu_v1_pkt_size_file, 0);
        setSize(rbu_v1_mono_size_file, totalSizeBytes);

        monoUpdateLoop(hdr_fh, data_fh);
        
        fclose(data_fh);
        data_fh = 0;
        fflush(NULL);
    
        cout << "BIOS staging is complete." << endl;
    }


/****************************************
   RBU Linux v2 functions (newer, included in 2.6.14+)
****************************************/

    static void setPacketType(packet_type type)
    {
        FILE *type_fh = 0;
        type_fh = fopen(rbu_v2_img_type_file, "wb");
        if (!type_fh)
            throw RbuDriverIOErrorImpl(strerror(errno));
    
        switch(type)
        {
        case pt_mono:
            fwrite("mono\0", 1, 5, type_fh);
            break;
        case pt_packet:
            fwrite("packet\0", 1, 7, type_fh);
            break;
        case pt_any:  /*fall thru*/
        case pt_init:  /*fall thru*/
        default:
            // not really a packet type, but causes driver to free its memory
            fwrite("init\0", 1, 7, type_fh);
            break;
        }
    
        if (ferror(type_fh))
            throw RbuDriverIOErrorImpl(strerror(errno));

        fclose(type_fh);
    }        // Step 5: set rbu cmos token
    
    
    static void waitForFile(const char *fn, time_t wait)
    {
        time_t start = time(NULL);
        while( access(fn, F_OK) && (time(NULL) - start < wait))
            /*nothing*/;
    }
    
    static void setLoadValue(char val)
    {
        FILE *load_fh = 0;
    
        waitForFile(rbu_v2_fw_load_file, 10);
    
        load_fh = fopen(rbu_v2_fw_load_file, "wb");
        if (!load_fh)
            throw RbuDriverIOErrorImpl(strerror(errno));
    
        fwrite(&val, 1, 1, load_fh);
        if (ferror(load_fh))
            throw RbuDriverIOErrorImpl(strerror(errno));
        fclose(load_fh);
        fflush(NULL);
    }
    
    static void doPacketUpdate_v2(FILE *hdr_fh) 
    {
        const size_t bufSize = 4096;
        char buffer[bufSize] = {0};

        setSize(rbu_v2_pkt_size_file, bufSize);
        setLoadValue('1');
        pktUpdateLoop(hdr_fh, rbu_v2_fw_data_file, buffer, bufSize, false);
        setLoadValue('0');
    }
    
    static void doMonoUpdate_v2(FILE *hdr_fh) 
    {
        cout << "Prep driver for data load." << endl;
        setLoadValue('1');
    
        FILE *data_fh = fopen(rbu_v2_fw_data_file, "wb");
        if (!data_fh)
            throw RbuDriverIOErrorImpl(strerror(errno));
    
        monoUpdateLoop(hdr_fh, data_fh);
       
        fclose(data_fh);
        data_fh = 0;
        fflush(NULL);
    
        cout << "Notify driver data is finished." << endl;
        setLoadValue('0');
    }



/*****************************************************************************
******************************************************************************

main entry points for this module.

******************************************************************************
*****************************************************************************/
    
    void dellBiosUpdate(string fileName, packet_type force_type)
    {
        FILE *hdr_fh = fopen( fileName.c_str(), "rb" );
        if (!hdr_fh)
            throw InvalidHdrFileImpl(strerror(errno));
        
        // TODO: verify that it is a HDR file
        // TODO: checksum HDR file

        packet_type pt = getSupportedPacketType();
        cout << "Supported RBU type for this system: "
            << (pt == pt_packet ? "MONOLITHIC, PACKET" : "MONOLITHIC")
            << endl;
    
        if( force_type != pt_init || force_type != pt_any ) 
            pt = force_type;
    
        driver_type dt = getDriverType();

        if (pt == pt_packet)
        {
            cout << endl;
            cout << "WARNING:" << endl;
            cout << "        Packet support not yet complete/tested." << endl;
            cout << "        please report ID information for this system so" << endl;
            cout << "        that packet support can be tested." << endl;
            cout << endl;
        }

        if (dt == rbu_linux_v2)
        {
            // initialize RBU driver.
            cout << "Using RBU v2 driver. Initializing Driver. ";
            setPacketType(pt_init);
        
            // set packet/mono type
            cout << "Setting RBU type in v2 driver to: " 
                << (pt == pt_packet ? "PACKET" : "MONOLITHIC")
                << (pt == force_type ? " (FORCED) ": "" )
                << endl;
            setPacketType(pt);
        
            if (pt == pt_packet)
                doPacketUpdate_v2(hdr_fh);
            else
                doMonoUpdate_v2(hdr_fh);
        } 
        else if(dt == rbu_linux_v1)
        {
            cout << "Using RBU v1 method: " 
                << (pt == pt_packet ? "PACKET" : "MONOLITHIC")
                << (pt == force_type ? " (FORCED) ": "" )
                << endl;
            if (pt == pt_packet)
                doPacketUpdate_v1(hdr_fh);
            else
                doMonoUpdate_v1(hdr_fh);
        }
        else
        {
            throw RbuNotSupportedImpl("Could not open Dell RBU driver.");
        }
    
        fclose(hdr_fh);
    
        cout << "Activate CMOS bit to notify BIOS that update is ready on next boot." << endl;
        activateRbuToken();
    
        cout << "Update staged sucessfully. BIOS update will occur on next reboot." << endl;
    }
    
    void cancelDellBiosUpdate()
    {
        // FOR LOAD CANCEL:
        // Step 1: always unset CMOS first
        cout << "Cancel BIOS CMOS notification bit." << endl;
        cancelRbuToken();

        driver_type dt = getDriverType();
        switch(dt)
        {
        case rbu_linux_v2:
            // Step 2: make sure firmware class doesn't think we are loading
            cout << "Free kernel driver memory." << endl;
            setLoadValue('0');

            // Step 3: tell the dell_rbu driver to free its allocated memory
            cout << "Re-initialize driver for next user." << endl;
            setPacketType(pt_init);
            break;

        case rbu_linux_v1:
            // Step 2: Free monolithic, if present
            cout << "Re-initialize driver for next user." << endl;
            setSize(rbu_v1_mono_size_file, 0);
            setSize(rbu_v1_pkt_size_file, 0);
            fflush(NULL);
            break;

        default:
            cout << "Could not determine RBU driver present, skipping." << endl;
            break;
        }
    }
}

