/* libspf - Sender Policy Framework library
*
*  ANSI C implementation of spf-draft-200405.txt
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*  Author: Sean Comeau <scomeau@obscurity.org>
*
*  File:   dns.c
*  Desc:   DNS related functions
*
*  License:
*
*  The libspf Software License, Version 1.0
*
*  Copyright (c) 2004 James Couzens & Sean Comeau  All rights
*  reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*  1. Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*
*  2. Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
*  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
*  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS MAKING USE OF THIS LICESEN
*  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
*  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
*  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
*  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
*  SUCH DAMAGE.
*
*/

#include "../../config.h" /* autoconf */
#include "spf.h"          /* SPF */
#include "util.h"         /* Utility functions */
#include "dns.h"          /* our header */


/*! \file dns.c
*  \brief DNS Functions.
*
*  dns.c contains functions relating to the execution and processing of
*  DNS queries.  Each particular DNS record type has a unique function
*  that is designed to process such a response.
*/

/*! \fn char *DNS_query(peer_info_t *peer_info, const char *s, const int T_TYPE,
const char *mta)
*  \param peer_info Global information structure containing client information
*  \param s Domain name with which to perform the query against
*  \param T_TYPE Type of DNS record to lookup (eg: T_TXT, T_PTR, T_A, etc..)
*  \param mta Used by DNS_ptr_answer in validation of an SPF ptr mechanism
*  \brief Execute a DNS query.
*
* Author: James Couzens <jcouzens@codeshare.ca>\n
* Author: Travis Anderson <travis@anthrax.ca>\n
*
* Date:   12/10/03\n
* Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>\n
*
* Desc:
*         Executes a DNS query of type T_TYPE and then calls the
* appropriate answer parsing function based on that type.  Returns
* a pointer to allocated memory (a string of space delimited
* records).  Upon failure returns NULL.
*
*/

/*! \fn char *DNS_txt_answer (int16_t ancount, const u_char *msg_ptr, const u_char *eom_ptr, u_char *rd_ptr, char *buf, int *ttl)
*  \param ancount Answer count
*  \param msg_ptr DNS message
*  \param eom_ptr Pointer to the end of the DNS message
*  \param rd_ptr  Pointer to a position in the DNS message
*  \param buf     Utiltiy buffer
*  \param ttl     Time To Live value of the DNS message
*  \brief Process a DNS message of type T_TXT
*
*  Author: James Couzens <jcouzens@codeshare.ca>\n
*
*  Date:   01/02/04\n
*  Date:   02/23/04 - Bugfix from Albert Weichselbraun <albert@atnet.at>\n
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>\n
*
*  Desc:
*          SPF_PACKETSZ bytes are allocated and then filled with \0 chars.
*  This buffer is then used in a TXT DNS query using data from the passed
*  peer_info_t structure.  Upon success this buffer is re-cast as a char *
*  and then a pointer to this memory is returned.  Upon failure a NULL
*  pointer is returned.
*
*/

/*! \fn char *DNS_mx_answer(int16_t ancount, const u_char *msg_ptr, const u_char *eom_ptr, u_char *rd_ptr, char *buf, int *ttl)
*  \param ancount Answer count
*  \param msg_ptr DNS message
*  \param eom_ptr Pointer to the end of the DNS message
*  \param rd_ptr  Pointer to a position in the DNS message
*  \param buf     Utiltiy buffer
*  \param ttl     Time To Live value of the DNS message
*  \brief Process a DNS message of type T_MX
*
*  Author: James Couzens <jcouzens@codeshare.ca>\n
*
*  Date:   01/02/04\n
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>\n
*
*  Desc:
*          SPF_PACKETSZ bytes are allocated and then filled with \0 chars.
*  This buffer is then used in an MX DNS query using data from the passed
*  peer_info_t structure.  Upon success this buffer is re-cast as a char *
*  and then a pointer to this memory is returned.  Upon failure a NULL
*  pointer is returned.
*
*/

/*! \fn SPF_BOOL DNS_ptr_answer(peer_info_t *peer_info, int16_t ancount, const u_char *msg_ptr, const u_char *eom_ptr, u_char *rd_ptr, char *buf, const char *mta, int *ttl)
*  \param peer_info Global information structure containing client information
*  \param ancount Answer count
*  \param msg_ptr DNS message
*  \param eom_ptr Pointer to the end of the DNS message
*  \param rd_ptr  Pointer to a position in the DNS message
*  \param buf     Utiltiy buffer
*  \param mta     IP address of the authoritative MX
*  \param ttl     Time To Live value of the DNS message
*  \brief Process a DNS message of type T_PTR
*
*  Author: James Couzens <jcouzens@codeshare.ca>\n
*
*  Date:   01/02/04\n
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>\n
*
*  Desc:
*          A reverse lookup on an IP address leads to a lookup per returned
*  PTR answer to see if the returned answer matches.  The forward lookups are
*  handled by a separate function which calls gethostbyname.  Upon a single
*  successful match of a forward lookup with a reverse lookup, returns TRUE.
*  Returns FALSE upon failure.
*
*/

/*! \fn char *DNS_cname_answer(int16_t ancount, const u_char *msg_ptr, const u_char *eom_ptr, u_char *rd_ptr, char *buf, int *ttl)
*  \param ancount Answer count
*  \param msg_ptr DNS message
*  \param eom_ptr Pointer to the end of the DNS message
*  \param rd_ptr  Pointer to a position in the DNS message
*  \param buf     Utiltiy buffer
*  \param ttl     Time To Live value of the DNS message
*  \brief Process a DNS message of type T_MX
*
*  Author: Teddy <teddy@teddy.ch>\n
*
*  Date:   29/04/04\n
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>\n
*
*  Desc:
*          SPF_PACKETSZ bytes are allocated and then filled with \0 chars.
*  This buffer is then used in a TXT DNS query using data from the passed
*  peer_info_t structure.  Upon success this buffer is re-cast as a char *
*  and then a pointer to this memory is returned.  Upon failure a NULL
*  pointer is returned.
*
*/

/* DNS_query
*
* Author: James Couzens <jcouzens@codeshare.ca>\n
* Author: Travis Anderson <travis@anthrax.ca>\n
*
* Date:   12/10/03
* Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>
*
* Desc:
*         Executes a DNS query of type T_TYPE and then calls the
* appropriate answer parsing function based on that type.  Returns
* a pointer to allocated memory (a string of space delimited
* records).  Upon failure returns NULL.
*
*/
char *DNS_query(peer_info_t *peer_info, const char *s, const int T_TYPE,
  const char *mta)
{
  int16_t         r_len;                 /* res_search return code & packet len */
  int16_t         rc;                    /* generic return code / length of */

  u_char          *msg_ptr;              /* pointer to beginning of the message */
  u_char          *eom_ptr;              /* pointer to the end of the message */

  u_char          *rd_ptr;               /* pointer to uncompressed message */

  int8_t          ancount;               /* number of answers */

  HEADER          *hdr;                  /* pointer to the header of the packet */

  char            buf[SPF_MAXDNAME];     /* record extraction buffer */
  char            answer[SPF_PACKETSZ];  /* answer buffer */

  char            *rr_data = NULL;       /* record */

  int             ttl;                   /* answer TTL */

  if (s == NULL)
  {
    xeprintf("Passed a NULL char.  Aborting.\n");
    return(NULL);
  }

  xvprintf("Called with (%s) and type: %i\n", s, T_TYPE);

  memset(answer, '\0', SPF_PACKETSZ);
  memset(buf, '\0', SPF_MAXDNAME);
  r_len = res_search(s, C_IN, T_TYPE, (u_char *)answer, SPF_PACKETSZ);

  if (r_len <= 0)
  {
    switch (h_errno)
    {
      case HOST_NOT_FOUND:
        snprintf(peer_info->error, MAX_ERROR, "DNS failure (Host not found)");
        UTIL_assoc_prefix(peer_info, SPF_NONE, NULL);
        xvprintf("%s", peer_info->error);

        return(NULL);

      case TRY_AGAIN:
        snprintf(peer_info->error, MAX_ERROR, "DNS failure " \
          "(Non-Authoritative Host not found, or SERFAIL.)");
        UTIL_assoc_prefix(peer_info, SPF_NONE, NULL);
        xvprintf("%s", peer_info->error);

        return(NULL);

      case NO_RECOVERY:
        snprintf(peer_info->error, MAX_ERROR, "DNS failure (Non " \
          "recoverable errors, FORMERR, REFUSED, NOTIMP.)");
        UTIL_assoc_prefix(peer_info, SPF_ERROR, NULL);
        xvprintf("%s", peer_info->error);

        return(NULL);

      case NO_DATA:
        snprintf(peer_info->error, MAX_ERROR, "DNS failure (Valid name, " \
          " no data record of requested type.)");
        UTIL_assoc_prefix(peer_info, SPF_NONE, NULL);
        xvprintf("%s", peer_info->error);

        return(NULL);

       default:
        snprintf(peer_info->error, MAX_ERROR, "DNS failure (internal error)");
        UTIL_assoc_prefix(peer_info, SPF_ERROR, NULL);
        xvprintf("%s", peer_info->error);
        return(NULL);
    } /* switch */
  }

  hdr     = (HEADER *)&answer;
  ancount = ntohs(hdr->ancount);

  xvprintf("Received packet size of %i bytes which contains %i answers.\n",
    r_len, ancount);
  xvprintf("ANSWERS: %i\n", ancount);
  xvprintf("QUESTIONS: %i\n", ntohs(hdr->qdcount));

  if (ancount > 0)
  {
    msg_ptr = (u_char *)&answer;             /* point to start of message */
    eom_ptr = (u_char *)&answer + r_len;     /* point to end of message */
    rd_ptr  = (u_char *)&answer + HFIXEDSZ;  /* point to start of RDATA */

    if ((rc = dn_skipname(rd_ptr, eom_ptr)) < 0)
    {
      xeprintf("Error obtaining QUESTION!\n");
      return(NULL);
    }

  rd_ptr += rc + QFIXEDSZ;                   /* jump to start of ANSWER */

  switch (T_TYPE)
  {
    case T_A:
      return((char *)TRUE);

    case T_TXT:
      if ((rr_data = DNS_txt_answer(ancount, (u_char *)msg_ptr, (u_char *)eom_ptr,
        (u_char *)rd_ptr, buf, &ttl)) == NULL)
      {

        return(NULL);
      }
      break;

    case T_MX:
      if ((rr_data = DNS_mx_answer(ancount, (u_char *)msg_ptr, (u_char *)eom_ptr,
        (u_char *)rd_ptr, buf, &ttl)) == NULL)
      {

        return(NULL);
      }
      break;

    case T_PTR:

  /* Comment: James Couzens <jcouzens@codeshare.ca>
  *  Date: 01/04/04
  *
  *  DNS_ptr_answer doesn't allocate any memory and simply returns TRUE or FALSE,
  *  however, this function returns a char so we return a cast TRUE or FALSE
  *  depending on the outcome.
  *
  */
      if (DNS_ptr_answer(peer_info, ancount, (u_char *)msg_ptr, (u_char *)eom_ptr,
        (u_char *)rd_ptr, buf, mta, &ttl) == FALSE)
      {
        return((char *)FALSE);
      }
      else
      {
        return((char *)TRUE);
      }
      break;
    case T_CNAME:
      if ((rr_data = DNS_cname_answer(ancount, (u_char *)msg_ptr, (u_char *)eom_ptr,
        (u_char *)rd_ptr, buf, &ttl)) == NULL)
      {

        return(NULL);
      }
      break;
  }

    return(rr_data);
  }

  return(NULL);
}


/* DNS_txt_answer
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/02/04
*  Date:   02/23/04 - Bugfix from Albert Weichselbraun <albert@atnet.at>
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>
*
*  Desc:
*          SPF_PACKETSZ bytes are allocated and then filled with \0 chars.
*  This buffer is then used in a TXT DNS query using data from the passed
*  peer_info_t structure.  Upon success this buffer is re-cast as a char *
*  and then a pointer to this memory is returned.  Upon failure a NULL
*  pointer is returned.
*
*/
char *DNS_txt_answer(int16_t ancount, const u_char *msg_ptr,
  const u_char *eom_ptr, u_char *rd_ptr, char *buf, int *ttl)
{
  int16_t   i;                /* utility */
  int16_t   j;                /* utility */
  int16_t   rc;               /* generic return code / length of */
  int16_t   rd_type;          /* answer type */
  int16_t   rd_len;           /* res_search return code & packet len */
  int32_t   rd_ttl;           /* TTL */

  char      *rr_data = NULL;  /* data pointer */
  char      *r_buf   = NULL;  /* return buffer */
  char      *pos     = NULL;  /* utility pointer */

  if (msg_ptr == NULL || eom_ptr == NULL || rd_ptr == NULL || buf == NULL)
  {
    xeprintf("Called with NULL pointers\n");
    return(NULL);
  }

  i = 0;
  j = ancount;
  while (ancount > 0 && rd_ptr < eom_ptr)
  {
    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(NULL);
    }

    /* dname, type, class, TTL, rdata len, rdata */
    rd_ptr += rc;               /* jump to start of ANSWER data */
    GETSHORT(rd_type, rd_ptr);  /* get response type */
    rd_ptr += INT16SZ;          /* skip class */
    /*rd_ptr += INT32SZ;*/      /* skip TTL */
    GETLONG(rd_ttl, rd_ptr);    /* get TTL */
    GETSHORT(rd_len, rd_ptr);   /* get data length */

    *ttl = rd_ttl;              /* TTL working pointer */

    if (rd_type != T_TXT)
    {
      xvprintf("Ignoring record not of T_TXT type. (%i)\n", rd_type);
      rd_ptr += rd_len;
      continue;
    }

    rd_ptr++;                  /* skip  */
    rd_ptr[rd_len] = '\0';
    rd_ptr[rd_len - 1] = ' ';
    i += rd_len;

    xvprintf("Answer %i has length %i. (%i)\n", ancount, rd_len, i);

    /* Only received one answer, so this MUST start with v=spf1 or its
     * not a valid SPFv1 record
    */
    if (j == 1 && (*rd_ptr != 'v' && *(rd_ptr + 1) != '='))
    {
      xvprintf("INVALID Answer Data: (%s) len: %i\n", rd_ptr, rd_len);
      return(NULL);
    }

    xvprintf("Answer Data: (%s) len: %i\n", rd_ptr, rd_len);

    if (rd_len <= SPF_MAXDNAME)
    {
      if (rr_data == NULL)
      {
        rr_data = xmalloc(i + 1);
        memset(rr_data, '\0', (i + 1));
      }
      else
      {
        rr_data = xrealloc(rr_data, (i + 1));
      }

      xvprintf("REALLOCATE memory: %i bytes\n", i);

      strncat(rr_data, (char *)rd_ptr, rd_len);
      rr_data[i - 1] = ' ';
      rr_data[i] = '\0';
    }

    rd_ptr += rd_len;
    ancount--;
  }

  if (rr_data == NULL)
  {
    return(NULL);
  }

  rr_data[i] = '\0';

  xvprintf("RR_DATA: (%s)\n", rr_data);

  if (*rr_data == 'v' && *(rr_data + 1) == '=' && *(rr_data + 2) == 's' &&
    *(rr_data + 3) == 'p' && *(rr_data + 4) == 'f' && *(rr_data + 5) == '1')
  {
    xvprintf("Returning with valid SPFv1 record\n", rr_data);
    return(rr_data);
  }
  else
  {
    /* search the buffer for a valid SPFv1 version mechanism */
    if ((pos = strstr(rr_data, "v=spf1")) != NULL)
    {
      xvprintf("Found SPFv1 version mechanism: (%s)\n", pos);
      r_buf = xstrndup(pos, strlen(pos) + 1);
      r_buf[strlen(pos)] = '\0';
      xvprintf("Old buffer: (%s)\n", rr_data);
      xvprintf("New buffer: (%s)\n", r_buf);
      xfree(rr_data);
      pos = NULL;
    }
    /* no valid SPFv1 record here */
    return(r_buf);
  }

  xeprintf("Returning NULL\n");
  return(NULL);
}


/* DNS_mx_answer
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/02/04
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>
*
*  Desc:
*          SPF_PACKETSZ bytes are allocated and then filled with \0 chars.
*  This buffer is then used in an MX DNS query using data from the passed
*  peer_info_t structure.  Upon success this buffer is re-cast as a char *
*  and then a pointer to this memory is returned.  Upon failure a NULL
*  pointer is returned.
*
*/
char *DNS_mx_answer(int16_t ancount, const u_char *msg_ptr,
  const u_char *eom_ptr, u_char *rd_ptr, char *buf, int *ttl)
{
  int16_t   i;                /* utility */
  int16_t   rc;               /* generic return code / length of */
  int16_t   rd_len = 0;       /* res_search return code & packet len */
  int16_t   rd_pref;          /* MX preference */
  int16_t   rd_type;          /* answer type */
  int32_t   rd_ttl;           /* TTL */
  size_t    buf_len;          /* buffer length */

  char      *rr_data = NULL;  /* data pointer */

  i = 0;
  while ((ancount > 0) && (rd_ptr < eom_ptr))
  {
    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(NULL);
    }

    /* dname, type, class, TTL, rdata len, rdata */
    rd_ptr += rc;               /* jump to start of ANSWER data */
    GETSHORT(rd_type, rd_ptr);  /* get response type */
    rd_ptr += INT16SZ;          /* skip class */
    /*rd_ptr += INT32SZ;*/      /* skip TTL */
    GETLONG(rd_ttl, rd_ptr);    /* get TTL */
    GETSHORT(rd_len, rd_ptr);   /* get data length */

    *ttl = rd_ttl;

    if (rd_type != T_MX)
    {
      xprintf("Forged packet?!  We requested T_MX (15) but got %i\n", rd_type);
      rd_ptr += rd_len;
      continue;
    }

    GETSHORT(rd_pref, rd_ptr);  /* get MX preference */

    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(NULL);
    }

    xvprintf("MX: %s Preference: %i\n", buf, rd_pref);

    buf_len = strlen(buf);
    i += (buf_len + 1);

    if (rd_len <= SPF_MAXDNAME)
    {
      if (rr_data == NULL)
      {
        rr_data = xmalloc(i + 1);
        memset(rr_data, '\0', (i + 1));
      }
      else
      {
        rr_data = xrealloc(rr_data, (i + 1));
      }

      xvprintf("REALLOCATE memory: %i bytes\n", (i + 1));

      strncat(rr_data, buf, buf_len);
      rr_data[i - 1] = ' ';
      rr_data[i] = '\0';
    }

    rd_ptr += rc;
    ancount--;
  }

  rr_data[i - 1] = '\0';
  return(rr_data);
}


/* DNS_ptr_answer
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/02/04
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>
*
*  Desc:
*          A reverse lookup on an IP address leads to a lookup per returned
*  PTR answer to see if the returned answer matches.  The forward lookups are
*  handled by a separate function which calls gethostbyname.  Upon a single
*  successful match of a forward lookup with a reverse lookup, returns TRUE.
*  Returns FALSE upon failure.
*
*/
SPF_BOOL DNS_ptr_answer(peer_info_t *peer_info, int16_t ancount,
const u_char *msg_ptr, const u_char *eom_ptr, u_char *rd_ptr, char *buf,
  const char *mta, int *ttl)
{
  int16_t   rc;                 /* generic return code / length of */
  int16_t   rd_len = 0;         /* res_search return code & packet len */
  int16_t   rd_type;            /* answer type */
  int32_t   rd_ttl;             /* TTL */

  char      *cp       = NULL;   /* working pointer for copy of mta */
  char      *copy     = NULL;   /* copy of mta */
  char      *ptr_dom  = NULL;   /* stores domain from buf's (ptr res) hostname */
  char      *cur_dom  = NULL;   /* stores domain from rpeer's hostname */

  while ((ancount > 0) && (rd_ptr < eom_ptr))
  {
    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(FALSE);
    }

     /* dname, type, class, TTL, rdata len, rdata */
    rd_ptr += rc;               /* jump to start of ANSWER data */
    GETSHORT(rd_type, rd_ptr);  /* get response type */
    rd_ptr += INT16SZ;          /* skip class */
    /*rd_ptr += INT32SZ;*/      /* skip TTL */
    GETLONG(rd_ttl, rd_ptr);    /* get TTL */
    GETSHORT(rd_len, rd_ptr);   /* get data length */

    *ttl = rd_ttl;

    if (rd_type != T_PTR)
    {
      xvprintf("Forged packet?!  We requested T_PTR (12) but got %i\n",
      rd_type);
      rd_ptr += rd_len;
      continue;
    }

    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(FALSE);
    }

    xvprintf("Answer %i has length %i.\n", ancount, rd_len);
    xvprintf("Answer Data: (%s) len: %i\n", buf, strlen(buf));

    if (rd_len <= SPF_MAXDNAME)
    {
      if ((ptr_dom = UTIL_get_dname(buf)) == NULL)
      {
        xvprintf("Invalid hostname (%s)\n", buf);
        return(FALSE);
      }

      cp = copy = xstrndup(mta, (strlen(mta) + 1));

      if ((cp = strstr(mta, ":")) != NULL)
      {
        cp++;
      }
      else
      {
        cp = (char *)mta; /* stfu compiler */
      }

      if ((cur_dom = UTIL_get_dname(cp)) == NULL)
      {
        xvprintf("Invalid hostname (%s)\n", mta);
        xfree(ptr_dom);
        xfree(copy);
        return(FALSE);
      }

      if (UTIL_validate_hostname(peer_info, buf, 32) == FALSE)
      {
        xvprintf("Unable to validate hostname (%s [%s]) with (%s [%s])\n",
          buf, ptr_dom, mta, cur_dom);

        xfree(ptr_dom);
        xfree(cur_dom);
        xfree(copy);
        return(FALSE);
      }

      if (strcasecmp(ptr_dom, cur_dom) == 0)
      {
        xprintf("Validated (%s [%s]) against(%s [%s])\n", buf, ptr_dom,
          mta, cur_dom);

        peer_info->r_vhname = xstrndup(mta, (strlen(mta) + 1));
        xfree(ptr_dom);
        xfree(cur_dom);
        xfree(copy);
        return(TRUE);
      }
      else
      {
        xprintf("Unable to validate (%s : %s)\n", buf, mta);
        xfree(ptr_dom);
        xfree(cur_dom);
        xfree(copy);
      }
    }
    else
    {
      xeprintf("Answer length is too long!\n");
    }

    rd_ptr += rc;
    ancount--;
  }

  return(FALSE);
}


/* DNS_cname_answer
*
*  Author: Teddy <teddy@teddy.ch>
*
*  Date:   29/04/04
*  Date:   02/20/04 - Added cache by Travis Anderson <travis@anthrax.ca>
*
*  Desc:
*          SPF_PACKETSZ bytes are allocated and then filled with \0 chars.
*  This buffer is then used in a TXT DNS query using data from the passed
*  peer_info_t structure.  Upon success this buffer is re-cast as a char *
*  and then a pointer to this memory is returned.  Upon failure a NULL
*  pointer is returned.
*
*/
char *DNS_cname_answer(int16_t ancount, const u_char *msg_ptr,
  const u_char *eom_ptr, u_char *rd_ptr, char *buf, int *ttl)
{
  int16_t   i;                /* utility */
  int16_t   rc;               /* generic return code / length of */
  int16_t   rd_type;          /* answer type */
  int16_t   rd_len;           /* res_search return code & packet len */
  int32_t   rd_ttl;           /* TTL */
  size_t    buf_len;          /* buffer length */

  char      *rr_data = NULL;  /* data pointer */

  if (msg_ptr == NULL || eom_ptr == NULL || rd_ptr == NULL || buf == NULL)
  {
    xeprintf("Called with NULL pointers\n");
    return(NULL);
  }

  i = 0;
  while (ancount > 0 && rd_ptr < eom_ptr)
  {
    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(NULL);
    }

    /* dname, type, class, TTL, rdata len, rdata */
    rd_ptr += rc;               /* jump to start of ANSWER data */
    GETSHORT(rd_type, rd_ptr);  /* get response type */
    rd_ptr += INT16SZ;          /* skip class */
    /*rd_ptr += INT32SZ;*/      /* skip TTL */
    GETLONG(rd_ttl, rd_ptr);    /* get TTL */
    GETSHORT(rd_len, rd_ptr);   /* get data length */

    *ttl = rd_ttl;

    if (rd_type != T_CNAME)
    {
      xvprintf("Ignoring record not of T_CNAME type. (%i)\n", rd_type);
      rd_ptr += rd_len;
      continue;
    }

    if ((rc = dn_expand(msg_ptr, eom_ptr, rd_ptr, buf, SPF_MAXCDNAME)) < 0)
    {
      xeprintf("Error obtaining ANSWER!\n");
      return(NULL);
    }

    buf_len = strlen(buf);
    i += (buf_len + 1);

    if (rd_len <= SPF_MAXDNAME)
    {
      if (rr_data == NULL)
      {
        rr_data = xmalloc(i + 1);
        memset(rr_data, '\0', (i + 1));
      }
      else
      {
        rr_data = xrealloc(rr_data, (i + 1));
      }

      xvprintf("REALLOCATE memory: %i bytes\n", (i + 1));

      strncat(rr_data, buf, buf_len);
      rr_data[i - 1] = ' ';
      rr_data[i] = '\0';
    }

    rd_ptr += rc;
    ancount--;
  }

  rr_data[i - 1] = '\0';
  return(rr_data);
}

/* end dns.c */
