/* libspf - Sender Policy Framework library
*
*  ANSI C implementation of spf-draft-200405.txt
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*  Author: Sean Comeau   <scomeau@obscurity.org>
*
*  File:   util.c
*  Desc:   Utility functions
*
*  License:
*
*  The libspf Software License, Version 1.0
*
*  Copyright (c) 2004 James Couzens & Sean Comeau  All rights
*  reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*  1. Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*
*  2. Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
*  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
*  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS MAKING USE OF THIS LICESEN
*  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
*  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
*  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
*  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
*  SUCH DAMAGE.
*
*/


#include "util.h"    /* Utility functions */
#include "dns.h"     /* DNS Functions */
#undef  VERSION      /* autoconf */


/* _printf_dbg
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/25/03
*  Date:   02/18/04 (updated)
*
*  Desc:
*          Tied to a compile time switch this can instantly and at little
*  to no real expense enable a discreet debugging with out hoards of
*  #ifdefs all over the place.
*
*/
void _printf_dbg(u_int8_t level, const char *function, const char *format,...)
{
  char     _dbg_output[MAX_DEBUG_LEN];
  va_list  argptr;

#ifdef _SPF_DEBUG_LOGFILE
  FILE     *fp      = NULL;
#endif

  va_start(argptr, format);
  vsnprintf(_dbg_output, SIZEOF(_dbg_output), format, argptr);
  va_end(argptr);

  if (f_bit_set(confg.level, level))
  {
#ifndef _SPF_DEBUG_LOGFILE
    printf("%s :: %s", function, _dbg_output);
    fflush(stdout);
#else
    if ((fp = fopen(DEBUG_LOG_FILE, "a")) != NULL)
    {
        fprintf(fp, "%s :: %s", function, _dbg_output);
        fclose(fp);
    }
#endif

  }

  return;
}


#ifndef _SPF_DEBUG_LOGFILE
/* dummy_debug
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/25/03
*
*  Desc:
*          dummy function thats used instead of the _printf_dbg function
*  when compiling without debugging
*
*/
void dummy_debug(const u_int8_t level, const char *function,
  const char *format,...)
{
  return;
}
#endif


/* UTIL_printf
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/25/03
*
*  Desc:
*          Works on a static block of memory to operate like printf,
*  returning a pointer to the new string.  Uses a static buffer
*  of 1K.
*
*  THIS FUNCTION IS NOT THREAD SAFE !!!!!
*
*/
char *UTIL_printf(char *format, ...)
{
  va_list arg_ptr;
  static char s[1024];

  va_start (arg_ptr, format);
  vsnprintf (s, 1024, format, arg_ptr);
  va_end (arg_ptr);

  xvprintf("Called with string: %s\n", s);

  return(s);
}


/* UTIL_get_date
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*  Date:   Sun Jan 18 06:02:13 PST 2004
*
*  Desc:
*          Opens up STDIN and reads up to 63 chars from it or until it reads
*  a newline.  Data is written using the pointer passed from the calling
*  function.
*
*/
char *UTIL_get_date(void)
{
  struct tm *now;
  time_t curtime;
  static char my_time[22];

  curtime = time(NULL);
  now = localtime(&curtime);

  strftime(my_time, SIZEOF(my_time), "%Y-%m-%d %H:%M:%S ", now);

  return(my_time);
}


/* UTIL_log_result
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   02/04/04
*
*  Desc:
*          Tied to a compile time switch this can instantly and at little
*  to no real expense enable a discreet debugging with out hoards of
*  #ifdefs all over the place.
*
*/
void UTIL_log_result(peer_info_t *peer_info)
{
  FILE  *fp = NULL;
  char  buf[MAX_DEBUG_LEN];
  char *date;

  date = UTIL_get_date();
  *(date + (strlen(date) - 1)) = '\0';

  if (peer_info->spf_ver == 0)
  {
    peer_info->spf_ver = SPF_VERSION;
  }

  snprintf(buf, MAX_DEBUG_LEN,
    "[%s] result: %s :: %s [%s], ver: %i, depth: %i, error: (%s)\n",
    date, peer_info->spf_result[peer_info->RES].s, peer_info->from,
    peer_info->r_ip, spf_rlevel, peer_info->spf_ver, peer_info->error);

  if ((fp = fopen(OUTPUT_LOG_FILE, "a")) != NULL)
  {
    fprintf(fp, "%s", buf);
    fclose(fp);
  }

  return;
}


/* xstrndup
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/25/03
*
*  Desc:
*          n bytes are allocated and then filled with \0 chars.  Char s
*  is copied over to the allocated memory writing n -1 bytes leaving the
*  new string NULL terminated.  This new string is returned upon success
*  and NULL upon failure.
*
*/
char *UTIL_strndup(const char *s, const size_t n)
{
  char *ret_ptr = NULL;

  if (s == NULL)
  {
    xeprintf("Passed string is NULL.  Abort!.\n");
    return(FALSE);
  }

  ret_ptr = xmalloc(n);
  xvprintf("Allocated %u bytes of memory.\n", n);

  memset(ret_ptr, '\0', n);
  memcpy(ret_ptr, s, (n - 1));

  xvprintf("Returning string: (%s)\n",
    ret_ptr);

  return(ret_ptr);
}


/* xstrdup
*
*  Author: Patrick Earl (http://patearl.net/)
*          Adapted from xstrndup()
*
*  Date:   02/04/04
*
*  Desc:
*          strlen(s)+1 bytes are allocated and s is copied into the
* freshly allocated memory.  If the allocation or copy fails, NULL
* NULL will be returned.
*
*/
char *UTIL_strdup(const char *s)
{
  char *ret_ptr = NULL;

  if (s == NULL)
  {
    xeprintf("Passed string is NULL.  Abort!.\n");
    return(FALSE);
  }

  if ((ret_ptr = strdup(s)) == NULL)
  {
    xeprintf("Unable to allocate memory\n");
  }

  xvprintf("Returning string: (%s)\n", ret_ptr);

  return(ret_ptr);
}


/* UTIL_malloc
*
*  Author: Travis Anderson <travis@anthrax.ca>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   02/17/04
*
*  Desc:
*         Wrapper for malloc.  Upon success, behaves as malloc does.
*  Wrapper functionality is to print an error message and exit upon failure.
*
*/
void *UTIL_malloc(const int32_t n, const char *file, int32_t line,
  const char *func)
{
    void *x = malloc(n);

    if (x == NULL)
    {
        xvprintf("Unable to allocate %i bytes at %s:%i in %s\n",
          n, file, line, func);

        exit(0);
    }

    return(x);
}


/* UTIL_realloc
*
*  Author: Travis Anderson <travis@anthrax.ca>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   02/17/04
*
*  Desc:
*         Wrapper for realloc.  If 'p' is NULL, allocates memory via a call to
*  UTIL_malloc, otherwise if 'x' is assigned successfully, the behaviour is
*  identical to realloc.  Wrapper functionality is to print an error message
*  and exit upon failure.
*
*/
void *UTIL_realloc(void *p, const int32_t n, const char *file, const int32_t
line,
  const char *func)
{
    void *x = NULL;

    if (p == NULL)
    {
      return(UTIL_malloc(n, file, line, func));
    }

    x = realloc(p, n);
    if (x == NULL)
    {
        xvprintf("Unable to reallocate %i bytes at %s:%i in %s; " \
          "original address 0x%x\n", n, file, line, func, (uintptr_t)p);

        exit(0);
    }

    return(x);
}


/* UTIL_free
*
*  Author: Travis Anderson <travis@anthrax.ca>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   02/17/04
*
*  Desc:
*         Wrapper for free.  Upon success, behaves as free does.
*  Wrapper functionality is to print an error message and exit upon
*  failure.
*
*/
void UTIL_free(void *p, const char *file, const int32_t line, const char *func)
{
    if (p == NULL)
    {
        xvprintf("Unable to free() on NULL pointer at %s:%i in %s; " \
          "address 0x%x.\n", file, line, func, (uintptr_t)p);
        return;
    }

    xvprintf("Free address 0x%x by %s on line %i (%s)\n",
      (uintptr_t)p, func, line, file);

    free(p);
    return;
}


/* UTIL_index
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/19/03
*
*  Desc:
*          s is walked until c is found, at which time i is returned
*  which is the number of bytes from the left it walked until c was
*  found (not including c its self);  Returns 0 upon failure.
*
*/
int16_t UTIL_index(const char *s, const char c)
{
  int16_t   i;     /* utility */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(0);
  }

  i = 0;
  while (*s)
  {
    if (*s == c)
    {
      xvprintf("Found search char: (%c); Returning: (%i)\n", *s, i);

      return(i);
    }
    i++;
    s++;
  }
  return(0);
}


/* UTIL_split_str
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/21/04
*
*  Desc:
*          s is walked through to find 'delim' until it finds 'delim'
*  num times.  Upon final match it returns the remainder of the string
*  in a newly allocated buffer.  Upon failure returns NULL.
*
*/
char *UTIL_split_str(const char *s, const char c, const u_int8_t num)
{
  u_int8_t i;

  char *cp;
  char *p;
  char *ret;

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  p = cp = xstrndup(s, (strlen(s) + 1));

  i = 0;
  while(*p)
  {
    if (*p == c)
    {
      i++;
      if (i == num)
      {
        p++;
        ret = xstrndup(p, (strlen(p) + 1));
        xfree(cp);
        xvprintf("returning: %s\n", ret);
        return(ret);
      }
    }
    p++;
  }

  xvprintf("[%i] ]returning NULL\n", i);

  return(NULL);
}


/* UTIL_split_strr
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/30/04
*
*  Desc:
*          s is walked to the end of its self, and then is walked back
*  towards the beginning of the string until it has found 'c' the
*  delimiter, 'num' times.  Upon success memory is allocated and the
*  remainder of s is returned.  Upon failure NULL is returned.
*
*/
char *UTIL_split_strr(const char *s, const char c, const u_int8_t num)
{
  u_int8_t  i;        /* number of times delim (c) is found */

  char      *p;       /* pointer to the last character of s before the \0 */
  char      *ret;     /* return buffer */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  xvprintf("Called with (%s)\n", s);

  /* assign p  the last char before \0 of s */
  p = (char *)&(s[strlen(s) - 1]);

  i = 0;
  while(p != s)
  {
    if (*p == c)
    {
      i++;
      if (i == num)
      {
        if (*p == '.')
        {
          p++; /* don't want that period */
        }

        ret = xstrdup(p);

        xvprintf("[%i] returning: %s\n", i, ret);
        return(ret);
      }
    }
    p--;
  }

  xvprintf("[%i] returning NULL\n", i);
  return(NULL);
}


/* UTIL_count_delim
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/21/04
*
*  Desc:
*          s is walked through and each time 'delim' is found a
*  counter is incremented and once complete, that integer is
*  returned to the calling function.  Specifically for the purposees
*  of this project its limited by its type to 255.  Returns any number
*  greater than 0 and less than or equal to 255.  Upon failure returns
*  0.
*
*/
u_int8_t UTIL_count_delim(const char *s, const char c)
{
  u_int8_t i = 0;

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(0);
  }

  while (*s)
  {
    if (*s == c)
    {
      i++;
    }
    s++;
  }

  xvprintf("returning: %i\n", i);
  return(i);
}


/* UTIL_guess_whats_in_the_box ?!
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/21/04
*
*  Desc:
*          s is walked through and each time 'delim' is found a
*  counter is incremented, all the while through each iteration the
*  overall byte count is decremented.  When the number of delimiter
*  SPF_MATCHes found is identical to 'num' the number of bytes are
*  decremented once more (to remove the delimiter) and then returned.
*  Upon failure 0 is returned.
*
*/
size_t UTIL_guess_whats_in_the_box(char *s, const char c, const u_int16_t num)
{
  u_int8_t  i       = 0;
  size_t    bytes   = 0;

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");

    return(0);
  }

  bytes = strlen(s);

  while (*s)
  {
    if (*s == c)
    {
      i++;
    }

    if (i == num)
    {
      bytes--;
      s++;

      xvprintf("return (%i bytes) (%s)\n", bytes, s);

      return(bytes);
    }
    else
    {
      bytes--;
      s++;
    }
  }

  xvprintf("[return (%i bytes) (%s)\n", bytes, s);

  return(bytes);
}


/* UTIL_is_spf_delim
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/21/04
*
*  Desc:
*          c is compared against all the valid spf delimiters and
*  in the case of a SPF_PASS, the function returns true, otherwise it will
*  return false.
*
*/
SPF_BOOL UTIL_is_spf_delim(const char c)
{
  xvprintf("Called with char (%c)\n", c);

  if (c == '.' || c == '-' || c == '+' || c == ','
    || c == '|' || c == '_')
  {
    return(TRUE);
  }

  return(FALSE);
}


/* UTIL_is_spf_result
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/27/04
*
*  Desc:
*          c is compared against all the valid spf prefixes and
*  in the case of a SPF_PASS, the function returns true, otherwise it will
*  return false.
*
*/
SPF_BOOL UTIL_is_spf_result(const char c)
{
  xvprintf("Called with char (%c)\n", c);

  if (c == '+' || c == '-' || c == '~' || c == '?')
  {
    return(TRUE);
  }

  return(FALSE);
}


/* UTIL_is_macro
*
*  Author:  James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/30/04
*
*  Desc:
*          s is walked through in search of a macro which consist of
*  %{?}, ? being an SPF_UNKNOWN number of chars in between.  An instance of
*  a macro is searched for.  Upon success TRUE is returned.  Upon failure
*  to find a macro, FALSE is returned.
*
*/
SPF_BOOL UTIL_is_macro(const char *s)
{
  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");

    return(FALSE);
  }

  xvprintf("Called with char (%s)\n", s);

  while (*s++)
  {
    if (*s == '%' && *(s + 1) == '{')
    {
      if (strstr(s, "}"))
      {
        return(TRUE);
      }
    }
  }

  return(FALSE);
}


/* UTIL_mx_cmp
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/02/04
*
*  Desc:
*          Using the domain name found in the passed peer_info
*  structure, the associated T_MX records are looked up and then their
*  respective hostnames resolved (if they happen to be IP addresses the
*  resolver library takes care of this. <3).  Each IP is compared
*  against the remote peer's IP (from peer_info).  Upon success returns
*  TRUE, upon failure returns FALSE.
*
*/
SPF_BOOL UTIL_mx_cmp(peer_info_t *peer_info, const char *s, const int8_t cidr)
{
  SPF_BOOL  MX_SPF_MATCH = FALSE;   /* true / false */

  char      *rr_data = NULL;        /* record */
  char      *token;                 /* token for splitting records */
  char      *token_ptr;
  char      *peer_ip;               /* remote host converted to string */

  token_ptr = rr_data;

  if ((rr_data = DNS_query(peer_info, s, T_MX, NULL)) == NULL)
  {
    xeprintf("SPF_ERROR parsing DNS Query\n");
    return(FALSE);
  }

  xvprintf("rr_data is: (%s)\n", rr_data);

  peer_ip = xstrndup(inet_ntoa(peer_info->addr), 16);
  token = strtok_r(rr_data, " ", &token_ptr);

  while (token != NULL)
  {
    xvprintf("TOKEN: (%s)\n", token);

    if (UTIL_validate_hostname(peer_info, token, cidr) == TRUE)
    {
      xvprintf("%s validated via (%s)\n", peer_info->from, token);

      MX_SPF_MATCH = TRUE;
      UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
      token = NULL;
    }
    else
    {
      token = strtok_r(NULL, " ", &token_ptr);
    }
  }

  xfree(peer_ip);
  xfree(rr_data);

  return(MX_SPF_MATCH);
}


/* UTIL_a_cmp
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/02/04
*
*  Desc:
*          Calls gethostbyname to grab all records associated with a the
*  hostname contained within s.  On success returns TRUE, on SPF_ERROR
*  returns FALSE.
*
*  Note:
*          We need to define and set a limit on the number of recursive
*  lookups.  I recall seeing this in the RFC, we should discuss this.
*
*/
SPF_BOOL UTIL_a_cmp(peer_info_t *peer_info, const char *s, const int8_t cidr)
{
  int16_t   pos;

  char      *rr_data = NULL;
  char      *token_ptr;

  char      *cp;
  char      *copy;
  char      **a;

  size_t    s_len;

  struct    hostent *host;

  policy_addr_t *policy_addr;

  if (s == NULL)
  {
    xeprintf("Passed string is NULL.  Abort!.\n");

    return(FALSE);
  }

  xvprintf("Called with (%s) and cidr: %i\n", s, cidr);

  token_ptr = rr_data;
  s_len = strlen(s);

  /* we're dealing with a:some.hostname/cidr */
  if (s_len > 1 && *(s + 1) == ':')
  {
    cp = copy = xstrndup(s, (s_len + 1));

    if (cidr != 32)
    {
      /* We don't want a netmask, so lets remove it */
      cp[s_len - 3] = '\0';
    }

    if ((pos = UTIL_index(cp, ':')) <= 0)
    {
      xeprintf("ERROR parsing passed mechanism token\n");

      xfree(copy);
      return(FALSE);
    }

    /* move passed the mechanism text */
    cp += (pos + 1);
  }
  else
  {
    cp = copy = xstrndup(peer_info->cur_dom,
      (strlen(peer_info->cur_dom) + 1));
  }

  policy_addr = xmalloc(SIZEOF(policy_addr_t));

  if ((host = gethostbyname(cp)) != NULL)
  {
    for (a = host->h_addr_list; *a; a++)
    {
      memcpy(&policy_addr->addr.s_addr, *a, SIZEOF(struct in_addr));

      xvprintf("Checking IP: %lu\n",policy_addr->addr.s_addr);

      /* cidr is assumed checked by the calling function! */
      policy_addr->cidr = cidr;

      if (UTIL_cidr_cmp(/*peer_info, */policy_addr, &peer_info->addr) == TRUE)
      {
        *a = NULL;
        xfree(policy_addr);
        xfree(copy);
        UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
        return(TRUE);
      }
    }

    for (a = host->h_aliases; *a; a++)
    {
      memcpy(&policy_addr->addr.s_addr, *a, SIZEOF(struct in_addr));

      xvprintf("Checking ALIAS: %lu\n", policy_addr->addr.s_addr);

      /* cidr is assumed checked by the calling function! */
      policy_addr->cidr = cidr;

      if (UTIL_cidr_cmp(/*peer_info, */policy_addr, &peer_info->addr) == TRUE)
      {
        *a = NULL;
        xfree(policy_addr);
        xfree(copy);
        UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
        return(TRUE);
      }
    }
  }

  xfree(policy_addr);
  xfree(copy);

  return(FALSE);
}


/* UTIL_ptr_cmp
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/08/04
*
*  Desc:
*          gethostbyaddr is broken in linux.  ?->h_aliases is not NULL,
*  however, it doesn't contain any valid pointers either.  It grabs the
*  first (and of course this is random) hostname it gets and thats all.
*  As tested in FreeBSD however this call works.  At any rate, I've
*  written a function to deal with this called DNS_ptr_answer.  From here
*  that function is called which handles the reverse lookups and then
*  attempts to "validate" each returned hostname by in turn looking it up
*  to confirm if the ip address SPF_MATCHes.  Returns TRUE on succes, and FALSE
*  on failure.
*
*  When passed TRUE copies into peer_info->ptr_mhost, when FALSE copies
*  into peer_info->r_vhname
*
*  Note:
*          We need to define and set a limit on the number of recursive
*  lookups.  I recall seeing this in the RFC, we should discuss this.
*
*/
SPF_BOOL UTIL_ptr_cmp(peer_info_t *peer_info, const char *s)
{
  char      *ptr_addr;      /* reversed address into PTR format */
  char      *tmp_ptr;       /* utility pointer */

  if (s == NULL)
  {
    xeprintf("Passed string is NULL.  Abort!\n");
    return(FALSE);
  }

  xvprintf("Called with (%s)\n", s);

  /* reverse of rpeer */
  ptr_addr = UTIL_rev_addr(peer_info->r_ip);

  xvprintf("address: %s\n", ptr_addr);

  if ((strstr(s, ":")) != NULL)
  {
    tmp_ptr = xstrndup(s, (strlen(s) + 1));
  }
  else
  {
    tmp_ptr = xstrndup(peer_info->cur_dom,
      (strlen(peer_info->cur_dom) + 1));
  }

  if (DNS_query(peer_info, ptr_addr, T_PTR, tmp_ptr) != (char *)TRUE)
  {
    xvprintf("PTR lookup failed: (%s) (%s)\n", peer_info->rs, peer_info->error);

    xfree(ptr_addr);
    xfree(tmp_ptr);
    return(FALSE);
  }

  xvprintf("PTR lookup succeeded: (%s):(%s)\n", peer_info->rs,
    peer_info->error);

  xfree(ptr_addr);
  xfree(tmp_ptr);
  return(TRUE);
}


/* UTIL_get_policy_mech
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/19/03
*
*  Desc:
*          Examins s and attempts to determine which SPF_MECHANISM
*  enumeration the string SPF_MATCHes based on its conent.
*
*/
SPF_MECHANISM UTIL_get_policy_mech(const char *s)
{
  xvprintf("Called with: (%s)\n", s);

  if (s == NULL || !s)
  {
    xeprintf("Passed a NULL string. Abort!\n");
    return(NO_POLICY);
  }

  if (strncmp(s, "v=", 2) == 0)
  {
    xvprintf("Returning %i (VERSION)\n", VERSION);
    return(VERSION);
  }
  else if (strncmp(s, "ip4:", 4) == 0 )
  {
    xvprintf("Returning %i (IP4)\n", IP4);
    return(IP4);
  }
  else if (strncmp(s, "ip6:", 4) == 0)
  {
    xvprintf("Returning %i (IP6)\n", IP6);
    return(IP6);
  }
  else if (strncmp(s, "all", 3) == 0)
  {
    xvprintf("Returning %i (ALL)\n", ALL);
    return(ALL);
  }
  else if (strncmp(s, "mx", 2) == 0)
  {
    xvprintf("Returning %i (MX)\n", MX);
    return(MX);
  }
  else if (strncmp(s, "a:", 2) == 0 || (*s == 'a' && *(s + 1) == '/')
    ||  ((*s == 'a') && !*(s + 1)))
  {
    xvprintf("Returning %i (A)\n", A);
    return(A);
  }
  else if (strncmp(s, "ptr", 3) == 0)
  {
    xvprintf("Returning %i (PTR)\n", PTR);
    return(PTR);
  }
  else if (strncmp(s, "include:", 7) == 0)
  {
    xvprintf("Returning %i (INCLUDE)\n", INCLUDE);
    return(INCLUDE);
  }
  else if (strncmp(s, "exists:", 6) == 0)
  {
    xvprintf("Returning %i (EXISTS)\n", EXISTS);
    return(EXISTS);
  }
  else if (strncmp(s, "redirect=", 9) == 0)
  {
    xvprintf("Returning %i (REDIRECT)\n", REDIRECT);
    return(REDIRECT);
  }
  else if (strncmp(s, "exp=", 3) == 0)
  {
    xvprintf("Returning %i (EXPLAIN)\n", EXPLAIN);
    return(EXPLAIN);
  }
  else if (strncmp(s, "default", 7) == 0)
  {
    xvprintf("Returning %i (DEFAULT)\n", DEFAULT);
    return(DEFAULT);
  }
  else if (strstr(s, ":"))
  {
    xvprintf("Returning %i (UNMECH)\n", UNMECH);
    return(UNMECH);
  }

  return(NO_POLICY);
}


/* UTIL_assoc_prefix
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/28/04
*
*  Desc:
*          Examins s and attempts to determine which SPF_MECHANISM_PREFIX
*  enumeration the string SPF_MATCHes based on the its contents (which is a
*  single char.  Upon a SPF_PASS it stores the appropriate value inside of
*  the passed peer_info structure.  TRUE upon success, FALSE upon failure.
*  Also upon failure SPF_ERROR (SPF_RESULT_TYPE) is stored in peer_info.
*
*
*/
SPF_BOOL UTIL_assoc_prefix(peer_info_t *peer_info, SPF_RESULT p, const char *s)
{
  int16_t pos;      /* position in s string */

  if (s != NULL)
  {
    xvprintf("(QID: %u) :: Entering function (%i) (%s)\n", spf_rlevel, p, s);

    /* support for old school deprecated mechanism "default" */
    if (strncmp(s, "default", 7) == 0 && (pos = UTIL_index(s, '=')) > 0)
    {
      s += (pos + 1); /* move past default= */
      if (strncmp(s, "deny", 4) == 0)
      {
        xvprintf("(QID: %u) :: Stored SPF_H_FAIL (%i) (%i)\n",
          spf_rlevel, p, SPF_H_FAIL);

        peer_info->RES  = SPF_H_FAIL;
        peer_info->rs   = peer_info->spf_result[SPF_H_FAIL].s;

        snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
          "(%s)\n", peer_info->rs, peer_info->last_m);

        return(TRUE);
      }
      else if (strncmp(s, "pass", 4) == 0)
      {
        xvprintf("(QID: %u) :: Stored SPF_PASS (%i) (%i)\n",
          spf_rlevel, p, SPF_PASS);

        peer_info->RES  = SPF_PASS;
        peer_info->rs   = peer_info->spf_result[SPF_PASS].s;
        snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
          "(%s)\n", peer_info->rs, peer_info->last_m);
        return(TRUE);
      }
      else if (strncmp(s, "softdeny", 8) == 0)
      {
        xvprintf("(QID: %u) :: Stored SPF_S_FAIL (%i) (%i)\n",
          spf_rlevel, p, SPF_S_FAIL);

        peer_info->RES  = SPF_S_FAIL;
        peer_info->rs   = peer_info->spf_result[SPF_S_FAIL].s;
        snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
          "(%s)\n",  peer_info->rs, peer_info->last_m);
        return(TRUE);
      }
      else if (strncmp(s, "unknown", 7) == 0)
      {
        xvprintf("(QID: %u) :: Stored SPF_NEUTRAL (%i) (%i)\n",
          spf_rlevel, p, SPF_NEUTRAL);

        peer_info->RES  = SPF_NEUTRAL;
        peer_info->rs   = peer_info->spf_result[SPF_NEUTRAL].s;
        snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
          "(%s)\n", peer_info->rs, peer_info->last_m);
        return(TRUE);
      }
      else if (strncmp(s, "include", 7) == 0)
      {
        xvprintf("(QID: %u) :: Stored SPF_UNKNOWN (%i) (%i)\n",
          spf_rlevel, p, SPF_UNKNOWN);

        peer_info->RES  = SPF_UNKNOWN;
        peer_info->rs   = peer_info->spf_result[SPF_UNKNOWN].s;
        snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
          "(%s)\n", peer_info->rs, peer_info->last_m);
        return(TRUE);
      }
      else
      {
        xvprintf("(QID: %u) :: Stored SPF_ERROR (%i) (%i)\n",
          spf_rlevel, p, SPF_ERROR);

        peer_info->RES  = SPF_UNKNOWN;
        peer_info->rs   = peer_info->spf_result[SPF_UNKNOWN].s;
        snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
          "(%s)\n", peer_info->rs, peer_info->last_m);
        return(FALSE);
      }
    }
  }

  switch (p)
  {
    case SPF_PASS:
      xvprintf("(QID: %u) :: Stored SPF_PASS (%i) (%i)\n",
        spf_rlevel, p, SPF_PASS);

      peer_info->RES  = SPF_PASS;
      peer_info->rs   = peer_info->spf_result[SPF_PASS].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_NONE:
      xvprintf("(QID: %u) :: Stored SPF_NONE (%i) (%i)\n",
        spf_rlevel, p, SPF_NONE);

      peer_info->RES  = SPF_NONE;
      peer_info->rs   = peer_info->spf_result[SPF_NONE].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_S_FAIL:
      xvprintf("(QID: %u) :: Stored SPF_S_FAIL (%i) (%i)\n",
        spf_rlevel, p, SPF_S_FAIL);

      peer_info->RES  = SPF_S_FAIL;
      peer_info->rs   = peer_info->spf_result[SPF_S_FAIL].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_H_FAIL:
      xvprintf("(QID: %u) :: Stored SPF_H_FAIL (%i) (%i)\n",
        spf_rlevel, p, SPF_H_FAIL);

      peer_info->RES  = SPF_H_FAIL;
      peer_info->rs   = peer_info->spf_result[SPF_H_FAIL].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_NEUTRAL:
      xvprintf("(QID: %u) :: Stored SPF_NEUTRAL (%i) (%i)\n",
        spf_rlevel, p, SPF_NEUTRAL);

      peer_info->RES  = SPF_NEUTRAL;
      peer_info->rs   = peer_info->spf_result[SPF_NEUTRAL].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_UNKNOWN:
      xvprintf("(QID: %u) :: Stored SPF_UNKNOWN (%i) (%i)\n",
        spf_rlevel, p, SPF_UNKNOWN);

      peer_info->RES  = SPF_UNKNOWN;
      peer_info->rs   = peer_info->spf_result[SPF_UNKNOWN].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_ERROR:
      xvprintf("(QID: %u) :: Stored SPF_ERROR (%i) (%i)\n",
        spf_rlevel, p, SPF_ERROR);

      peer_info->RES  = SPF_ERROR;
      peer_info->rs   = peer_info->spf_result[SPF_ERROR].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
    case SPF_UNMECH:
      xvprintf("(QID: %u) :: Stored SPF_UNMECH (%i) (%i)\n",
        spf_rlevel, p, SPF_UNMECH);

      peer_info->RES  = SPF_UNMECH;
      peer_info->rs   = peer_info->spf_result[SPF_UNMECH].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);

    default:
      xvprintf("(QID: %u) :: Stored SPF_PASS (%i) (%i)\n",
        spf_rlevel, p, SPF_PASS);

      peer_info->RES  = SPF_PASS;
      peer_info->rs   = peer_info->spf_result[SPF_PASS].s;
      snprintf(peer_info->error, MAX_ERROR, "policy result: (%s) from rule " \
        "(%s)\n", peer_info->rs, peer_info->last_m);
      return(TRUE);
  }

  return(FALSE);
}


/* UTIL_get_mech_prefix
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/31/03
*
*  Desc:
*          Examins s and attempts to determine which SPF_RESULT_P
*  enumeration the string SPF_MATCHes based on the its contents (which is a
*  single char.  Returns SPF_PASS if a valid prefix is not specified.
*  Valid prefixes are: + (permit), - (deny), ~ (softfail) and ? (SPF_NEUTRAL).
*
*/
SPF_RESULT UTIL_get_mech_prefix(peer_info_t *peer_info, const char *s)
{
  int16_t pos;

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(SPF_ERROR);
  }

  snprintf(peer_info->last_m, (strlen(s) +1), "%s", s);

  switch (*s)
  {
    case '+':
      xvprintf("(QID: %u) :: Returning SPF_PASS (%s) %i\n",
        spf_rlevel, s, SPF_PASS);
      peer_info->RES_P = SPF_PASS;
      return(SPF_PASS);
    case '-':
      xvprintf("(QID: %u) :: Returning SPF_H_FAIL (%s) %i\n",
        spf_rlevel, s, SPF_H_FAIL);
       peer_info->RES_P = SPF_H_FAIL;
       return(SPF_H_FAIL);
    case '?':
      xvprintf("(QID: %u) :: Returning SPF_NEUTRAL (%s) %i\n",
        spf_rlevel, s, SPF_NEUTRAL);
      peer_info->RES_P = SPF_NEUTRAL;
      return(SPF_NEUTRAL);
    case '~':
      xvprintf("(QID: %u) :: Returning SPF_S_FAIL (%s) %i\n",
        spf_rlevel, s, SPF_S_FAIL);
      peer_info->RES_P = SPF_S_FAIL;
      return(SPF_S_FAIL);
    default:
      if (peer_info->ALL == TRUE)
      {
        xvprintf("(QID: %u) :: Returning (def) SPF_NEUTRAL (%s) %i\n",
          spf_rlevel, s, SPF_NEUTRAL);
        peer_info->RES_P = SPF_NEUTRAL;
      }
      else
      {
        xvprintf("(QID: %u) :: Returning (def) SPF_PASS (%s) %i\n",
          spf_rlevel, s, SPF_PASS);
        peer_info->RES_P = SPF_PASS;
      }
      return(peer_info->RES_P);
  }

  /* support for old school deprecated mechanism "default" */
  if ((pos = UTIL_index(s, '=')) > 0)
  {
    s += (pos + 1); /* move past default= */

    if (strncmp(s, "deny", 4) == 0)
    {
      peer_info->RES_P = SPF_H_FAIL;
      return(SPF_H_FAIL);
    }
    else if (strncmp(s, "pass", 4) == 0)
    {
      peer_info->RES_P = SPF_PASS;
      return(SPF_PASS);
    }
    else if (strncmp(s, "softdeny", 8) == 0)
    {
      peer_info->RES_P = SPF_S_FAIL;
      return(SPF_S_FAIL);
    }
    else if (strncmp(s, "unknown", 7) == 0)
    {
      peer_info->RES_P = SPF_NEUTRAL;
      return(SPF_NEUTRAL);
    }
    else
    {
      return(SPF_NEUTRAL);
    }
  }

  xvprintf("Undetermined, returning "
    "SPF_PASS (%s)\n", s);

  return(SPF_ERROR);
}


/* UTIL_expand_ip
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/26/03
*
*  Desc:
*          Expands an ip4 policy mechanism string into a policy_addr_t
*  structure which it allocates memory for and then returns.  The RFC
*  specifies that if a cidr block is not specified then /32 isused.
*
*  On success returns a pointer to a policy_addr_t structure, on
*  SPF_ERROR returns NULL.
*
*/
policy_addr_t *UTIL_expand_ip(const char *s)
{
  size_t        len;           /* length of string passed */
  const char    *token_ptr;    /* our position in the token */
  size_t        pos;           /* position of break points in string */

  char          *ip;           /* temporary storage for ip address */
  int8_t        cidr = 0;      /* temporary storage for netmask */

  policy_addr_t *policy_addr;  /* ip/mask return structure */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  len = strlen(s);
  token_ptr = s;

  xvprintf("Called with string: (%s)\n", token_ptr);

  if ((pos = UTIL_index(token_ptr, ':')) == 0)
  {
    xvprintf("SPF_ERROR: Unable to get position on token (%s)\n",
      token_ptr);

    return(NULL);
  }

  /* jump past the ip4: portion */
  token_ptr += pos + 1;

  policy_addr = xmalloc(SIZEOF(policy_addr_t));

  /* jump past the ip4: portion (to get length of ip) */
  if ((pos = UTIL_index(token_ptr, '/')) == 0)
  {
    xvprintf("Unable to get position on token (%s), assuming /32 cidr "
      "block\n", token_ptr);

    pos = strlen(token_ptr);
    cidr = 32;
  }

  /* allocate space and dump the ip address there */
  ip = xstrndup(token_ptr, (pos + 1));

  /* copy it over to the policy_addr structure */
  if ((inet_pton(AF_INET, ip, &policy_addr->addr)) == 0)
  {
    xvprintf("SPF_ERROR: inet_pton unable to convert ip to binary "
      "(%s)\n", ip);

    xfree(ip);
    return(NULL);
  }

  if (cidr != 32)
  {
    token_ptr += (pos + 1);   /* skip over the forward slash */
    cidr = atoi(token_ptr);   /* convert the string to an integer */
  }

  xfree(ip);                   /* don't need this memory anymore */

  if (cidr < 0 || cidr > 32)
  {
    xvprintf("SPF_ERROR: cidr violation (%u)\n", cidr);
    return(NULL);
  }

  policy_addr->cidr = cidr; /* copy it over to the policy_addr structure */

  xvprintf("CIDR: (%i) IP: (%s)\n", policy_addr->cidr,
    inet_ntoa(policy_addr->addr));

  return(policy_addr);
}


/*  UTIL_is_ip
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/18/04
*
*  Desc:
*         Is the passed string (limited to between 0 - 255 by type)
*  contains a valid ip address or not.  I'm not sure if this is useful
*  anymore but at one point I was using it to tell if the passed macro
*  contained an ip address or not when passed as a string.
*
*/
SPF_BOOL UTIL_is_ip(const char *id)
{
  u_int8_t i = 0;

  while (*id)
  {
    if (*id == '.')
    {
      i++;
    }
    else if (isdigit(*id) == 0)
    {
      return(FALSE);
    }
    id++;
  }

  if (i == 3)
  {
    return(TRUE);
  }

  return(FALSE);
}


/* UTIL_rev_addr
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/04/04
*
*  Desc:
*          Using a passed ip address it will reverse it and slap
*  .in-addr.arpa on the end of it making it ready for a proper PTR query
*
*  Notes:  Write additional code to handle IP6 addresses
*
*/
char *UTIL_rev_addr(const char *s)
{
  u_int8_t i = 0;       /* utility */
  u_int16_t len = 0;     /* length of to be allocated string */
  u_int8_t tmp[4][1];   /* temporary storage for ip integers */

  char *cp;             /* copy of passed string */
  char *token;          /* token for splitting apart ip address */
  char *new_addr;       /* allocate string for reversed address */

  if (s == NULL)
  {
    xeprintf("Passed a null string.  Abort!\n");
    return(NULL);
  }

  xvprintf("Called with: (%s) len: %i\n", s, strlen(s));

  len = (strlen(s) + 1);
  cp = xstrndup(s, len);
  token = strtok(cp, ".");

  while (token != NULL && i <= 3)
  {
    xvprintf("token : (%s)\n", token);
    tmp[i][0] = atoi(token);
    token = strtok(NULL, ".");
    i++;
  }

  xfree(cp);

  /* + .in-addr.arpa\0 */
  new_addr = xmalloc(len + 13);

  memset(new_addr, '\0', len);
  snprintf(new_addr, (len + 13), "%u.%u.%u.%u.in-addr.arpa", tmp[3][0], tmp[2][0],
    tmp[1][0], tmp[0][0]);

  xvprintf("Returning reversed ip: %s\n", new_addr);

  return(new_addr);
}


/* UTIL_get_dname
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/27/04
*
*  Desc:
*          s contains a hostname which is then looked through and split
*  apart, always leaving one '.' delimiter left so you are left with
*  domain.tld.  domain.tld is then put into newly allocated memory and
*  returned to the calling function.  Upon failure, returns NULL.
*  Calling function is required to free the memory.
*
*/
char *UTIL_get_dname(const char *s)
{
  u_int8_t  i     = 0;     /* how many delimiters in s */
  char      *buf  = NULL;  /* return string to be allocated */

  if (s == NULL)
  {
    xeprintf("Called with NULL.  Abort!\n");

    return(NULL);
  }

  xvprintf("Called with (%s)\n", s);

  i = UTIL_count_delim(s, '.');

  switch (i)
  {
    case 0:
      return(NULL);
    case 1:
      buf = xstrndup(s, (strlen(s) + 1));
      return(buf);
    default:
      buf = UTIL_split_str(s, '.', (i - 1));
      return(buf);
  }

  return(NULL);
}


/* UTIL_cidr_cmp
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/26/03
*
*  Desc:
*          Compares an IP address (connected peer) against another IP
*  address (found in a policy lookup) using a netmask (also found from
*  a policy lookup) to see if the peer address is legal within that
*  netmask.  Returns TRUE if its valid, FALSE if not.
*
*  Notes:  Write additional comparisons to handle IP6 addresses
*
*/
SPF_BOOL UTIL_cidr_cmp(const policy_addr_t *policy_addr, const struct in_addr
  *peer_addr)
{
  u_int32_t a;
  u_int32_t b;

  char *peer_ip = NULL;
  char *policy_ip = NULL;

  if ((policy_addr->addr.s_addr <= 0) &&
    (peer_addr->s_addr <= 0))
  {
    xeprintf("Passed with NULL chars.  Aborting.\n");
    return(FALSE);
  }

  xvprintf("POL: %lu PEER: %lu CIDR: %i\n", policy_addr->addr.s_addr,
    peer_addr->s_addr, policy_addr->cidr);

  a = ntohl(peer_addr->s_addr);
  b = ntohl(policy_addr->addr.s_addr);

  if (policy_addr->cidr != 32)
  {
    if ((a&(~0U<<(32-policy_addr->cidr))) != (b&(~0U<<(32-policy_addr->cidr))))
    {
      return(FALSE);
    }
  }
  else
  {
    if (peer_addr->s_addr != policy_addr->addr.s_addr)
    {
      xvprintf("%lu and %lu using 32 cidr do not match\n", peer_addr->s_addr,
        policy_addr->addr.s_addr);
      return(FALSE);
    }
  }

  /* these are done here just for debugger remove later */
  peer_ip = xstrndup(inet_ntoa(*peer_addr), IP_ADDR);
  policy_ip = xstrndup(inet_ntoa(policy_addr->addr), IP_ADDR);

  xvprintf("Peer: (%s) matches address %s with network %i\n", peer_ip,
    policy_ip, policy_addr->cidr);

  xfree(peer_ip);
  xfree(policy_ip);

  return(TRUE);
}


/* UTIL_validate_ptr
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/26/03
*
*  Desc:
*          Using the ip address found in the passed peer_info structure,
*  a PTR lookup is made and if a record exists then this is record is compared
*  against the hostname the client wishes to send mail from.  If the tld matches
*  the tld found during this query, this would be a 'valdiated' PTR.  If a match
*  can not be found we return FALSE.  If a match is made we return TRUE.
*
*  Notes:  Write additional comparisons to handle IP6 addresses
*
*/
SPF_BOOL UTIL_validate_ptr(peer_info_t *peer_info)
{
  char      *ptr_addr;      /* reversed address into PTR format */
  char      *tmp_ptr;       /* utility pointer */

  /* reverse of the address in peer_info */
  ptr_addr = UTIL_rev_addr(peer_info->r_ip);
  xvprintf("[address: %s]\n", ptr_addr);

  tmp_ptr = xstrndup(peer_info->cur_dom, (strlen(peer_info->cur_dom) + 1));

  if (DNS_query(peer_info, ptr_addr, T_PTR, tmp_ptr) != (char *)TRUE)
  {
    xvprintf("PTR lookup failed: (%s) (%s)\n", peer_info->rs,
      peer_info->error);

    xfree(ptr_addr);
    xfree(tmp_ptr);
    return(FALSE);
  }

  xvprintf("Peer (%s) successfully validated hostname (%s)\n",
    peer_info->r_ip, peer_info->r_vhname);

  xfree(ptr_addr);
  xfree(tmp_ptr);
  return(TRUE);
}


/* UTIL_validate_hostname
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/08/04
*
*  Desc:
*          s contains a hostname which is then looked up to find any
*  of the returned ip addresses SPF_PASS the remote peers.  On a successful
*  SPF_PASS returns TRUE.  When no SPF_PASS is made, returns FALSE.
*
*  Note:
*          We need to define and set a limit on the number of recursive
*  lookups.  I recall seeing this in the RFC, we should discuss this.
*
*/
SPF_BOOL UTIL_validate_hostname(peer_info_t *peer_info, const char *s,
  const int8_t cidr)
{
  char            **a;
  char            *val_ip;

  struct in_addr  addr;
  char            *ip = NULL;

  struct in_addr  val_addr;
  struct hostent  *hp;

  policy_addr_t   *policy_addr;

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.\n");
    return(FALSE);
  }

  xvprintf("Called with: (%lu) and (%s)\n", peer_info->r_ip, s);

  if ((hp = gethostbyname(s)) != NULL)
  {
    for (a = hp->h_addr_list; *a; a++)
    {
      memcpy(&addr.s_addr, *a, SIZEOF(struct in_addr));
      ip = xstrndup(inet_ntoa(addr), IP_ADDR);

      xvprintf("CLI: %s (%lu) SRV: %s (%lu)\n", ip, addr.s_addr,
        peer_info->r_ip, peer_info->addr.s_addr);

      if (cidr == 32)
      {
        if (((struct in_addr *)(*a))->s_addr == peer_info->addr.s_addr)
        {
          xvprintf("%s (%lu) matches %s (%lu)\n", ip,
            ((struct in_addr *)(*a))->s_addr, peer_info->r_ip,
            peer_info->addr.s_addr);

          xfree(ip);
          UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
          return(TRUE);
        }
      }
      else if ((cidr < 32) && (cidr >= 8))
      {
        memcpy(&val_addr.s_addr, hp->h_addr, hp->h_length);
        val_ip = xstrndup(inet_ntoa(val_addr), IP_ADDR);

        xvprintf("IP: (%s)\n", val_ip);

        policy_addr = xmalloc(SIZEOF(policy_addr_t));

        if ((inet_pton(AF_INET, val_ip, &policy_addr->addr)) == 0)
        {
          xeprintf("Unable to execute inet_pton()\n");
        }

        policy_addr->cidr = cidr;

        if ((UTIL_cidr_cmp(/*peer_info, */policy_addr, &peer_info->addr))
          == TRUE)
        {
          xvprintf("(%lu) matches (%lu) with CIDR %u\n",
            policy_addr->addr.s_addr, peer_info->addr.s_addr, cidr);

          xfree(ip);
          xfree(policy_addr);
          xfree(val_ip);
          UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
          return(TRUE);
        }
        else
        {
          xfree(policy_addr);
          xfree(val_ip);
        }
      }
      xfree(ip);
    }

    for (a = hp->h_aliases; *a; a++)
    {
      memcpy(&addr.s_addr, *a, SIZEOF(struct in_addr));
      ip = xstrndup(inet_ntoa(addr), IP_ADDR);

      xvprintf("CLI: %s (%lu) SRV: %s (%lu)\n", ip, addr.s_addr,
        peer_info->r_ip, peer_info->addr.s_addr);

      if (cidr == 32)
      {
        if (((struct in_addr *)(*a))->s_addr == peer_info->addr.s_addr)
        {
          xvprintf("%s (%lu) SPF_MATCHes %s (%lu)\n", ip,
            ((struct in_addr *)(*a))->s_addr, peer_info->r_ip,
            peer_info->addr.s_addr);

          xfree(ip);
          return(TRUE);
        }
      }
      else if ((cidr < 32) && (cidr >= 8))
      {
        memcpy(&val_addr.s_addr, hp->h_addr, hp->h_length);
        val_ip = xstrndup(inet_ntoa(val_addr), IP_ADDR);

        xvprintf("ALIAS: (%s)\n",
          val_ip);

        policy_addr = xmalloc(SIZEOF(policy_addr_t));

        if ((inet_pton(AF_INET, val_ip, &policy_addr->addr)) == 0)
        {
          xeprintf("Unable to execute inet_pton()\n");
        }

        policy_addr->cidr = cidr;

        if ((UTIL_cidr_cmp(/*peer_info, */policy_addr, &peer_info->addr))
          == TRUE)
        {
          xvprintf("(%lu) matches (%lu) with CIDR %u\n",
            policy_addr->addr.s_addr, peer_info->addr.s_addr, cidr);

          xfree(ip);
          xfree(policy_addr);
          xfree(val_ip);
          UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
          return(TRUE);
        }
        else
        {
          xfree(policy_addr);
          xfree(val_ip);
        }
      }
      xfree(ip);
    }
  }
  else
  {
    xvprintf("Unable to obtain ip address for (%s)\n", s);
  }

  return(FALSE);
}


/* UTIL_url_encode
*
*  Author: Sean Comeau <scomeau@obscurity.org>
*
*  Date:   01/06/04
*
*  Desc:
*          "URL-Encode" characters that are "unsafe" or "prohibited" from a URL.
*  Upon success returns a pointer to a newly allocated memory containing the
*  encoded string.  Upon failure returns a NULL pointer.  Failure
*  indicates either a failure to allocate new memory, or the passed string
*  did not contain any questionable characters and hence did not require
*  encoding.
*
*/
char *UTIL_url_encode(const char *s)
{
  int   len;       /* length of passed string * 3*/
  char  *new;      /* newly allocated memory */
  char  *encoded;  /* encoded return string */

  if (s != NULL)
  {
    /* length of the string * 3 is the maximum possible size an
       encoded string could ever expand to */
    len = (strlen(s) * 3);
  }
  else
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  encoded = new = xmalloc(len);

  while (*s != '\0')
  {
    if (urlchr_test(*s) == 0)
    {
      *new++ = *s++;
    }
    else
    {
      snprintf(new, 4, "%%%x", *s); /* this NULL terminates do we want? */
      new += 3; s++;
    }
  }

  *new++ = '\0';
  xvprintf("[%s] UTIL_url_encode :: Returning (%s)\n", encoded);

  return(encoded);
}


/* UTIL_reverse
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/06/04
*
*  Desc:
*          Takes a string and rebuilds it in reverse order using the
*  supplied delimiter.  The string will always be rebuilt using a
*  '.' char as per the RFC.
*
*/
char *UTIL_reverse(const char *s, const char delim)
{
  size_t            len;          /* length of s */

  split_str_t       *master;      /* list pointers */
  split_str_node_t  *c_node;      /* c_node node */
  split_str_node_t  *kill_node;   /* temp node used in destruction */

  char              *buf;         /* return buffer */
  char              *p;           /* working pointer */
  char              *cp;          /* working pointer */
  char              *c;           /* working pointer */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  xvprintf("Called with (%s) and delim (%c)\n",
    s, delim);

  len = strlen(s);
  cp = c = xstrndup(s, (len + 1));

  master = xmalloc(SIZEOF(split_str_t));
  master->head = NULL;
  master->tail = NULL;
  master->elements = 0;

  /* Comment: James Couzens <jcouzens@codeshare.ca>
  *  Date: 01/22/04
  *
  * we do not want the trailing delim so the first iteration of this
  * loop we call UTIL_addnode with 0, which tells it not to allocate
  * the extra byte for the trailing delimiter.  This is done only
  * when passed an IP address, and this is because when reversing we
  * want: 1.0.168.192. and not .1.0.168.192
  *
  */

  buf = xmalloc(len + 1);
  memset(buf, '\0', (len + 1));

  while ((p = strrchr(cp, delim)) != NULL)
  {
    p++; /* remove period */
    UTIL_addnode(master, p, TRUE);
    p--; /* bring it back */
    *p = '\0';
  }

  UTIL_addnode(master, cp, FALSE);

  c_node = master->head;
  while ((kill_node = c_node) != NULL)
  {
    strncat(buf, kill_node->s, kill_node->len);
    xfree(kill_node->s);
    c_node = c_node->next;
    xfree(kill_node);
  }

  xfree(cp);
  xfree(master);

  xvprintf("Returning (%s)\n", buf);

  return(buf);
}


/*  UTIL_addnode
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/18/04
*
*  Desc:
*         Allocates memory for a new node and slaps it on the end of
*  of the passed list.  In the process of doing so, when it comes to
*  the last element in the list, which is indicated by the SPF_BOOL
*  that is passed, a final '.' is either appended (TRUE) or not
*  (FALSE).  Function returns TRUE upon success, and FALSE upon
*  failure.
*
*/
SPF_BOOL UTIL_addnode(split_str_t *master, const char *s, SPF_BOOL last)
{
  size_t             len;         /* length of s */

  split_str_node_t   *c_node;     /* c_node working node */
  split_str_node_t   *new_node;   /* newly allocated node */
  split_str_node_t   *prev_node;  /* previous working node */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(FALSE);
  }

  len = strlen(s);

  if (last == TRUE)
  {
    len += 2;
  }
  else
  {
    len++;
  }

  /* create new node */
  new_node = xmalloc(SIZEOF(split_str_node_t));

  /* set the new nodes next value NULL, and store s */
  new_node->next = NULL;

  new_node->s = xmalloc(len);
  memset(new_node->s, '\0', len);

  /* Comment: James Couzens <jcouzens@codeshare.ca>
  *  Date: 01/22/04
  *
  * section 7.1 (Macro definitions) of the RFC v02.9.5 indicates that we must
  * always rebuild the macro using the delimiter '.' and not the passed
  * delimiting character.
  *
  */
  snprintf(new_node->s, len, "%s%c", s, last ? '.' : '\0');
  new_node->len = (len - 1); /* don't count \0 */

  prev_node = NULL;
  c_node = master->head;

  /* reorder the list with the NEW element on the end */
  while (c_node != NULL)
  {
    prev_node = c_node;
    c_node = c_node->next;
  }

  if (prev_node != NULL)
  {
    new_node->next = prev_node->next;
    prev_node->next = new_node;
  }
  else
  {
    master->head = new_node;
  }

  master->tail = new_node;
  master->elements++;

  return(TRUE);
}


/* end of util.c */
