/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffToolbox.cpp
 * Implementation of Spiff::Toolbox.
 */

#include <spiff/SpiffToolbox.h>

namespace Spiff {
namespace Toolbox {



/// @cond DOXYGEN_NON_API

bool SpiffStringCompare::operator()(const XML_Char * s1, const XML_Char * s2) const {
	// Same address implies same content
	// and saves the string comparison
	if (s1 == s2) {
		return false;
	} else {
		return ::PORT_STRCMP(s1, s2) < 0;
	}
}

/// @endcond



XML_Char * newAndCopy(const XML_Char * source) {
	if (source == NULL) {
		return NULL;
	}
	XML_Char * dup = new XML_Char[static_cast<int>(PORT_STRLEN(source)) + 1];
	PORT_STRCPY(dup, source);
	return dup;
}



void deleteNewAndCopy(XML_Char ** dest, const XML_Char * src) {
	if (dest == NULL) {
		return;
	}

	if (*dest != NULL) {
		delete [] *dest;
	}

	if (src == NULL) {
		*dest = NULL;
	} else {
		const int srcLen = static_cast<int>(::PORT_STRLEN(src));
		if (srcLen > 0) {
			*dest = new XML_Char[static_cast<int>(srcLen) + 1];
			PORT_STRCPY(*dest, src);
		} else {
			*dest = NULL;
		}
	}
}



void deleteNewAndCopy(const XML_Char * & dest, bool & destOwnership,
		const XML_Char * source, bool sourceCopy) {
	// Delete old memory if owner
	if (destOwnership && (dest != NULL)) {
		delete [] dest;
	}

	if (source == NULL) {
		dest = NULL;
		destOwnership = false;
	} else {
		// Copy new memory if desired
		if (sourceCopy) {
			const int sourceLen = static_cast<int>(::PORT_STRLEN(source));
			if (sourceLen > 0) {
				XML_Char * const tempDest = new XML_Char[static_cast<int>(sourceLen) + 1];
				PORT_STRCPY(tempDest, source);
				dest = tempDest;
				destOwnership = true;
			} else {
				dest = NULL;
				destOwnership = false;
			}
		} else {
			dest = source;
			destOwnership = false;
		}
	}
}



void copyIfOwned(const XML_Char * & dest, bool & ownDest, const XML_Char * source, bool ownSource) {
	if (source != NULL) {
		// Memory owned?
		if (ownSource) {
			// Deep copy, both own their own copy
			dest = newAndCopy(source);
		} else {
			// Shallow copy, both lend the same memory
			dest = source;
		}
		ownDest = ownSource;
	} else {
		dest = NULL;
	}
}



void freeIfOwned(const XML_Char * & dest, bool ownDest) {
	if (ownDest && (dest != NULL)) {
		delete [] dest;
	}
	// dest = NULL;
}



int PORT_ANTOI(const XML_Char * text, int len) {
	XML_Char * final = new XML_Char[len + 1];
	::PORT_STRNCPY(final, text, len);
	final[len] = _PT('\0');
	const int res = ::PORT_ATOI(final);
	delete [] final;
	return res;
}



}
}
