/* Description.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */
package be.ugent.caagt.swirl.actions;

import java.awt.event.KeyEvent;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;
import javax.swing.UIManager;

/**
 * An object of this class describes caption, mnemonic and accelerator key of a
 * GUI component. It is initialized by a single description string (either at
 * construction time or by calling {@link #setDescription}) which is then
 * internally split into different parts.  For example, the description string
 * "Javadoc inde&amp;x search [shift F1]" will be split into the caption
 * 'Javadoc index search', mnemonic 'X' and accelerator key shift-F1.
 *
 * <p>
 * Accelerator keys, if present, are written between square brackets as the
 * last part of the description string and use the conventions of method
 * {@link javax.swing.KeyStroke#getKeyStroke(String)}. Mnemonics are indicated
 * by placing an ampersand in front of the first occurrence of that letter in
 * the string. The name of the action is obtained by removing from the
 * description string the bracketed accelerator
 * key, any surrounding white space and all ampersands. 
 * Two consecutive ampersands are
 * not interpreted as a mnemonic, but result in a single ampersand in the
 * name.
 * </p>
 *
 * <p>In most cases this class should not be used directly, for buttons and menu
 * items can most easily be internationalized using actions that extend {@link SimpleAction}.
 */
public class Description {
    
    //
    private Description () {
        // empty
    }
    
    /**
     * Create a new object based on the given description string.
     */
    public Description(String description) {
        setDescription(description);
    }
    
    //
    private final static Description OK_BUTTON_DESCRIPTION
            = getFromUIManager("OptionPane.okButton"); 
        // if lazy instantiation is needed, use initailize-on-demand holder class
    
    /**
     * Return a default description for an OK-button. The caption and mnemonics
     * are the same as those used for an option pane in the look-and-feel of
     * the application, and are therefore internationalized.
     */
    public static Description getOkButtonDescription() {
        return OK_BUTTON_DESCRIPTION;
    }
    
    /**
     * Returns a description taken from the UIManager.
     */
    private static Description getFromUIManager(String prefix) {
        Description description = new Description();
        description.caption = UIManager.getString(prefix + "Text");
        description.mnemonicIndex = -1;
        description.mnemonic = 0;
        description.acceleratorKey = null;
        String mstr = UIManager.getString(prefix + "Mnemonic");
        if (mstr != null && mstr.length() != 0)
            try {
                description.mnemonic = Integer.parseInt(mstr);
                description.mnemonicIndex = mstr.indexOf(description.mnemonic);
                if (description.mnemonicIndex <= 0)
                    description.mnemonicIndex = mstr.indexOf(description.mnemonic +
                        'a' - 'A');
            } catch (NumberFormatException ex) {
                throw new RuntimeException 
                        ("Received unexpected nonnumeric mnemonic from UIManager", ex);
            }
        return description;
    }
    
    //
    private static final Description CANCEL_BUTTON_DESCRIPTION
            = getFromUIManager("OptionPane.cancelButton");
            // see OK_BUTTON_DESCRIPTION
    
    /**
     * Return a default description for a Cancel-button. The caption and
     * mnemonics are the same as those used for an option pane in the
     * look-and-feel of the application, and are therefore internationalized.
     */
    public static Description getCancelButtonDescription() {
        return CANCEL_BUTTON_DESCRIPTION;
    }
    
    //
    private static final Description YES_BUTTON_DESCRIPTION
            = getFromUIManager("OptionPane.yesButton");
            // see OK_BUTTON_DESCRIPTION
    
    /**
     * Return a default description for a Yes-button. The caption and mnemonics
     * are the same as those used for an option pane in the look-and-feel of
     * the application, and therefore internationalized.
     */
    public static Description getYesButtonDescription() {
        return YES_BUTTON_DESCRIPTION;
    }
    
    //
    private static final Description NO_BUTTON_DESCRIPTION
            = getFromUIManager("OptionPane.noButton");
            // see OK_BUTTON_DESCRIPTION
    
    /**
     * Return a default description for a No-button. The caption and mnemonics
     * are the same as those used for an option pane in the look-and-feel of
     * the application, and therefore internationalized.
     */
    public static Description getNoButtonDescription() {
        return NO_BUTTON_DESCRIPTION;
    }
    
    /**
     * (Re)initialize this object based on the given description string.
     */
    public final void setDescription(String newDescription) {
        // final because used in constructor
        
        String description = newDescription.trim();
        int len = description.length();
        if (len == 0)
            return;
        
        // determine accelerator key
        acceleratorKey = null;
        if (description.charAt(len - 1) == ']') {
            int pos = description.lastIndexOf('[', len - 1);
            if (pos >= 0) {
                String key = description.substring(pos + 1, len - 1).trim();
                description = description.substring(0, pos).trim();
                acceleratorKey = KeyStroke.getKeyStroke(key);
            }
        }
        
        // create caption and mnemonic
        mnemonicIndex = -1;
        StringBuffer buf = new StringBuffer();
        len = description.length();
        int i = 0;
        int index = 0;
        while (i < len) {
            char ch = description.charAt(i);
            if (ch == '&') {
                i++;
                if (i < len) {
                    ch = description.charAt(i);
                    if (ch >= 'A' && ch <= 'Z') {
                        mnemonic = KeyEvent.VK_A + ch - 'A';
                        mnemonicIndex = index;
                    } else if (ch >= 'a' && ch <= 'z') {
                        mnemonic = KeyEvent.VK_A + ch - 'a';
                        mnemonicIndex = index;
                    }
                }
            }
            buf.append(ch);
            i++;
            index++;
        }
        caption = buf.toString();
    }
    
    //
    private String caption;
    
    /**
     * Return the caption for this description.
     */
    public String getCaption() {
        return this.caption;
    }
    
    //
    private int mnemonic;
    
    /**
     * Return the mnemonic for this description
     */
    public int getMnemonic() {
        return this.mnemonic;
    }
    
    //
    private int mnemonicIndex;
    
    /**
     * Return the index of the mnemonic for this description, or -1 when no
     * mnemonic was given.
     */
    public int getMnemonicIndex() {
        return this.mnemonicIndex;
    }
    
    //
    private KeyStroke acceleratorKey;
    
    /**
     * Return the accelerator key for this description, or 0 when no key was
     * given.
     */
    public KeyStroke getAcceleratorKey() {
        return this.acceleratorKey;
    }
    
    /**
     * Initialize caption, mnemonic and accelerator key for the given action,
     * according to this description.
     */
    public void init(Action action) {
        if (caption != null)
            action.putValue(Action.NAME, caption);
        if (acceleratorKey != null)
            action.putValue(Action.ACCELERATOR_KEY, acceleratorKey);
        if (mnemonicIndex >= 0)
            action.putValue(Action.MNEMONIC_KEY, Integer.valueOf(mnemonic));
    }
    
    /**
     * Initialize caption and mnemonic for the given button, according to this
     * description. If the button is a menu item, also set the accelerator
     * key.
     */
    public void init(AbstractButton button) {
        if (caption != null)
            button.setText(caption);
        if (mnemonicIndex >= 0) {
            button.setMnemonic(mnemonic);
            button.setDisplayedMnemonicIndex(mnemonicIndex);
        }
        if (button instanceof JMenuItem && !(button instanceof JMenu))
            ((JMenuItem)button).setAccelerator(acceleratorKey);
    }
    
    /**
     * Initialize the approve button and title of the given file chooser, according to this
     * description.
     */
    public void initApproveButton (JFileChooser chooser) {
        if (caption != null) {
            chooser.setApproveButtonText(caption);
            chooser.setDialogTitle(caption);
        }
        if (mnemonicIndex >= 0)
            chooser.setApproveButtonMnemonic(mnemonic);
    }
}
