/* ArrayListModel.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */
package be.ugent.caagt.swirl.lists;

import java.util.ArrayList;
import java.util.Collection;
import javax.swing.AbstractListModel;
import javax.swing.event.ListDataListener;


/**
 * Implementation of {@link javax.swing.ListModel} that behaves like an
 * {@link ArrayList} and implements the interface {@link java.util.List} from
 * the collections API.
 */
public class ArrayListModel<E> extends ArrayList<E>
        implements TypedListModel<E> {
    
    /**
     * Inner class which will take care of handling event
     * propagation. This is a workaround for Java not supporting
     * multiple inheritance.
     */
    private static class DummyListModel extends AbstractListModel {
        // this anonymous class is a workaround for Java not
        // supporting multiple inheritance
        
        DummyListModel  () { 
            // avoids creation of accessor class
        }
        
        // never used
        public Object getElementAt(int index) { return null; }
        public int getSize() { return 0; }
        
        // made public instead of protected
        public void fireIntervalRemoved(Object source, int index0, int index1) {
            super.fireIntervalRemoved(source, index0, index1);
        }
        
        // made public instead of protected
        public void fireIntervalAdded(Object source, int index0, int index1) {
            super.fireIntervalAdded(source, index0, index1);
        }
        
        // made public instead of protected
        public void fireContentsChanged(Object source, int index0, int index1) {
            super.fireContentsChanged(source, index0, index1);
        }
    };
    
    //
    private final DummyListModel delegate;
    
    public void removeListDataListener(ListDataListener l) {
        delegate.removeListDataListener(l);
    }
    
    public void addListDataListener(ListDataListener l) {
        delegate.addListDataListener(l);
    }
    
    /**
     * Called <b>after</b> one or more elements are removed from the model. 
     * <code>index0</code> and <code>index1</code> are the end points
     * of the interval that's been removed.  Note that <code>index0</code>
     * need not be less than or equal to <code>index1</code>.
     * 
     * @param source the <code>ArrayListModel</code> that changed, typically "this"
     * @param index0 one end of the removed interval,
     *               including <code>index0</code>
     * @param index1 the other end of the removed interval,
     *               including <code>index1</code>
     * @see AbstractListModel
     */
    protected void fireIntervalRemoved(Object source, int index0, int index1) {
        delegate.fireIntervalRemoved(source, index0, index1);
    }
    
    /**
     * Called <b>after</b>
     * one or more elements are added to the model.  The new elements
     * are specified by a closed interval index0, index1 (endpoints
     * included).  Note that
     * index0 need not be less than or equal to index1.
     * 
     * @param source the <code>ArrayListModel</code> that changed, typically "this"
     * @param index0 one end of the new interval
     * @param index1 the other end of the new interval
     * @see AbstractListModel
     */
    protected void fireIntervalAdded(Object source, int index0, int index1) {
        delegate.fireIntervalAdded(source, index0, index1);
    }
    
    /**
     * Called <b>after</b>
     * one or more elements of the list change.  The changed elements
     * are specified by the closed interval index0, index1 (endpoints
     * included).  Note that
     * index0 need not be less than or equal to index1.
     *
     * @param source the <code>ArrayListModel</code> that changed, typically "this"
     * @param index0 one end of the interval
     * @param index1 the other end of the interval
     * @see AbstractListModel
     */
    protected void fireContentsChanged(Object source, int index0, int index1) {
        delegate.fireContentsChanged(source, index0, index1);
    }
    
    /**
     * Constructs an empty list with the specified initial capacity.
     */
    public ArrayListModel(int initialCapacity) {
        super(initialCapacity);
        delegate = new DummyListModel();
    }
    
    /**
     * Constructs an empty list with an initial capacity of ten.
     */
    public ArrayListModel() {
        this(10);
    }
    
    /**
     * Constructs a list containing the elements of the specified
     * collection, in the order they are returned by the collection's
     * iterator.  The {@code ArrayListModel} instance has an initial capacity of
     * 110% the size of the specified collection.
     *
     * @param c the collection whose elements are to be placed into this list.
     * @throws NullPointerException if the specified collection is null.
     */
    public ArrayListModel(Collection<? extends E> c) {
        super(c);
        delegate = new DummyListModel();
    }
    
    // implements ListModel
    public int getSize() {
        return super.size();
    }
    
    // implements ListModel
    public Object getElementAt(int index) {
        return get(index);
    }
    
    // implements List
    public boolean add(E o) {
        int index = super.size();
        super.add(o);
        delegate.fireIntervalAdded(this, index, index);
        return true;
    }
    
    // implements List
    public void add(int index, E element) {
        super.add(index, element);
        delegate.fireIntervalAdded(this, index, index);
    }
    
    // implements List
    public boolean addAll(Collection<? extends E> c) {
        int index = super.size();
        if (super.addAll(c)) {
            delegate.fireIntervalAdded(this, index, super.size() - 1);
            return true;
        } else
            return false;
    }
    
    // implements List
    public boolean addAll(int index, Collection<? extends E> c) {
        int size = super.size();
        if (super.addAll(index,  c)) {
            delegate.fireIntervalAdded(this, index, index + super.size() - size - 1);
            return true;
        } else
            return false;
    }
    
    // implements List
    public void clear() {
        int index1 = super.size() - 1;
        super.clear();
        if (index1 >= 0)
            delegate.fireIntervalRemoved(this, 0, index1);
    }
    
    // implements List
    public boolean remove(Object obj) {
        int index = indexOf(obj);
        if (index >= 0) {
            super.remove(obj);
            delegate.fireIntervalRemoved(this, index, index);
            return true;
        } else
            return false;
    }
    
    // implements List
    public E remove(int index) {
        E rv = super.remove(index);
        delegate.fireIntervalRemoved(this, index, index);
        return rv;
    }
    
    // implements List
    protected void removeRange(int fromIndex, int toIndex) {
        super.removeRange(fromIndex, toIndex);
        if (fromIndex != toIndex)
            delegate.fireIntervalRemoved(this, fromIndex, toIndex-1);
    }
    
    // implements List
    public E set(int index, E element) {
        E rv = super.set(index, element);
        delegate.fireContentsChanged(this, index, index);
        return rv;
    }
}
