/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "support.h"

#include <libsyncml/objects/sml_auth.h>
#include <libsyncml/objects/sml_devinf_obj.h>

#include <libsyncml/transports/http_client.h>
#include <libsyncml/transports/http_server.h>

#define NUM_SESSIONS 30

typedef struct managerTracker {
	SmlManager *manager;
	SmlSession *session;
} managerTracker;

unsigned int transport_errors = 0;
unsigned int num_sessions = 0;
unsigned int num_finals = 0;
unsigned int num_end = 0;
unsigned int session_errors = 0;

static void _manager_event(SmlManager *manager, SmlManagerEventType type, SmlSession *session, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p)", __func__, manager, type, session, error, userdata);
	managerTracker *tracker = userdata;
	smlAssert(manager);
	smlAssert(userdata);
	
	switch (type) {
		case SML_MANAGER_SESSION_FLUSH:
			break;
		case SML_MANAGER_CONNECT_DONE:
		case SML_MANAGER_DISCONNECT_DONE:
			fail(NULL);
			break;
		case SML_MANAGER_TRANSPORT_ERROR:
			transport_errors++;
			break;
		case SML_MANAGER_SESSION_NEW:
			smlAssert(session);
			tracker->session = session;
			num_sessions++;
			smlSessionRef(session);
			break;
		case SML_MANAGER_SESSION_FINAL:
			num_finals++;
			break;
		case SML_MANAGER_SESSION_END:
			num_end++;
			break;
		case SML_MANAGER_SESSION_ERROR:
		case SML_MANAGER_SESSION_WARNING:
			session_errors++;
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

START_TEST (devinf_crash)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	/* The devinf obj */
	SmlDevInf *devinf = smlDevInfNew("LibSyncmML", SML_DEVINF_DEVTYPE_WORKSTATION, &error);
	SmlDevInfAgent *agent = smlDevInfAgentNew(devinf, &error);
	smlDevInfAgentRegister(agent, servermanager, &error);

	/* And we also add the devinfo to the devinf agent */
	SmlDevInfDataStore *datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	
	smlDevInfAddDataStore(devinf, datastore);

	smlLocationUnref(loc);
	
	
	char *datastr = "<?xml version=\"1.0\"?><!DOCTYPE SyncML PUBLIC \"-//SYNCML//DTD SyncML 1.1//EN\" \"http://www.syncml.org/docs/syncml_represent_v11_20020213.dtd\"><SyncML xmlns=\"syncml:SYNCML1.1\"><SyncHdr><VerDTD>1.1</VerDTD><VerProto>SyncML/1.1</VerProto><SessionID>10</SessionID><MsgID>1</MsgID><Target><LocURI>PC Suite</LocURI></Target><Source><LocURI>IMEI:355661000507384</LocURI></Source><Meta><MaxMsgSize xmlns=\"syncml:metinf\">10000</MaxMsgSize></Meta></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>SyncHdr</Cmd><TargetRef>/</TargetRef><SourceRef>PC Suite</SourceRef><Data>200</Data></Status><Status><CmdID>2</CmdID><MsgRef>1</MsgRef><CmdRef>1</CmdRef><Cmd>Alert</Cmd><SourceRef>Contacts</SourceRef><Data>200</Data></Status><Alert><CmdID>3</CmdID><Data>201</Data><Item><Target><LocURI>./Contacts</LocURI></Target><Source><LocURI>./C\\System\\Data\\Contacts.cdb</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Last></Last><Next>20060317T083025Z</Next></Anchor></Meta></Item></Alert><Put><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">application/vnd.syncml-devinf+xml</Type></Meta><Item><Source><LocURI>./devinf11</LocURI></Source><Data><DevInf xmlns=\"syncml:devinf\"><VerDTD>1.1</VerDTD><Man>NOKIA</Man><Mod>6680</Mod><SwV>2.04.15</SwV><DevID>IMEI:355661000507384</DevID><DevTyp>phone</DevTyp><SupportLargeObjs></SupportLargeObjs><SupportNumberOfChanges></SupportNumberOfChanges><DataStore><SourceRef>./C\\System\\Data\\Contacts.cdb</SourceRef><MaxGUIDSize>8</MaxGUIDSize><Rx-Pref><CTType>text/x-vcard</CTType><VerCT>2.1</VerCT></Rx-Pref><Rx><CTType>text/vcard</CTType><VerCT>3.0</VerCT></Rx><Tx-Pref><CTType>text/x-vcard</CTType><VerCT>2.1</VerCT></Tx-Pref><SyncCap><SyncType>1</SyncType><SyncType>2</SyncType><SyncType>3</SyncType><SyncType>4</SyncType><SyncType>5</SyncType><SyncType>6</SyncType><SyncType>7</SyncType></SyncCap></DataStore><CTCap><CTType>text/x-vcard</CTType><PropName>BEGIN</PropName><ValEnum>VCARD</ValEnum><PropName>END</PropName><ValEnum>VCARD</ValEnum><PropName>VERSION</PropName><ValEnum>2.1</ValEnum><PropName>REV</PropName><PropName>N</PropName><PropName>ADR</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>TEL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><ParamName>CELL</ParamName><ParamName>PAGER</ParamName><ParamName>FAX</ParamName><ParamName>VIDEO</ParamName><PropName>FN</PropName><PropName>EMAIL</PropName><ParamName>INTERNET</ParamName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>URL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>NOTE</PropName><PropName>TITLE</PropName><PropName>ORG</PropName><PropName>PHOTO</PropName><PropName>BDAY</PropName><PropName>SOUND</PropName><ParamName>X-IRMC-N</ParamName><CTType>text/vcard</CTType><PropName>BEGIN</PropName><ValEnum>VCARD</ValEnum><PropName>END</PropName><ValEnum>VCARD</ValEnum><PropName>VERSION</PropName><ValEnum>3.0</ValEnum><PropName>REV</PropName><PropName>N</PropName><PropName>ADR</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>TEL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><ParamName>CELL</ParamName><ParamName>PAGER</ParamName><ParamName>FAX</ParamName><ParamName>VIDEO</ParamName><PropName>FN</PropName><PropName>EMAIL</PropName><ParamName>INTERNET</ParamName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>URL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>NOTE</PropName><PropName>TITLE</PropName><PropName>ORG</PropName><PropName>PHOTO</PropName><PropName>BDAY</PropName><PropName>SOUND</PropName><ParamName>X-IRMC-N</ParamName></CTCap></DevInf></Data></Item></Put><Get><CmdID>5</CmdID><Meta><Type xmlns=\"syncml:metinf\">application/vnd.syncml-devinf+xml</Type></Meta><Item><Target><LocURI>./devinf11</LocURI></Target></Item></Get><Final></Final></SyncBody></SyncML>";
	
	SmlTransportData *data = smlTransportDataNew(datastr, strlen(datastr), SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	while (num_finals != 1) {
		smlManagerDispatch(servermanager);
		usleep(100);
	}
	
	SmlDevInf *devinf2 = smlDevInfAgentGetDevInf(agent);
	fail_unless(devinf2 != NULL, NULL);
	
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 1, NULL);
	fail_unless(num_finals == 1, NULL);
	fail_unless(num_end == 0, NULL);
	fail_unless(session_errors == 1, NULL);
	
	g_free(servertracker);
	
	smlManagerStop(servermanager);
	
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

/* client -- devinf get --> server
 * server -- result --> client
 * client -- status --> server
 * server -- status --> client
 */
START_TEST (devinf_result)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;
	
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	/* The devinf obj */
	SmlDevInf *devinf = smlDevInfNew("LibSyncmML", SML_DEVINF_DEVTYPE_WORKSTATION, &error);
	SmlDevInfAgent *clientagent = smlDevInfAgentNew(devinf, &error);
	smlDevInfAgentRegister(clientagent, clientmanager, &error);

	/* And we also add the devinfo to the devinf agent */
	SmlDevInfDataStore *datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc1), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	devinf = smlDevInfNew("LibSyncmML", SML_DEVINF_DEVTYPE_SERVER, &error);
	SmlDevInfAgent *serveragent = smlDevInfAgentNew(devinf, &error);
	smlDevInfAgentRegister(serveragent, servermanager, &error);
	
	datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc1), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	smlLocationUnref(loc1);
	
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	clienttracker->session = smlSessionNew(SML_SESSION_TYPE_CLIENT, SML_MIMETYPE_XML, SML_VERSION_11, SML_PROTOCOL_SYNCML, loc, loc, NULL, 0, &error);
	
	fail_unless(smlManagerSessionAdd(clientmanager, clienttracker->session, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlDevInfAgentRequestDevInf(clientagent, clienttracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlDevInfAgentSendDevInf(clientagent, clienttracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (num_sessions != 2 || !smlDevInfAgentGetDevInf(serveragent)) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	SmlDevInf *recvClientDevInf = smlDevInfAgentGetDevInf(serveragent);
	fail_unless(recvClientDevInf != NULL, NULL);
	fail_unless(smlDevInfNumDataStores(recvClientDevInf) == 1, NULL);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	while (!smlDevInfAgentGetDevInf(clientagent)) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	SmlDevInf *recvServerDevInf = smlDevInfAgentGetDevInf(clientagent);
	fail_unless(recvServerDevInf != NULL, NULL);
	fail_unless(smlDevInfNumDataStores(recvServerDevInf) == 1, NULL);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(clienttracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (num_end != 2 || num_finals != 4) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlSessionUnref(servertracker->session);
	smlSessionUnref(clienttracker->session);
	
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 4, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	smlAuthFree(auth);
	smlAuthFree(auth2);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

Suite *devinf_suite(void)
{
	Suite *s = suite_create("Devinf");
	//Suite *s2 = suite_create("Devinf");
	
	create_case(s, "devinf_crash", devinf_crash);
	create_case(s, "devinf_result", devinf_result);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = devinf_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_NORMAL);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
