/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#include <libsyncml/sml_command_internals.h>
#include <libsyncml/sml_elements_internals.h>

START_TEST (check_parser_empty)
{
	char *data = " ";
	SmlParser *parser = start_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_syncml_empty)
{
	char *data = "<SyncML></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_syncml_wrong_case)
{
	char *data = "<SyncMl></SyncMl>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_syncml_additional)
{
	char *data = "<SyncML></SyncML><test></test>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_no_syncbody)
{
	char *data = "<SyncML><SyncHdr></SyncHdr></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_no_synchdr)
{
	char *data = "<SyncML><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_synchdr_empty)
{
	char *data = "<SyncML><SyncHdr></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_verproto)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_verdtd)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_msgid)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_sessionid)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_target)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_source)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	fail_unless(header->target != NULL, NULL);
	fail_unless(header->source != NULL, NULL);
	fail_unless(header->protocol == SML_PROTOCOL_SYNCML, NULL);
	fail_unless(header->version == SML_VERSION_11, NULL);
	fail_unless(!strcmp(header->sessionID, "1"), NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_source2)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>asd</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	fail_unless(header->target != NULL, NULL);
	fail_unless(header->source != NULL, NULL);
	fail_unless(header->protocol == SML_PROTOCOL_SYNCML, NULL);
	fail_unless(header->version == SML_VERSION_11, NULL);
	fail_unless(!strcmp(header->sessionID, "asd"), NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_additional)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source><test></test></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_empty)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status></Status></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_cmdid)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID></Status></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);

	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_msgref)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef></Status></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_cmdref)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>1</CmdRef></Status></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status2)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = TRUE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == FALSE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status3)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Final/></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = FALSE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == TRUE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_final)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = FALSE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == TRUE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_final2)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Final/></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = FALSE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == TRUE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_cmd)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Alert><CmdID>1</CmdID><Item><Target><LocURI>/test</LocURI></Target><Source><LocURI>/test</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Next>Next</Next><Last>last</Last></Anchor></Meta></Item><Data>200</Data></Alert></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = TRUE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == FALSE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_cmd_final)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Alert><CmdID>1</CmdID><Item><Target><LocURI>/test</LocURI></Target><Source><LocURI>/test</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Next>Next</Next><Last>last</Last></Anchor></Meta></Item><Data>200</Data></Alert><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_sync_subcommand)
{
	char *data = "<SyncML xmlns=\"SYNCML:SYNCML1.2\"><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.2</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source></SyncHdr><SyncBody><Sync><CmdID>3</CmdID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source><NumberOfChanges>1</NumberOfChanges><Add><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">text/x-vcard</Type></Meta><Item><Source><LocURI>uid</LocURI></Source><Data><![CDATA[data]]></Data></Item></Add></Sync><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OPEN, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 3, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_SYNC, NULL);
	fail_unless(cmd->private.sync.numChanged == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 4, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_ADD, NULL);
	fail_unless(cmd->private.change.item != NULL, NULL);
	fail_unless(cmd->private.change.type == SML_CHANGE_ADD, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_CLOSE, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OTHER, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST


START_TEST (check_parser_emi)
{
	char *data = "<?xml version=\"1.0\"?><!DOCTYPE SyncML PUBLIC \"-//SYNCML//DTD SyncML 1.1//EN\" \"http://www.syncml.org/docs/syncml_represent_v11_20020213.dtd\"><SyncML xmlns=\"syncml:SYNCML1.1\"><SyncHdr><VerDTD>1.1</VerDTD><VerProto>SyncML/1.1</VerProto><SessionID>1</SessionID><MsgID>1</MsgID><Target><LocURI>opium</LocURI></Target><Source><LocURI>IMEI:355020000873652</LocURI></Source><Meta><MaxMsgSize xmlns=\"syncml:metinf\">10000</MaxMsgSize><EMI xmlns=\"syncml:metinf\">NokiaInq/1.1</EMI></Meta></SyncHdr><SyncBody><Status><CmdID>1</CmdID><Cmd>SyncHdr</Cmd><Data>403</Data></Status><Status><CmdID>2</CmdID><CmdRef>1</CmdRef><Cmd>Alert</Cmd><Data>500</Data></Status><Status><CmdID>3</CmdID><CmdRef>2</CmdRef><Cmd>Alert</Cmd><Data>500</Data></Status><Status><CmdID>4</CmdID><CmdRef>3</CmdRef><Cmd>Alert</Cmd><Data>500</Data></Status><Final/></SyncBody></SyncML>";

	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(header->messageID == 1, NULL);

	smlHeaderFree(header);
	
	smlParserFree(parser);
}
END_TEST



START_TEST (check_parser_doctype)
{
	char *data = "<?xml version=\"1.0\"?>\n<!DOCTYPE SyncML PUBLIC \"-//SYNCML//DTD SyncML 1.1//EN\" \"http://www.syncml.org/docs/syncml_represent_v11_20020213.dtd\">\n<SyncML xmlns=\"syncml:SYNCML1.1\">\n<SyncHdr>\n<VerDTD>\n1.1\n</VerDTD>\n<VerProto>\nSyncML/1.1\n</VerProto>\n<SessionID>\n1125161435\n</SessionID>\n<MsgID>\n1\n</MsgID>\n<Target>\n<LocURI>\nhttp://test.com:7962/\n</LocURI>\n</Target>\n<Source>\n<LocURI>\n35686843434\n</LocURI>\n<LocName>\nasd\n</LocName>\n</Source>\n<Cred>\n<Meta>\n<Format xmlns=\"syncml:metinf\">\nb64\n</Format>\n<Type xmlns=\"syncml:metinf\">\nsyncml:auth-basic\n</Type>\n</Meta>\n<Data>\nYXNkOmFzZA==\n</Data>\n</Cred>\n<Meta>\n<MaxMsgSize xmlns=\"syncml:metinf\">\n3300\n</MaxMsgSize>\n</Meta>\n</SyncHdr>\n<SyncBody>\n<Alert>\n<CmdID>\n1\n</CmdID>\n<Data>\n201\n</Data>\n<Item>\n<Target>\n<LocURI>\npeople\n</LocURI>\n</Target>\n<Source>\n<LocURI>\npeople\n</LocURI>\n</Source>\n<Meta>\n<Anchor xmlns=\"syncml:metinf\">\n<Last>\n</Last>\n<Next>\n1\n</Next>\n</Anchor>\n</Meta>\n</Item>\n</Alert>\n<Put>\n<CmdID>\n2\n</CmdID>\n<Meta>\n<Format xmlns=\"syncml:metinf\">\nwbxml\n</Format>\n<Type xmlns=\"syncml:metinf\">\napplication/vnd.syncml-devinf+xml\n</Type>\n</Meta>\n<Item>\n<Source>\n<LocURI>\n./devinf11\n</LocURI>\n</Source>\n<Data>\n<DevInf xmlns=\"syncml:devinf\">\n<VerDTD>\n1.1\n</VerDTD>\n<Man>\nMotorola CE, Copyright 2004\n</Man>\n<Mod>\nV3\n</Mod>\n<SwV>\nR374_G_0E.41.C3R_A\n</SwV>\n<HwV>\nSJUG0851CA\n</HwV>\n<DevID>\n3564564556453\n</DevID>\n<DevTyp>\nphone\n</DevTyp>\n<DataStore>\n<SourceRef>\npeople\n</SourceRef>\n<Rx-Pref>\n<CTType>\ntext/x-vcard\n</CTType>\n<VerCT>\n2.1\n</VerCT>\n</Rx-Pref>\n<Tx-Pref>\n<CTType>\ntext/x-vcard\n</CTType>\n<VerCT>\n2.1\n</VerCT>\n</Tx-Pref>\n<SyncCap>\n<SyncType>\n1\n</SyncType>\n<SyncType>\n2\n</SyncType>\n</SyncCap>\n</DataStore>\n<CTCap>\n<CTType>\ntext/x-vcard\n</CTType>\n<PropName>\nBEGIN\n</PropName>\n<ValEnum>\nVCARD\n</ValEnum>\n<PropName>\nEND\n</PropName>\n<ValEnum>\nVCARD\n</ValEnum>\n<PropName>\nVERSION\n</PropName>\n<ValEnum>\n2.1\n</ValEnum>\n<PropName>\nFN\n</PropName>\n<PropName>\nN\n</PropName>\n<PropName>\nTEL\n</PropName>\n<ParamName>\nPREF\n</ParamName>\n<ParamName>\nWORK\n</ParamName>\n<ParamName>\nHOME\n</ParamName>\n<ParamName>\nCELL\n</ParamName>\n<ParamName>\nFAX\n</ParamName>\n<ParamName>\nPAGER\n</ParamName>\n<PropName>\nEMAIL\n</PropName>\n<ParamName>\nINTERNET\n</ParamName>\n<PropName>\nCHARSET\n</PropName>\n</CTCap>\n</DevInf>\n</Data>\n</Item>\n</Put>\n<Final>\n</Final>\n</SyncBody>\n</SyncML>\n";

	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	smlCredFree(cred);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OTHER, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_large_obj)
{
	char *data = "<SyncML xmlns=\"SYNCML:SYNCML1.2\"><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.2</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source></SyncHdr><SyncBody><Sync><CmdID>3</CmdID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source><NumberOfChanges>1</NumberOfChanges><Add><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">text/x-vcard</Type><Size>200</Size></Meta><Item><Source><LocURI>uid</LocURI></Source><Data><![CDATA[data]]></Data><MoreData></MoreData></Item></Add></Sync><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OPEN, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 3, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_SYNC, NULL);
	fail_unless(cmd->private.sync.numChanged == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 4, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_ADD, NULL);
	fail_unless(cmd->private.change.item != NULL, NULL);
	fail_unless(cmd->private.change.type == SML_CHANGE_ADD, NULL);
	fail_unless(cmd->private.change.item->moreData == TRUE, NULL);
	fail_unless(cmd->size == 200, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_CLOSE, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OTHER, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_large_obj2)
{
	char *data = "<SyncML xmlns=\"SYNCML:SYNCML1.2\"><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.2</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source></SyncHdr><SyncBody><Sync><CmdID>3</CmdID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source><NumberOfChanges>1</NumberOfChanges><Add><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">text/x-vcard</Type><Size>200</Size></Meta><Item><Source><LocURI>uid</LocURI></Source><Data><![CDATA[data]]></Data><MoreData/></Item></Add></Sync><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OPEN, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 3, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_SYNC, NULL);
	fail_unless(cmd->private.sync.numChanged == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 4, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_ADD, NULL);
	fail_unless(cmd->private.change.item != NULL, NULL);
	fail_unless(cmd->private.change.type == SML_CHANGE_ADD, NULL);
	fail_unless(cmd->private.change.item->moreData == TRUE, NULL);
	fail_unless(cmd->size == 200, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_CLOSE, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OTHER, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_large_obj_error)
{
	char *data = "<SyncML xmlns=\"SYNCML:SYNCML1.2\"><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.2</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source></SyncHdr><SyncBody><Sync><CmdID>3</CmdID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source><NumberOfChanges>1</NumberOfChanges><Add><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">text/x-vcard</Type><Size>200</Size></Meta><Item><Source><LocURI>uid</LocURI></Source><Data><![CDATA[data]]></Data><MoreData></MoreData></Item></Add><Add><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">text/x-vcard</Type><Size>200</Size></Meta><Item><Source><LocURI>uid</LocURI></Source><Data><![CDATA[data]]></Data></Item></Add></Sync><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OPEN, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 3, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_SYNC, NULL);
	fail_unless(cmd->private.sync.numChanged == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 4, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_ADD, NULL);
	fail_unless(cmd->private.change.item != NULL, NULL);
	fail_unless(cmd->private.change.type == SML_CHANGE_ADD, NULL);
	fail_unless(cmd->private.change.item->moreData == TRUE, NULL);
	fail_unless(cmd->size == 200, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_ERROR, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_ERROR, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(!smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_large_obj_mangled)
{
	char *data = "<SyncML xmlns=\"SYNCML:SYNCML1.2\"><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.2</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source></SyncHdr><SyncBody><Sync><CmdID>3</CmdID><Target><LocURI>/vcards</LocURI></Target><Source><LocURI>/vcards</LocURI></Source><NumberOfChanges>1</NumberOfChanges><Add><CmdID>4</CmdID><Meta><Type xmlns=\"syncml:metinf\">text/x-vcard</Type></Meta><Item><Source><LocURI>uid</LocURI></Source><Meta><Size>10</Size></Meta><Data><![CDATA[data\r\n\r\nda]]></Data><MoreData></MoreData></Item></Add></Sync><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OPEN, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 3, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_SYNC, NULL);
	fail_unless(cmd->private.sync.numChanged == 1, NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 4, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_ADD, NULL);
	fail_unless(cmd->private.change.item != NULL, NULL);
	fail_unless(cmd->private.change.type == SML_CHANGE_ADD, NULL);
	fail_unless(cmd->private.change.item->moreData == TRUE, NULL);
	fail_unless(cmd->size == 10, NULL);
	
	char *data2 = NULL;
	unsigned int size = 0;
	fail_unless(smlItemGetData(cmd->private.change.item, &data2, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strcmp(data2, "data\r\n\r\nda"), NULL);
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_CLOSE, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OTHER, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_devinf)
{
	char *data = "<?xml version=\"1.0\"?><!DOCTYPE SyncML PUBLIC \"-//SYNCML//DTD SyncML 1.1//EN\" \"http://www.syncml.org/docs/syncml_represent_v11_20020213.dtd\"><SyncML xmlns=\"syncml:SYNCML1.1\"><SyncHdr><VerDTD>1.1</VerDTD><VerProto>SyncML/1.1</VerProto><SessionID>21</SessionID><MsgID>1</MsgID><Target><LocURI>PC Suite</LocURI></Target><Source><LocURI>IMEI:XXXX</LocURI></Source><Meta><MaxMsgSize xmlns=\"syncml:metinf\">10000</MaxMsgSize></Meta></SyncHdr><SyncBody><Put><CmdID>8</CmdID><Meta><Type xmlns=\"syncml:metinf\">application/vnd.syncml-devinf+xml</Type></Meta><Item><Source><LocURI>./devinf11</LocURI></Source><Data><DevInf xmlns=\"syncml:devinf\"><VerDTD>1.1</VerDTD><Man>NOKIA</Man><Mod>6680</Mod><SwV>3.04.37</SwV><DevID>IMEI:XXXX</DevID><DevTyp>phone</DevTyp><SupportLargeObjs/><SupportNumberOfChanges/><DataStore><SourceRef>./C\\Systemata\\Calendar</SourceRef><MaxGUIDSize>8</MaxGUIDSize><Rx-Pref><CTType>text/x-vcalendar</CTType><VerCT>1.0</VerCT></Rx-Pref><Rx><CTType>text/calendar</CTType><VerCT>2.0</VerCT></Rx><Tx-Pref><CTType>text/x-vcalendar</CTType><VerCT>1.0</VerCT></Tx-Pref><SyncCap><SyncType>1</SyncType><SyncType>2</SyncType><SyncType>3</SyncType><SyncType>4</SyncType><SyncType>5</SyncType><SyncType>6</SyncType><SyncType>7</SyncType></SyncCap></DataStore><DataStore><SourceRef>./C\\System\\\\\\Data\\Contacts.cdb</SourceRef><MaxGUIDSize>8</MaxGUIDSize><Rx-Pref><CTType>text/x-vcard</CTType><VerCT>2.1</VerCT></Rx-Pref><Rx><CTType>text/vcard</CTType><VerCT>3.0</VerCT></Rx><Tx-Pref><CTType>text/x-vcard</CTType><VerCT>2.1</VerCT></Tx-Pref><SyncCap><SyncType>1</SyncType><SyncType>2</SyncType><SyncType>3</SyncType><SyncType>4</SyncType><SyncType>5</SyncType><SyncType>6</SyncType><SyncType>7</SyncType></SyncCap></DataStore><DataStore><SourceRef>./C\\System\\Data\\Notepad.dat</SourceRef><MaxGUIDSize>8</MaxGUIDSize><Rx-Pref><CTType>text/plain</CTType><VerCT>1.0</VerCT></Rx-Pref><Rx><CTType>text/plain</CTType><VerCT>1.0</VerCT></Rx><Tx-Pref><CTType>text/plain</CTType><VerCT>1.0</VerCT></Tx-Pref><Tx><CTType>text/plain</CTType><VerCT>1.0</VerCT></Tx><SyncCap><SyncType>1</SyncType><SyncType>2</SyncType><SyncType>3</SyncType><SyncType>4</SyncType><SyncType>5</SyncType><SyncType>6</SyncType><SyncType>7</SyncType></SyncCap></DataStore><CTCap><CTType>text/x-vcalendar</CTType><PropName>BEGIN</PropName><ValEnum>VCALENDAR</ValEnum><ValEnum>VEVENT</ValEnum><ValEnum>VTODO</ValEnum><PropName>END</PropName><ValEnum>VCALENDAR</ValEnum><ValEnum>VEVENT</ValEnum><ValEnum>VTODO</ValEnum><PropName>VERSION</PropName><ValEnum>1.0</ValEnum><PropName>UID</PropName><PropName>SUMMARY</PropName><PropName>DESCRIPTION</PropName><PropName>DTEND</PropName><PropName>DTSTART</PropName><PropName>AALARM</PropName><ParamName>TYPE</ParamName><PropName>CLASS</PropName><ValEnum>PUBLIC</ValEnum><ValEnum>PRIVATE</ValEnum><ValEnum>CONFIDENTIAL</ValEnum><PropName>COMPLETED</PropName><PropName>LOCATION</PropName><PropName>DCREATED</PropName><PropName>LAST-MODIFIED</PropName><PropName>PRIORITY</PropName><PropName>STATUS</PropName><PropName>RRULE</PropName><PropName>DUE</PropName><PropName>EXDATE</PropName><PropName>X-EPOCAGENDAENTRYTYPE</PropName><CTType>text/calendar</CTType><PropName>BEGIN</PropName><ValEnum>VCALENDAR</ValEnum><ValEnum>VEVENT</ValEnum><ValEnum>VTODO</ValEnum><PropName>END</PropName><ValEnum>VCALENDAR</ValEnum><ValEnum>VEVENT</ValEnum><ValEnum>VTODO</ValEnum><PropName>VERSION</PropName><ValEnum>2.0</ValEnum><PropName>UID</PropName><PropName>SUMMARY</PropName><PropName>DESCRIPTION</PropName><PropName>DTEND</PropName><PropName>DTSTART</PropName><PropName>AALARM</PropName><ParamName>TYPE</ParamName><PropName>CLASS</PropName><ValEnum>PUBLIC</ValEnum><ValEnum>PRIVATE</ValEnum><ValEnum>CONFIDENTIAL</ValEnum><PropName>COMPLETED</PropName><PropName>LOCATION</PropName><PropName>DCREATED</PropName><PropName>LAST-MODIFIED</PropName><PropName>PRIORITY</PropName><PropName>STATUS</PropName><PropName>RRULE</PropName><PropName>DUE</PropName><PropName>EXDATE</PropName><PropName>X-EPOCAGENDAENTRYTYPE</PropName><CTType>text/x-vcard</CTType><PropName>BEGIN</PropName><ValEnum>VCARD</ValEnum><PropName>END</PropName><ValEnum>VCARD</ValEnum><PropName>VERSION</PropName><ValEnum>2.1</ValEnum><PropName>REV</PropName><PropName>N</PropName><PropName>ADR</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>TEL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><ParamName>CELL</ParamName><ParamName>PAGER</ParamName><ParamName>FAX</ParamName><ParamName>VIDEO</ParamName><PropName>FN</PropName><PropName>EMAIL</PropName><ParamName>INTERNET</ParamName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>URL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>NOTE</PropName><PropName>TITLE</PropName><PropName>ORG</PropName><PropName>PHOTO</PropName><PropName>BDAY</PropName><PropName>SOUND</PropName><ParamName>X-IRMC-N</ParamName><CTType>text/vcard</CTType><PropName>BEGIN</PropName><ValEnum>VCARD</ValEnum><PropName>END</PropName><ValEnum>VCARD</ValEnum><PropName>VERSION</PropName><ValEnum>3.0</ValEnum><PropName>REV</PropName><PropName>N</PropName><PropName>ADR</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>TEL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><ParamName>CELL</ParamName><ParamName>PAGER</ParamName><ParamName>FAX</ParamName><ParamName>VIDEO</ParamName><PropName>FN</PropName><PropName>EMAIL</PropName><ParamName>INTERNET</ParamName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>URL</PropName><ParamName>HOME</ParamName><ParamName>WORK</ParamName><PropName>NOTE</PropName><PropName>TITLE</PropName><PropName>ORG</PropName><PropName>PHOTO</PropName><PropName>BDAY</PropName><PropName>SOUND</PropName><ParamName>X-IRMC-N</ParamName><CTType>text/plain</CTType></CTCap></DevInf></Data></Item></Put><Final/></SyncBody></SyncML>";
	
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_NORMAL, NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 8, NULL);
	fail_unless(cmd->type == SML_COMMAND_TYPE_PUT, NULL);
	fail_unless(cmd->private.access.item != NULL, NULL);
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlItemGetData(cmd->private.access.item, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlDevInf *devinf = smlDevInfParse(buffer, size, &error);
	fail_unless(devinf != NULL, NULL);
	fail_unless(error == NULL, NULL);
	smlDevInfUnref(devinf);
	
	smlCommandUnref(cmd);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error) == SML_PARSER_RESULT_OTHER, NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

Suite *parser_suite(void)
{
	Suite *s = suite_create("Parser");
	//Suite *s2 = suite_create("Parser");
	
	create_case(s, "check_parser_empty", check_parser_empty);
	create_case(s, "check_parser_syncml_empty", check_parser_syncml_empty);
	create_case(s, "check_parser_syncml_wrong_case", check_parser_syncml_wrong_case);
	create_case(s, "check_parser_syncml_additional", check_parser_syncml_additional);
	create_case(s, "check_parser_no_syncbody", check_parser_no_syncbody);
	create_case(s, "check_parser_no_synchdr", check_parser_no_synchdr);
	create_case(s, "check_parser_synchdr_empty", check_parser_synchdr_empty);
	create_case(s, "check_parser_synchdr_verproto", check_parser_synchdr_verproto);
	create_case(s, "check_parser_synchdr_verdtd", check_parser_synchdr_verdtd);
	create_case(s, "check_parser_synchdr_msgid", check_parser_synchdr_msgid);
	create_case(s, "check_parser_synchdr_sessionid", check_parser_synchdr_sessionid);
	create_case(s, "check_parser_synchdr_additional", check_parser_synchdr_additional);
	create_case(s, "check_parser_synchdr_target", check_parser_synchdr_target);
	create_case(s, "check_parser_synchdr_source", check_parser_synchdr_source);
	create_case(s, "check_parser_synchdr_source2", check_parser_synchdr_source2);
	create_case(s, "check_parser_status_empty", check_parser_status_empty);
	create_case(s, "check_parser_status_cmdid", check_parser_status_cmdid);
	create_case(s, "check_parser_status_msgref", check_parser_status_msgref);
	create_case(s, "check_parser_status_cmdref", check_parser_status_cmdref);
	create_case(s, "check_parser_status", check_parser_status);
	create_case(s, "check_parser_status2", check_parser_status2);
	create_case(s, "check_parser_status3", check_parser_status3);
	create_case(s, "check_parser_final", check_parser_final);
	create_case(s, "check_parser_final2", check_parser_final2);
	create_case(s, "check_parser_cmd", check_parser_cmd);
	create_case(s, "check_parser_cmd_final", check_parser_cmd_final);
	create_case(s, "check_parser_sync_subcommand", check_parser_sync_subcommand);
	create_case(s, "check_parser_doctype", check_parser_doctype);

	create_case(s, "check_parser_emi", check_parser_emi);

	
	create_case(s, "check_parser_large_obj", check_parser_large_obj);
	create_case(s, "check_parser_large_obj2", check_parser_large_obj2);
	create_case(s, "check_parser_large_obj_error", check_parser_large_obj_error);
	create_case(s, "check_parser_large_obj_mangled", check_parser_large_obj_mangled);
	
	create_case(s, "check_parser_devinf", check_parser_devinf);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = parser_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_NORMAL);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
