/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <inttypes.h>
#include <kstat.h>
#include <string.h>

#include "libsysactivity.h"

static kstat_t* get_next_ksp(kstat_t* ksp);
static int assign_dst(struct sa_network_interface* dst, kstat_t* ksp);
static void assign_dst_field(kstat_t* ksp, uint64_t* field, char* tag64, char* tag32);

__thread kstat_ctl_t* network_kc;

int sa_open_network() {
	network_kc = kstat_open();
	if (network_kc == NULL)
		return ENOSYS;

	return 0;
}

int sa_close_network() {
	if (network_kc != NULL)
		kstat_close(network_kc);

	return 0;
}

int sa_count_network_interfaces(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	kstat_t* ksp = NULL;
	*number = 0;
	while ((ksp = get_next_ksp(ksp)) != NULL) {

		(*number)++;
	}
	return 0;
}

int sa_get_network_interface(char* name, struct sa_network_interface* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	kstat_t* ksp = NULL;
	while ((ksp = get_next_ksp(ksp)) != NULL) {
		if (strncmp(ksp->ks_name, name, sizeof ksp->ks_name) != 0)
			continue;

		assign_dst(dst, ksp);
		return 0;
	}
	return ENODEV;
}

int sa_get_network_interfaces(struct sa_network_interface* dst, uint16_t dst_size,
		uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int i;
	kstat_t* ksp = NULL;

	*written = 0;
	for (i = 0; i < dst_size; i++) {
		ksp = get_next_ksp(ksp);
		if (ksp == NULL)
			return ENODEV;

		if (assign_dst(&dst[i], ksp) != 0)
			return ENOSYS;

		(*written)++;
	}
	return 0;
}

static kstat_t* get_next_ksp(kstat_t* ksp) {
	do {
		if (ksp == NULL)
			ksp = network_kc->kc_chain;
		else
			ksp = ksp->ks_next;

		if (ksp == NULL)
			return ksp;
	} while (strcmp(ksp->ks_module, "link") != 0 && strcmp(ksp->ks_module, "lo") != 0);

	return ksp;
}

static int assign_dst(struct sa_network_interface* dst, kstat_t* ksp) {
	if (kstat_read(network_kc, ksp, NULL) == -1)
			return errno;

	strncpy(dst->name, ksp->ks_name, sizeof dst->name);
	dst->received_bytes = 0;
	assign_dst_field(ksp, &dst->received_bytes, "rbytes64", "rbytes");
	assign_dst_field(ksp, &dst->received_packets, "ipackets64", "ipackets");
	kstat_named_t* knp = kstat_data_lookup(ksp, "ierrors");
	if (knp != NULL)
		dst->received_errors = knp->value.ui32;
	else
		dst->received_errors = 0;

	assign_dst_field(ksp, &dst->sent_bytes, "obytes64", "obytes");
	assign_dst_field(ksp, &dst->sent_packets, "opackets64", "opackets");
	knp = kstat_data_lookup(ksp, "oerrors");
	if (knp != NULL)
		dst->sent_errors = knp->value.ui32;
	else
		dst->sent_errors = 0;

	return 0;
}

static void assign_dst_field(kstat_t* ksp, uint64_t* field, char* tag64, char* tag32) {
	kstat_named_t* knp = kstat_data_lookup(ksp, tag64);
	if (knp != NULL) {
		*field = knp->value.ui64;
		return;
	}

	knp = kstat_data_lookup(ksp, tag32);
	if (knp != NULL)
		*field = knp->value.ui32;
	else
		*field = 0;
}
