#ifndef TAGEXPR_SYNTREE_H
#define TAGEXPR_SYNTREE_H

/*
 * Tag expressions
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <OpSet.h>
#include <string>
#include <map>

class Tagexpr;

class TagexprContext
{
protected:
	const OpSet<std::string>& tags;
	const std::map<std::string, Tagexpr*>& derivedTags;
	// Tags "visited" during tag evaluation: used to break circular loops
	mutable OpSet<std::string> seen;

public:
	TagexprContext(const OpSet<std::string>& tags, const std::map<std::string, Tagexpr*>& derivedTags)
		throw () : tags(tags), derivedTags(derivedTags) {}

	bool eval(const std::string& tag) const throw ();
};

class Tagexpr
{
public:
	virtual ~Tagexpr() throw () {}

	virtual std::string format() const throw () = 0;

	virtual bool eval(const TagexprContext& context) const throw () = 0;
	virtual bool eval(const OpSet<std::string>& tags) const throw () = 0;

	virtual Tagexpr* clone() const throw () = 0;
};

class TagexprTag : public Tagexpr
{
protected:
	std::string _tag;
	
public:
	TagexprTag(const std::string& tag) throw () : _tag(tag) {}
	virtual ~TagexprTag() throw () {}

	virtual std::string format() const throw () { return _tag; }

	virtual bool eval(const TagexprContext& context) const throw ()
	{
		return context.eval(_tag);
	}
	virtual bool eval(const OpSet<std::string>& tags) const throw ()
	{
		return tags.find(_tag) != tags.end();
	}

	virtual Tagexpr* clone() const throw () { return new TagexprTag(_tag); }
};

class TagexprNot : public Tagexpr
{
protected:
	Tagexpr* _operand;

public:
	TagexprNot(Tagexpr* operand) throw () : _operand(operand) {}
	~TagexprNot() throw () { delete _operand; }

	virtual std::string format() const throw () { return "!" + _operand->format(); }

	virtual bool eval(const TagexprContext& context) const throw ()
	{
		return ! _operand->eval(context);
	}
	virtual bool eval(const OpSet<std::string>& tags) const throw ()
	{
		return ! _operand->eval(tags);
	}

	virtual Tagexpr* clone() const throw () { return new TagexprNot(_operand->clone()); }
};

class TagexprAnd : public Tagexpr
{
protected:
	Tagexpr* _operand1;
	Tagexpr* _operand2;

public:
	TagexprAnd(Tagexpr* operand1, Tagexpr* operand2) throw ()
		: _operand1(operand1), _operand2(operand2) {}
	~TagexprAnd() throw () { delete _operand1; delete _operand2; }

	virtual std::string format() const throw ()
	{
		return "( " + _operand1->format() + " && " + _operand2->format() + " )";
	}

	virtual bool eval(const TagexprContext& context) const throw ()
	{
		return _operand1->eval(context) && _operand2->eval(context);
	}
	virtual bool eval(const OpSet<std::string>& tags) const throw ()
	{
		return _operand1->eval(tags) && _operand2->eval(tags);
	}

	virtual Tagexpr* clone() const throw ()
	{
		return new TagexprAnd(_operand1->clone(), _operand2->clone());
	}
};

class TagexprOr : public Tagexpr
{
protected:
	Tagexpr* _operand1;
	Tagexpr* _operand2;

public:
	TagexprOr(Tagexpr* operand1, Tagexpr* operand2) throw ()
		: _operand1(operand1), _operand2(operand2) {}
	~TagexprOr() throw () { delete _operand1; delete _operand2; }

	virtual std::string format() const throw ()
	{
		return "( " + _operand1->format() + " || " + _operand2->format() + " )";
	}

	virtual bool eval(const TagexprContext& context) const throw ()
	{
		return _operand1->eval(context) || _operand2->eval(context);
	}
	virtual bool eval(const OpSet<std::string>& tags) const throw ()
	{
		return _operand1->eval(tags) || _operand2->eval(tags);
	}

	virtual Tagexpr* clone() const throw ()
	{
		return new TagexprOr(_operand1->clone(), _operand2->clone());
	}
};

// vim:set ts=4 sw=4:
#endif
