/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <errno.h>
#include <limits.h>
#include <memory.h>
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#elif HAVE_VARARGS_H
#include <varargs.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <dbh.h>

#ifdef XFCE_DISABLE_DEPRECATED
#undef XFCE_DISABLE_DEPRECATED
#endif

#include "constants.h"
#include "types.h"
#include "primary.h"
#include "gui.h"
#include "actions_lib.h"

#include "tubo.h"

#include "run.i"

/* terminal stuff */

G_MODULE_EXPORT
void
RECENT_add2history(widgets_t *widgets_p, gchar *path){
    if (!function_rational("plugins","xffm_recent",(void *)path,widgets_p,"add2history"))
	function_rational("plugins","xffm_frequent",(void *)path,widgets_p,"add2history");
}

G_MODULE_EXPORT
const gchar *xffm_what_term(void){
    static gchar *term=NULL;
    if (term){
       g_free(term);
       term=NULL;
    }
    term=NULL;
    if (getenv("TERMCMD") && strlen( getenv("TERMCMD"))){
	    if (MIME_is_valid_command(getenv("TERMCMD"))){
		    term = g_strdup(getenv("TERMCMD"));
	    } else {
		    g_warning("TERMCMD=%s: %s",getenv("TERMCMD"),strerror(errno));
	    }
    }
    if (!term) {
	term=g_strdup(FALLBACK_TERMINAL);
    }
    return (const gchar *)term;	
}


G_MODULE_EXPORT
void xffm_new_terminal_path(widgets_t *widgets_p,const gchar *wd){
    GError *error=NULL;
  
    chdir(wd);
    g_spawn_command_line_async (xffm_what_term(),&error);
    chdir(g_get_home_dir());

    return;
}


/* end terminal stuff */

G_MODULE_EXPORT
const 
gchar *
xffm_get_from_open_history(	gchar *path2find, 
				int *in_term){
    gchar *dbh_file=g_build_filename(xdg_cache_dir(),OPEN_FILE,NULL);
    GString *gs;
    DBHashTable *d;
    gchar *data;
    gchar *basename=g_path_get_basename(path2find);
    static gchar *in_cmd=NULL;

    if ((d=DBH_open(dbh_file))==NULL){
    	if ((d=DBH_create(dbh_file, 11))==NULL) {
		unlink(dbh_file);
		if ((d=DBH_create(dbh_file, 11))==NULL)	{
		    g_free(basename);
		    g_free(dbh_file);
		    return NULL;
		}
	}
    }
    g_free(dbh_file);
    gs = g_string_new(basename);
    g_free(basename);
    sprintf((char *)DBH_KEY(d), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    data = (gchar *)DBH_DATA(d);

    if (!DBH_load(d)) return NULL;
    

    
    if (*data==0) *in_term = 0;
    else *in_term=1;
    data++;
    g_free(in_cmd);
    in_cmd=g_strdup(data);
    
    DBH_close(d);
    return (const gchar *)in_cmd;
}

G_MODULE_EXPORT
void * 
xffm_runvwd_nopipes(	widgets_t *widgets_p,
			const gchar *wd,
			const gchar *argv[]){
    return private_xffm_runvwd(widgets_p, wd,argv, FALSE);
}
G_MODULE_EXPORT
void * 
xffm_runvwd(		widgets_t *widgets_p,
			const gchar *wd,
			const gchar *argv[]){
    return private_xffm_runvwd(widgets_p, wd,argv, TRUE);
}

G_MODULE_EXPORT
void 
xffm_save_flags(	gchar *in_cmd, 
			gboolean interm, 
			gboolean hold)
{
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *g=g_build_filename(xdg_cache_dir(),RUN_FLAG_FILE,NULL);
    
    if((runflags = DBH_open(g)) == NULL)
    {
	if((runflags = DBH_create(g, 11)) == NULL){
	    g_warning("Cannot create %s\n",g);
	    g_free(g);
	    return;
	}
    }
    gs = g_string_new(in_cmd);
    sprintf((char *)DBH_KEY(runflags), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    /*printf("TRACE: bookmarking with key=%s\n",d_path);*/
    flags = (int *)runflags->data;
    flags[0]=interm;
    flags[1]=hold;
    DBH_set_recordsize(runflags, 2*sizeof(int));
    
    DBH_update(runflags);
    DBH_close(runflags);
    TRACE("flags saved in dbh file for %s\n",in_cmd);
    g_free(g);
}

G_MODULE_EXPORT
void
xffm_recover_flags(	gchar *in_cmd,
			gboolean *interm,
			gboolean *hold)
{
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *g=g_build_filename(xdg_cache_dir(),RUN_FLAG_FILE,NULL);
    if((runflags = DBH_open(g)) == NULL)
    {
	    TRACE("Cannot open %s\n",g);
	    *interm=0;
	    *hold=0;
	    return;
    }
    gs = g_string_new(in_cmd);
    sprintf((char *)DBH_KEY(runflags), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    /*printf("TRACE: bookmarking with key=%s\n",d_path);*/
    flags = (int *)runflags->data;
    DBH_load(runflags);
    *interm = flags[0];
    *hold = flags[1];
    DBH_close(runflags);

    TRACE("flags recovered from dbh file for %s, interm=%d hold=%d\n",in_cmd,*interm,*hold);
}

G_MODULE_EXPORT
int 
xffm_on_run_path(	widgets_t *widgets_p,
			char *in_cmd, 
			const gchar *path, 
			gboolean interm, 
			gboolean remember, 
			gboolean put_in_history)
{
    return private_xffm_on_run_path(widgets_p, 
	    in_cmd, 
	    path, 
	    interm, 
	    remember, 
	    put_in_history,
	    TRUE);
}

G_MODULE_EXPORT
int 
xffm_on_run_path_nopipes(	widgets_t *widgets_p,
			char *in_cmd, 
			const gchar *path, 
			gboolean interm, 
			gboolean remember, 
			gboolean put_in_history)
{
    return private_xffm_on_run_path(widgets_p, 
	    in_cmd, 
	    path, 
	    interm, 
	    remember, 
	    put_in_history,
	    FALSE);
}

G_MODULE_EXPORT
int 
xffm_on_run(	widgets_t *widgets_p,
		char *in_cmd, 
		record_entry_t * en, 
		gboolean in_terminal, 
		gboolean remember, 
		gboolean put_in_history)
{
	gchar *path=NULL;
	int result;

	if (en && IS_NETFILE(en->subtype)){
	    path = g_strdup(XF_SMBget_cache_file(en,widgets_p));
	} 
	else if (en) path=g_strdup(en->path);
	else path=g_strdup("");
	
	/*if (en) {
	    gchar *dir=g_path_get_dirname(en->path);
	    path=g_path_get_basename(en->path);
	    chdir(dir);
	}*/
	result = xffm_on_run_path(widgets_p, in_cmd, path, in_terminal, remember, put_in_history);
	g_free(path);
	return result;
}




G_MODULE_EXPORT
void 
xffm_set_run_combo(	widgets_t *widgets_p,
			record_entry_t *en,
			xfc_combo_info_t *combo_info)
{
    gchar *f=g_build_filename(xdg_cache_dir(),RUN_DBH_FILE,NULL); 
    gboolean update=FALSE;
    gboolean docheck=TRUE;

    if (access(f,F_OK)!=0){update=TRUE; docheck=FALSE;}
    else {
       struct stat st[2];	    
       if (stat(f,st) < 0) update=TRUE;
       else {
	 gchar *m=g_strconcat(APPLICATION_MIME_FILE,NULL);
	 if (stat(m,st+1) >= 0 && st[0].st_mtime < st[1].st_mtime) update=TRUE;
	 m=g_strconcat(xdg_cache_dir(),LOCAL_APPLICATION_MIME_FILE,NULL);
	 if (stat(m,st+1) >= 0 && st[0].st_mtime < st[1].st_mtime) update=TRUE;
       }
    }
    
    COMBO_read_history(combo_info,f);

    if (en) {
	const gchar *p=MIME_command(en->path);
    	if (p) combo_info->list = g_list_prepend(combo_info->list,g_strdup(p) );
    }
    COMBO_set_combo(combo_info,NULL);
    g_free(f);

    {
      gboolean interm,hold;
      GtkEntry * entry = (GtkEntry *) lookup_widget(widgets_p->window,"input_entry");
      gchar *choice = g_strdup((gchar *)gtk_entry_get_text(entry));
      GtkWidget *check = lookup_widget(widgets_p->window,"checkbutton1");
      xffm_recover_flags(choice,&interm, &hold);
      gtk_toggle_button_set_active ((GtkToggleButton *)check,interm);
       if(getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM"))) {
         gtk_toggle_button_set_active ((GtkToggleButton *)check,TRUE);
	 gtk_widget_set_sensitive(check, FALSE);
      } else {
	 gtk_widget_set_sensitive(check, TRUE);
      }
      
      if (!interm) gtk_widget_hide(check); else gtk_widget_show(check); 
      g_free(choice);
    }

    return;
}


G_MODULE_EXPORT
void 
xffm_double_click_run(	widgets_t *widgets_p, record_entry_t *en)
{
    GError *error=NULL;
    char *a=NULL;
    int argc;
    gchar **argv;
    gboolean interm=FALSE;
    /* interm specified? */
    /* obsolete (from xfapps branch):
     * if (IS_IN_TERM(en->subtype)) interm=TRUE; */
 
    if (!IS_EXE(en->type)) return;
    
    /* is it a script? */
    if (is_a_script(en->path)) interm=TRUE;
    
    if (interm){
	const gchar *the_terminal=xffm_what_term();
	const gchar *exec_flag="-e";
	TRACE("the terminal=%s",the_terminal);
	if (strstr(the_terminal,"gnome-terminal") 	|| 
		strstr(the_terminal,"gnome2-terminal")	||
		strstr(the_terminal,"Terminal")	||
		strstr(the_terminal,"terminal")	||
		strstr(the_terminal,"xterminal") ) exec_flag="-x";
		
	a = g_strdup_printf("%s %s \"%s\"",the_terminal, exec_flag,en->path);
    } else {
        a=g_strdup_printf("\"%s\"",en->path);
    }
    
    TRACE("terminal cmd=%s",a);
    
    g_shell_parse_argv (a, &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics(widgets_p,"xffm/error",msg,":\n",a,"\n",NULL);
	g_error_free(error);
	g_free(a);
	g_free (msg);
	return;
    }
    xffm_runvwd(widgets_p,NULL,(const gchar **)argv);
    g_strfreev (argv); 
    g_free(a);
    return;
}

G_MODULE_EXPORT
gboolean 
xffm_try_sudo(widgets_t *widgets_p, char *command, char *arg, char *path){
    int i;
    gchar *g;
    g = g_find_program_in_path("sudo");
    if (g){
	char *argument[6];
	static gchar *sudo_prompt=NULL;
       /* try SUDO if available before barfing...*/
	if (sudo_prompt) g_free(sudo_prompt);
	sudo_prompt = g_strconcat("SUDO_PROMPT=","Password:","\n",NULL);
	putenv(sudo_prompt);
	argument[0]="sudo";
	argument[1]="-S";
	argument[2]=command;
	argument[3]=arg;
	argument[4]=path;
	argument[5]=NULL;
	    
	print_diagnostics(widgets_p,"xffm/sudo",NULL);
	for (i=0; argument[i]; i++) print_diagnostics(widgets_p,NULL," ",argument[i],NULL);
	print_diagnostics(widgets_p,NULL,"\n",NULL);
	/*easy_widgets_p = widgets_p;*/
	Tubo_full(fork_function, 
			(void *)argument, 
			easy_finished_function, 
			&childFD, 
			easy_stdout, 
			easy_stderr,
			widgets_p,
			0);
	g_free(g);
	return TRUE;
    } 
    print_diagnostics(widgets_p,"xffm/error",strerror(EINVAL),": ",command," ",arg," ",path,"\n",NULL);
    
    return FALSE;
}
    

