/*
 * Copyright (C) 2003-2004 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


static GList **the_list;
static time_t last_hit;
static gboolean asian=FALSE;


static xfc_combo_functions *xfc_fun=NULL;

static void clean_history_list(GList **list);
static gint translate_key(gint x);
static gint on_key_press(GtkWidget * entry, GdkEventKey * event, gpointer data);
static gint on_key_press_history(GtkWidget * entry, GdkEventKey * event, gpointer data);
static gint on_combo_history_key_press(GtkWidget * window, GdkEventKey * event, gpointer data);
static int history_compare (gconstpointer a,gconstpointer b);
static void history_mklist(DBHashTable * d);
static void get_history_list(GList **in_list,char *dbh_file,char *top);

G_MODULE_EXPORT
const gchar *combo_utf_string(const gchar *t)
{
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    unsigned char *c;
    const char *fc;
    gchar *from_codeset=NULL;
    
    if(!t) {
	TRACE("combo_utf_string(NULL)");
	return "";
    }
    
    if (g_utf8_validate (t,-1,NULL)) {
	TRACE("TRACE: valid utf8: %s\n",t);
	return t;
    }
    {
    	g_get_charset(&fc);
    	if (fc) from_codeset = g_strdup(fc);
    	else from_codeset = g_strdup("ISO-8859-1");
    }
    
    TRACE("TRACE: codeset for system is: %s\n",from_codeset);    
    
    if(s) {
	    g_free(s);
	    s=NULL;
    }
    TRACE("TRACE: utfing %s\n",t); 
    for(c = (unsigned char *)t; *c != 0; c++)
	if(*c < 32 && *c != '\n')
	    *c = ' ';
    s = g_convert (t,strlen(t),"UTF-8",from_codeset,&r_bytes, &w_bytes, &error);

    if(!s)
    {
	s=g_strdup(t);
	for(c = (unsigned char *)s; *c != 0; c++) if(*c > 128) *c = '?';
    }
    if(error){
        g_warning("%s. Codeset for system is: %s\nunable to convert to utf-8",
			error->message,from_codeset);
	g_error_free(error);
    }
     g_free(from_codeset);
    from_codeset=NULL;
   return (const gchar *)s;
}


static
gchar *
recursive_utf_string(const gchar *path){
    gchar *dir,*base,*valid,*utf_base,*utf_dir;
    if (!path) return NULL;
    if (g_utf8_validate (path,-1,NULL)) return g_strdup(path);
    dir=g_path_get_dirname(path);
    TRACE("dir=%s",dir);
    if (!dir || !strlen(dir) || strcmp(dir,"./")==0 || strcmp(dir,".")==0) {
	/* short circuit non-paths */
	g_free(dir);
	return g_strdup(combo_utf_string(path));
    }
    /* otherwise asume a mixed utf/locale string */
    base=g_path_get_basename(path);
    utf_dir=recursive_utf_string(dir);
    if (!g_utf8_validate (base,-1,NULL)){
	utf_base=g_strdup(combo_utf_string(base));
	g_free(base);
    } else {
	utf_base=base;
    }
    
    valid=g_strconcat(utf_dir,G_DIR_SEPARATOR_S,utf_base,NULL);
	
    /*printf("dir=%s base=%s valide=%s\n",dir, base, valid);*/
    g_free(utf_base);
    g_free(utf_dir);
    g_free(dir);
    return valid;
}

static
const gchar *
combo_valid_utf_pathstring(const gchar *string){
    static gchar *utf_string=NULL;
    g_free(utf_string);
    utf_string=recursive_utf_string(string);
    TRACE("string=%s utf_string=%s",string, utf_string);

    return utf_string;
}

static void clean_history_list(GList **list){
	GList *tmp;
	if (!*list) return ;
	for (tmp=*list;tmp;tmp=tmp->next){
		/*printf("freeing %s\n",(char *)tmp->data);*/
		g_free(tmp->data);
		tmp->data=NULL;
	}
	g_list_free(*list);
	*list=NULL;
	return;
}

static gint translate_key(gint x){
	switch (x) {
		case GDK_KP_Divide: return GDK_slash;
		case GDK_KP_Subtract: return GDK_minus;
		case GDK_KP_Multiply: return GDK_asterisk;
		case GDK_KP_Add: return GDK_plus;
		case GDK_KP_Space: return GDK_space;
		case GDK_KP_0: return GDK_0;
		case GDK_KP_1: return GDK_1;
		case GDK_KP_2: return GDK_2;
		case GDK_KP_3: return GDK_3;
		case GDK_KP_4: return GDK_4;
		case GDK_KP_5: return GDK_5;
		case GDK_KP_6: return GDK_6;
		case GDK_KP_7: return GDK_7;
		case GDK_KP_8: return GDK_8;
		case GDK_KP_9: return GDK_9;
	}
	return x;
}

static
int compose_key(int key, int dead_key){
    switch (dead_key){
	case GDK_dead_grave:
	    switch (key){
		case GDK_A: return GDK_Agrave;
		case GDK_a: return GDK_agrave;
		case GDK_E: return GDK_Egrave;
		case GDK_e: return GDK_egrave;
		case GDK_I: return GDK_Igrave;
		case GDK_i: return GDK_igrave;
		case GDK_O: return GDK_Ograve;
		case GDK_o: return GDK_ograve;
		case GDK_U: return GDK_Ugrave;
		case GDK_u: return GDK_ugrave;
	    }
	    break;
	case GDK_dead_acute:
	     TRACE("dead key=0x%x composing %c",(unsigned)dead_key,(char)key);
	    switch (key){
		case GDK_A: return GDK_Aacute;
		case GDK_a: return GDK_aacute;
		case GDK_E: return GDK_Eacute;
		case GDK_e: return GDK_eacute;
		case GDK_I: return GDK_Iacute;
		case GDK_i: return GDK_iacute;
		case GDK_O: return GDK_Oacute;
		case GDK_o: return GDK_oacute;
		case GDK_U: return GDK_Uacute;
		case GDK_u: return GDK_uacute;
		case GDK_Y: return GDK_Yacute;
		case GDK_y: return GDK_yacute;
		case GDK_S: return  GDK_Sacute;  
		case GDK_Z: return  GDK_Zacute;
		case GDK_s: return  GDK_sacute;
		case GDK_z: return  GDK_zacute;
		case GDK_R: return  GDK_Racute;
		case GDK_r: return  GDK_racute;
		case GDK_L: return  GDK_Lacute;
		case GDK_l: return  GDK_lacute;
		case GDK_C: return  GDK_Cacute;
		case GDK_c: return  GDK_cacute;
		case GDK_N: return  GDK_Nacute;
		case GDK_n: return  GDK_nacute; 
	    }
	    break;
	case GDK_dead_diaeresis:
	    switch (key){
		case GDK_A: return GDK_Adiaeresis;
		case GDK_a: return GDK_adiaeresis;
		case GDK_E: return GDK_Ediaeresis;
		case GDK_e: return GDK_ediaeresis;
		case GDK_I: return GDK_Idiaeresis;
		case GDK_i: return GDK_idiaeresis;
		case GDK_O: return GDK_Odiaeresis;
		case GDK_o: return GDK_odiaeresis;
		case GDK_U: return GDK_Udiaeresis;
		case GDK_u: return GDK_udiaeresis;
		case GDK_Y: return GDK_Ydiaeresis;
		case GDK_y: return GDK_ydiaeresis;
	    }
	    break;
	case GDK_dead_cedilla:
	    switch (key){
		case GDK_C: return GDK_Ccedilla;
		case GDK_c: return GDK_ccedilla;
		case GDK_S: return GDK_Scedilla;
		case GDK_s: return GDK_scedilla;
		case GDK_T: return GDK_Tcedilla;
		case GDK_t: return GDK_tcedilla;
		case GDK_R: return GDK_Rcedilla;
		case GDK_r: return GDK_rcedilla;
		case GDK_L: return GDK_Lcedilla;
		case GDK_l: return GDK_lcedilla;
		case GDK_G: return GDK_Gcedilla;
		case GDK_g: return GDK_gcedilla;
		case GDK_N: return GDK_Ncedilla;
		case GDK_n: return GDK_ncedilla;
		case GDK_K: return GDK_Kcedilla;
		case GDK_k: return GDK_kcedilla;
	    }
	    break;
	case GDK_dead_circumflex:
	    switch (key){
		case GDK_A: return GDK_Acircumflex;
		case GDK_a: return GDK_acircumflex;
		case GDK_E: return GDK_Ecircumflex;
		case GDK_e: return GDK_ecircumflex;
		case GDK_I: return GDK_Icircumflex;
		case GDK_i: return GDK_icircumflex;
		case GDK_O: return GDK_Ocircumflex;
		case GDK_o: return GDK_ocircumflex;
		case GDK_U: return GDK_Ucircumflex;
		case GDK_u: return GDK_ucircumflex;
		case GDK_H: return GDK_Hcircumflex;
		case GDK_h: return GDK_hcircumflex;
		case GDK_J: return GDK_Jcircumflex;
		case GDK_j: return GDK_jcircumflex;
		case GDK_C: return GDK_Ccircumflex;
		case GDK_c: return GDK_ccircumflex;
		case GDK_G: return GDK_Gcircumflex;
		case GDK_g: return GDK_gcircumflex;
		case GDK_S: return GDK_Scircumflex;
		case GDK_s: return GDK_scircumflex;
	    }
	    break;
    }
    return key;
}

static gint on_key_press(GtkWidget * entry, GdkEventKey * event, gpointer data)
{
 xfc_combo_info_t *combo_info=(xfc_combo_info_t *)data;
 TRACE("on_key_press: got key= 0x%x",event->keyval);
 if (event->keyval == GDK_Escape && combo_info->cancel_func) {
	(*(combo_info->cancel_func))((GtkEntry *)entry,combo_info->cancel_user_data);
	return TRUE;
 }
 return FALSE;
}

static int
deadkey(int key){
 /* support for deadkeys */
 switch (key){
	 /* spanish */
     case GDK_dead_acute:
     case GDK_dead_diaeresis:
	 return key;
	 /* french */
     case GDK_dead_cedilla:
     case GDK_dead_grave:
     case GDK_dead_circumflex:
	 return key;
	 /* others (if you want any of these, submit a request)*/
     case GDK_dead_tilde:
     case GDK_dead_macron:
     case GDK_dead_breve:
     case GDK_dead_abovedot:
     case GDK_dead_abovering:
     case GDK_dead_doubleacute:
     case GDK_dead_caron:
     case GDK_dead_ogonek:
     case GDK_dead_iota:
     case GDK_dead_voiced_sound:
     case GDK_dead_semivoiced_sound:
     case GDK_dead_belowdot:
/* these two are > gtk-2.2: */
/*     case GDK_dead_hook:*/
/*     case GDK_dead_horn:*/
	 return 0;
     default:
	 return 0;
 }
}

static gint on_key_press_history(GtkWidget * entry, GdkEventKey * event, gpointer data)
{
    gchar *utf_fulltext=NULL;
    static int dead_key=0;
    gboolean find_match=FALSE;
    int i;
    gchar *text[2]={NULL,NULL};
    gchar c[]={0,0,0,0,0};
    gchar *fulltext=NULL;
    xfc_combo_info_t *combo_info = (xfc_combo_info_t *)data;
    GList *tmp=combo_info->list;
    GtkEditable *editable=(GtkEditable *)entry;
    static gint shift_pos=-1;
    static gint cursor_pos=-1;
    gint pos1,pos2,pos;
    gboolean preselection;
    
    TRACE("on_key_press_history: got key= 0x%x",event->keyval);
    
   /* asian input methods: turns off autocompletion */
    if (event->keyval == GDK_space &&(event->state&(GDK_CONTROL_MASK|GDK_SHIFT_MASK))){
	asian=TRUE;
	return FALSE;
    }
    else if (event->keyval == GDK_space &&(event->state&GDK_MOD1_MASK)){
	/* turn autocompletion back on */
	asian=FALSE;	
	return TRUE;
    }

    if (asian && !(event->keyval == GDK_Return) && !(event->keyval == GDK_KP_Enter)) return FALSE;



    if (event->keyval == GDK_Shift_L || event->keyval == GDK_Shift_R ){
     cursor_pos = shift_pos = pos; 
    }
    
    if (event->keyval == GDK_Menu || event->keyval == GDK_ISO_Level3_Shift) 
	return TRUE;
    
    if (dead_key) {
	if (event->keyval != GDK_Shift_L && event->keyval != GDK_Shift_R){
	  event->keyval = compose_key(event->keyval,dead_key);
	  TRACE("composing to %c",(char)event->keyval);
	  dead_key=0;
	}
    } else {
	dead_key=deadkey(event->keyval);
	TRACE("deadkey is  0x%x",(unsigned)dead_key);
	if (dead_key) return TRUE;
    }
    

    
    preselection=
	    gtk_editable_get_selection_bounds (editable,&pos1,&pos2);

    pos=gtk_editable_get_position(editable);
     
    if (!preselection) pos1 =  pos2 = -1;

    TRACE("TRACE(2):pos= %d, shift_pos=%d cursor_pos=%d",pos,shift_pos,cursor_pos);
    TRACE("TRACE(2):got key= 0x%x",event->keyval);
    /*TRACE("TRACE(2):dbhfile= %s\n",combo_info->active_dbh_file);*/

    
 if (event->keyval == GDK_KP_Down && (event->state&GDK_MOD1_MASK)){
	 goto returnFALSE;
 }
 if (event->keyval == GDK_KP_Up && (event->state&GDK_MOD1_MASK)){
	 goto returnFALSE;
 }
 if (event->keyval == GDK_Down && (event->state&GDK_MOD1_MASK)){
	 goto returnFALSE;
 }
 if (event->keyval == GDK_Up && (event->state&GDK_MOD1_MASK)){
	 goto returnFALSE;
 }
 
 
 g_signal_handlers_block_by_func (GTK_OBJECT (entry), (gpointer)on_key_press_history, data);
 if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter) {
	 /*printf("TRACE: got return;\n");*/
	 if (combo_info->activate_func) {
		/*( *(combo_info->activate_func))((GtkEntry *)entry,(gpointer)combo_info);*/
		( *(combo_info->activate_func))((GtkEntry *)entry,combo_info->activate_user_data);
	 }
	 goto end;
 }
    
 if (event->keyval == GDK_BackSpace && (event->state&GDK_CONTROL_MASK)){
        gchar *p;
        fulltext=gtk_editable_get_chars (editable,0,-1);
	p = strrchr(fulltext,' ');
	if (!p) p = strrchr(fulltext,G_DIR_SEPARATOR);
	if (!p) {
	    gtk_editable_delete_text (editable,0,-1);
	    /*xfc_set_blank(combo_info);*/
	}
	else {
	    gtk_editable_delete_text (editable,strlen(fulltext)-strlen(p),-1);
	}
	g_free(fulltext);
	fulltext=NULL;
	 goto end;
 }

 if (event->keyval == GDK_Tab) {
    gint start,finish;
    {	
    	if (gtk_editable_get_selection_bounds(editable,&start,&finish)){
	  gchar *p;
	  p=gtk_editable_get_chars (editable,start,finish);
	  if (*p ==G_DIR_SEPARATOR || *p ==' ' || *p =='.') start++;
	  g_free(p);p=NULL;	  
	  /*printf("***************selection at %d %d\n",start,finish);*/
	  for (;start<=finish;start++){
		p=gtk_editable_get_chars (editable,start,finish);
		/*printf("***************start=%d got char %s\n",start,p);*/
		/*if (*p ==G_DIR_SEPARATOR || *p ==' ' || *p =='.'f) */
		if ((*p == G_DIR_SEPARATOR && *(p+1) != G_DIR_SEPARATOR) ||
		    (*p ==' ' && *(p+1) != ' ') ||
		    (*p =='.' && *(p+1) != '.'))
		{
			start++;
			gtk_editable_delete_text (editable,start,finish);
			preselection=FALSE;
			g_free(p);p=NULL;
			break;
		}
		g_free(p);p=NULL;
	  }
   	  gtk_editable_select_region (editable, 0,0);
    	  gtk_editable_set_position(editable,-1);
      	  fulltext=gtk_editable_get_chars (editable,0,-1);
	  utf_fulltext=g_strdup(fulltext);
		/*printf("***************fulltext=%s\n",fulltext);*/
	  
	     cursor_pos = -1;
	     goto complete_text;
	  
	} 
	else {
 	  gchar *p,*q;
	  gboolean blank=TRUE;
	  p=gtk_editable_get_chars (editable,0,-1);
	  if (p){
		  for (q=p; *q; q++){
			  if (*q != ' ') blank=FALSE;
		  }
		  if (blank) g_signal_emit_by_name((gpointer)GTK_COMBO(combo_info->combo)->entry, "activate", NULL);
	  }
	  g_free(p);p=NULL;
          goto end;
	}
    }
 }
 else if (CURSOR_KEYSTROKE(event->keyval)) {
	 if (event->keyval == GDK_Right){
	     /*printf ("TRACE: right\n");*/
	     if (event->state&GDK_SHIFT_MASK){
	       cursor_pos++;
	       if (cursor_pos < shift_pos) gtk_editable_select_region (editable, cursor_pos,shift_pos);
	       else gtk_editable_select_region (editable,shift_pos,cursor_pos);
	     } else {
		gtk_editable_set_position (editable,pos+1);
		cursor_pos = pos+1;
	     }
	 }else{
	     /*printf ("TRACE: left\n");*/
	     /* cranky gtk design, position must be at end of selection... */
	     if (cursor_pos) cursor_pos--; 
	     if (event->state&GDK_SHIFT_MASK){
	           if (cursor_pos < shift_pos) 
		       gtk_editable_select_region (editable, cursor_pos,shift_pos);
	           else 
		       gtk_editable_select_region (editable,shift_pos,cursor_pos);
	     } else if (pos-1 >= 0){
		 gtk_editable_set_position (editable,pos-1);
		cursor_pos = pos-1;
	     }
	 }
	goto end;	 
 }


 /* construct search string... (limited to 128 bytes)*/

 /* control-delete will remove the selected item from the
  * history dbh file (to remove stale history items) */

 if (BASIC_KEYSTROKE(event->keyval)){
	/* get entry text */

   	if (event->keyval == GDK_BackSpace ){
    	    if (preselection) {
	        /* why was this -1? gtk_editable_delete_text (editable,pos1,-1);*/
	        gtk_editable_delete_text (editable,pos1,pos2);
		/*printf("TRACE:pos1=%d,pos2=%d\n",pos1,pos2);*/
		goto end;
	    }
	    if (pos==0) {
		    goto end;
	    }
      	    text[0]=gtk_editable_get_chars (editable,0,pos-1);
	    text[1]=gtk_editable_get_chars (editable,pos,-1);
	    fulltext=g_strconcat(text[0],text[1],NULL);
   	    g_free(text[0]);
    	    g_free(text[1]);
	    text[0]=text[1]=NULL;
	    gtk_editable_delete_text (editable,0,-1);
	    pos1=0;
	    if (fulltext && strlen(fulltext)){
	       gtk_editable_insert_text (editable,(const gchar *)fulltext,
			     strlen(fulltext),&pos1);
	       gtk_editable_set_position (editable,pos-1);
	     cursor_pos = pos-1;
	       /*printf("TRACE: inserting %s\n",fulltext);*/
	    } else {
	 	xfc_set_blank(combo_info);
	    }
	    g_free(fulltext); 
	    fulltext=NULL;
	    goto end;
	} 
	else if (event->keyval == GDK_Delete ||
     	    event->keyval == GDK_KP_Delete){
	  if (combo_info->active_dbh_file && event->state&GDK_CONTROL_MASK){ /* remove stale entries */
    	    fulltext=gtk_editable_get_chars (editable,0,-1);
	    if (fulltext && strlen(fulltext) && combo_info->association_hash) {
		gchar *local_choice = g_hash_table_lookup(combo_info->association_hash,fulltext);
		TRACE("converting back to non utf8 value %s ---> %s",
		    fulltext,local_choice);
		if (local_choice) {
		    g_free(fulltext);
		    fulltext=local_choice;
		}
	    }

	    xfc_remove_from_history(combo_info->active_dbh_file,fulltext);
	    xfc_set_blank(combo_info);
	    
	    g_free(fulltext);
	    fulltext=NULL;
	    if (combo_info->cancel_func)
	       (*(combo_info->cancel_func))((GtkEntry *)entry,combo_info->cancel_user_data);
	    goto end;
	  } else {
		  /*printf("TRACE: pos=%d, pos1=%d\n",pos,pos1);*/
	    g_free(fulltext);
	    fulltext=NULL;
    	    if (preselection) {
	        /* gtk_editable_delete_text (editable,pos1,-1);*/
	        gtk_editable_delete_text (editable,pos1,pos2);
		goto end;
	    }
      	    text[0]=gtk_editable_get_chars (editable,0,pos);
	    text[1]=gtk_editable_get_chars (editable,pos+1,-1);
	    /* do conversion to locale here? no. */
	    fulltext=g_strconcat(text[0],text[1],NULL);
   	    g_free(text[0]);
    	    g_free(text[1]);
	    text[0]=text[1]=NULL;
	    
	    gtk_editable_delete_text (editable,0,-1);
	    if (fulltext && strlen(fulltext)){
	      pos1=0;
	      gtk_editable_insert_text (editable,(const gchar *)fulltext,
			     strlen(fulltext),&pos1);
	      gtk_editable_set_position (editable,pos);
	     cursor_pos = pos;
	      /*printf("TRACE: inserting %s\n",fulltext);*/
	    } else {
	 	xfc_set_blank(combo_info);
	    }
	    g_free(fulltext); 
	    fulltext=NULL;
	    goto end;
	  }
	} else { /* normal key */
	  gchar *utf_c=NULL;
	  *c=translate_key(event->keyval);
	  
	  if (!g_utf8_validate (c,-1,NULL)){
		const char *fc;
		GError *error = NULL;
		gchar *from_codeset=NULL;
		gsize r_bytes, w_bytes;
	   	g_get_charset(&fc);
		if (fc) from_codeset = g_strdup(fc);
		else from_codeset = g_strdup("ISO-8859-1");
		utf_c = g_convert (c,strlen(c),"UTF-8",from_codeset,&r_bytes, &w_bytes, &error);
		g_free(from_codeset);
		if (error){
		    g_warning(error->message);
		    g_error_free(error);
		}

	  }
	  else utf_c=g_strdup(c);
    	  if (preselection) {
	        gtk_editable_delete_text (editable,pos1,-1);
      	        text[0]=gtk_editable_get_chars (editable,0,-1);
		/*FIXME convert text[0] to locale for fulltext*/
		
	  	fulltext=g_strconcat(text[0],c,NULL);
		
		utf_fulltext=g_strconcat(text[0],utf_c,NULL);
		text[1]=NULL;
	  	pos=0;
	        gtk_editable_delete_text (editable,0,-1);
	  	gtk_editable_insert_text (editable,(const gchar *)utf_fulltext,
			     strlen(utf_fulltext),&pos);
	  	gtk_editable_set_position (editable,pos);
	     cursor_pos = pos;
	  } else {
		  /*printf("TRACE: pos=%d\n",pos);*/
		/*FIXME convert text[0],text[1] to locale for fulltext*/
      	        text[0]=gtk_editable_get_chars (editable,0,pos);
    	        text[1]=gtk_editable_get_chars (editable,pos,-1);
		/* convert to locale */
	  	fulltext=g_strconcat(text[0],c,text[1],NULL);
		
	  	utf_fulltext=g_strconcat(text[0],utf_c,text[1],NULL);
		  /*printf("TRACE: pos=%d fulltext=%s\n",pos,fulltext);*/
	  	pos1=0;
	        gtk_editable_delete_text (editable,0,-1);
	  	gtk_editable_insert_text (editable,(const gchar *)utf_fulltext,
			     strlen(utf_fulltext),&pos1);
	  	gtk_editable_set_position (editable,pos+1);
	     cursor_pos = pos;
	  }
	  g_free(utf_c);
	}		
    	g_free(text[0]);
    	g_free(text[1]);
	text[0]=text[1]=NULL;
 }
 else if (event->keyval != GDK_Tab) {
         g_signal_handlers_unblock_by_func (GTK_OBJECT (entry), (gpointer)on_key_press_history, data);
	 goto returnFALSE;
 }

 for (i=0; i< strlen(fulltext); i++) if (fulltext[i] != ' ') find_match = TRUE;
 if (find_match && combo_info->combo) {
 	/*printf("TRACE: setting limited list and emitting signal...fulltext=%s\n",(fulltext)?fulltext:"null");*/
	if (xfc_set_combo(combo_info,fulltext)){
	    if (combo_info->limited_list && g_list_length(combo_info->limited_list) > 1){
		g_signal_emit_by_name((gpointer)(entry), "activate", NULL);
	    }
	}
 }
 
complete_text: 
 if (fulltext){
    if (!utf_fulltext) {
	g_warning("utf_fulltext is null");
	utf_fulltext=g_strdup(fulltext);
    }
    /* look for in ordered GList*/
    TRACE("TRACE:fulltext is %s\n",fulltext);
    for (tmp=combo_info->list;tmp;tmp=tmp->next){
	const gchar *p,*utf_p;
	if (!tmp->data) continue;
	utf_p=combo_valid_utf_pathstring((gchar *)(tmp->data));
	p=(gchar *)(tmp->data);
        /*printf("TRACE:compare %s  with-> %s\n",fulltext,p);*/
	if (strncmp(fulltext,p,strlen(fulltext))==0){
	    const gchar *to_write=utf_p+strlen(utf_fulltext);
	    
	    /*printf("TRACE: gotcha %s->%s, to write=%s\n",fulltext,p,to_write);*/
            /* if found, complete with untyped part selected.*/
	    gtk_editable_delete_text (editable,0,-1);
	    pos1=0;
	    gtk_editable_insert_text (editable,(const gchar *)utf_fulltext,
			     strlen(utf_fulltext),&pos1);
	    pos2=pos1;
	    gtk_editable_insert_text (editable,(const gchar *)to_write,
			     strlen(to_write),&pos2);
    	    gtk_editable_select_region (GTK_EDITABLE (entry), pos1, -1);
    	    g_free(fulltext);
	    fulltext=NULL;
	    goto end;
	}
    }
    g_free(utf_fulltext);
    g_free(fulltext);
    fulltext=NULL;
 }
 end:			    
 g_signal_handlers_unblock_by_func (GTK_OBJECT (entry), (gpointer)on_key_press_history, data);
 if (xfc_fun->extra_key_completion) (*(xfc_fun->extra_key_completion))(xfc_fun->extra_key_data);
 return (TRUE);
returnFALSE:
 if (xfc_fun->extra_key_completion) (*(xfc_fun->extra_key_completion))(xfc_fun->extra_key_data);
 return (FALSE);
}

static gint on_combo_history_key_press(GtkWidget * window, GdkEventKey * event, gpointer data)
{
	/* gtk bug: the default handler always gets called first, so you cannot do anything with
	 * return key. bummer. up to gtk2.4 at least. */
 xfc_combo_info_t *combo_info = (xfc_combo_info_t *)data; 
 if (event->keyval == GDK_KP_Down || event->keyval == GDK_KP_Up || event->keyval == GDK_Down || event->keyval == GDK_Up){
     if (xfc_fun->extra_key_completion) (*(xfc_fun->extra_key_completion))(xfc_fun->extra_key_data);
     return FALSE;
 }
     /*printf("TRACE(32):got key= 0x%x\n",event->keyval);*/
 if (event->keyval == GDK_Shift_L 
	 || event->keyval == GDK_Shift_R
	 || event->keyval == GDK_Control_L
	 || event->keyval == GDK_Control_R){
     gtk_widget_hide (window);
     if (GTK_WIDGET_HAS_GRAB (window))
     {
	  gtk_grab_remove (window);
	  gdk_pointer_ungrab (event->time);
     }
 }

 if (ENTRY_KEYSTROKE(event->keyval)) {
     
     /*
      * function not found:
      * g_signal_emit_stop_by_name (GTK_OBJECT (window), "key_press_event");
      *
      */
     gtk_widget_hide (window);
     if (GTK_WIDGET_HAS_GRAB (window))
     {
	  gtk_grab_remove (window);
	  gdk_pointer_ungrab (event->time);
     }
     
#if 10
     on_key_press_history((GtkWidget *)(GTK_COMBO(combo_info->combo)->entry), event, data);
#else
     /* alternatively it may be done by signal, : */
     {
       gint i;
       /*TRACE("signal_emit combo = 0x%x\n",combo_info->combo);)*/
       gtk_signal_emit_by_name (GTK_OBJECT (combo_info->entry), "key_press_event",event,&i);
       /*TRACE("signal_emit done, combo = 0x%x\n",combo_info->combo);)*/
	/* the last parameter is used to return a value from the function.*/
     }
#endif
     return TRUE;

 }
 if (xfc_fun->extra_key_completion) (*(xfc_fun->extra_key_completion))(xfc_fun->extra_key_data);
 return FALSE;
}

static int history_compare (gconstpointer a,gconstpointer b){
	history_dbh_t *da=(history_dbh_t *)a;
	history_dbh_t *db=(history_dbh_t *)b;
	/*if (db->last_hit >= last_hit) last_hit = db->last_hit;
	if (da->last_hit >= last_hit) last_hit = da->last_hit;*/
	       	
	if (db->last_hit >= last_hit && da->last_hit < last_hit){
		return 1;
	}
	if (da->last_hit >= last_hit && db->last_hit < last_hit){
		return -1;
	}
	if (db->hits != da->hits) return (db->hits - da->hits);
	/*if (db->last_hit != da->last_hit) return db->last_hit - da->last_hit;*/
	return (strcmp(da->path,db->path));
}

static void history_lasthit(DBHashTable * d)
{
    history_dbh_t *history_mem = (history_dbh_t *)DBH_DATA(d);
    if (!history_mem) g_assert_not_reached();
    if (history_mem->last_hit >= last_hit) {
		last_hit=history_mem->last_hit;
    }
}

static void history_mklist(DBHashTable * d)
{
    history_dbh_t *history_mem = (history_dbh_t *)malloc(sizeof(history_dbh_t));
    if (!history_mem) g_assert_not_reached();
    memcpy(history_mem,DBH_DATA(d),sizeof(history_dbh_t));
    if (!the_list) g_assert_not_reached(); 
    if (history_mem->path && strlen(history_mem->path)){
       *the_list = g_list_insert_sorted(*the_list,history_mem,history_compare);
       /*printf("TRACE: inserted %s\n",(char *)history_mem->path);*/
    }

}
/* if top==NULL, the top entry is left blank,
 * if top=="", the top entry is the one with the greatest access time for last hit
 * if top=="anything", the entry is set to "anything"
 *
 * (only "" is used now)
 * */

static void get_history_list(GList **in_list,char *dbh_file,char *top){
   DBHashTable *d;
   GList *tmp;
/*   char *first=NULL;*/
   the_list=in_list;

	/*printf("TRACE:at get_history_list with %s \n",dbh_file);*/
	   
   clean_history_list(the_list);	
   last_hit=0;
   if ((d=DBH_open(dbh_file))!=NULL){
	DBH_foreach_sweep(d,history_lasthit);	
	DBH_foreach_sweep(d,history_mklist);	
	DBH_close(d);
   }
   /* leave only strings in the history list: */
   for (tmp=*the_list;tmp;tmp=tmp->next){
 	history_dbh_t *history_mem=(history_dbh_t *)tmp->data;
	gchar *p=g_strdup(history_mem->path);
	TRACE("%s, hits=%d",history_mem->path,history_mem->hits);
	tmp->data=p;
	g_free(history_mem);
	history_mem=NULL;
   }
   
   
   /*if (!top) *the_list=g_list_prepend(*the_list,g_strdup(""));
   if (top && strlen(top)) *the_list=g_list_prepend(*the_list,g_strdup(top));*/
	
   if (*the_list==NULL) {
	   *the_list=g_list_prepend(*the_list,g_strdup(""));
   }
   return ;
}

static void on_select_child(GList *list, GtkWidget *child){
     if (xfc_fun->extra_key_completion) (*(xfc_fun->extra_key_completion))(xfc_fun->extra_key_data);
     return;     
}

/*
static const char *xfc_tod(void){
	time_t t=time(NULL);
	return ctime(&t);
}*/


