/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: stackdump.c,v 1.1.2.5 2003/07/28 14:14:01 sasa Exp $
 *
 * Author  : bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stackdump.h>
#include <zorp/log.h>


#if ZORPLIB_ENABLE_STACKDUMP

#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include <execinfo.h>

/**
 * z_stackdump_log_stack:
 * @p: signal context
 *
 * Log parts of the current stack in hexadecimal form.
 **/
void
z_stackdump_log_stack(ZSignalContext *p)
{
  unsigned long *esp = (unsigned long *) p->esp;
  int i;

  if (!esp)
    {
      z_log(NULL, CORE_ERROR, 0, "ESP is NULL, stackdump is not available, falling back to current frame\n");
      esp = (unsigned long *) &esp;
    }

  for (i = 0; i < 64; i++)
    {
      z_log(NULL, CORE_ERROR, 0, "Stack 0x%08lx: %08lx %08lx %08lx %08lx", (unsigned long) esp, esp[0], esp[1], esp[2], esp[3]);
      esp += 4;
    }
}

/**
 * z_stackdump_log_backtrace:
 * @p: signal context
 * 
 * x86 specific stack unrolling function which logs all return addresses and
 * their associated frame pointers.
 **/
void
z_stackdump_log_backtrace(ZSignalContext *p)
{
  /* NOTE: this is i386 specific */
  unsigned long *ebp = (unsigned long *) p->ebp;
  
  /*NOLOG*/
  z_log(NULL, CORE_ERROR, 0, "retaddr=0x%lx, ebp=0x%lx", p->eip, (unsigned long) ebp);
  
  while (ebp > (unsigned long *) &ebp && *ebp) 
    {
      /*NOLOG*/
      z_log(NULL, CORE_ERROR, 0, "retaddr=0x%lx, ebp=0x%lx", *(ebp+1), *ebp);
      ebp = (unsigned long *) *ebp;
    }
}

/**
 * z_stackdump_log_context:
 * @p: signal context
 *
 * Log information found directly in the signal context (register contents).
 **/
void
z_stackdump_log_context(ZSignalContext *p)
{
  /*LOG
    This message is logged when Zorp caught a fatal signal.
   */
  z_log(NULL, CORE_ERROR, 0, 
        "Fatal signal occurred, dumping stack; eax='%08lx', ebx='%08lx', ecx='%08lx', edx='%08lx', esi='%08lx', edi='%08lx', ebp='%08lx', esp='%08lx', eip='%08lx'",
        p->eax, p->ebx, p->ecx, p->edx, p->esi, p->edi, p->ebp, p->esp, p->eip);
}

/**
 * z_stackdump_log_maps:
 *
 * This function reads and logs the contents of the /proc/<pid>/maps file
 * which includes memory mapping for mapped shared objects.
 **/
void
z_stackdump_log_maps(void)
{
  int fd;

  fd = open("/proc/self/maps", O_RDONLY);
  if (fd != -1)
    {
      gchar buf[32768];
      int rc;
      gchar *p, *eol;
      gint avail, end = 0;

      while (1)
        {
          avail = sizeof(buf) - end;
          rc = read(fd, buf + end, avail);

          if (rc == -1)
            break;
	  end += rc;
	  if (rc == 0)
	    break;
          p = buf;
          while (*p && p < (buf + end))
            {
              eol = memchr(p, '\n', buf + end - p);
              if (eol)
                {
                  *eol = 0;
                  /*NOLOG*/
                  z_log(NULL, CORE_ERROR, 0, "%s", p);
                  p = eol + 1;
                }
              else
                {
                  end = end - (p - buf);
                  memmove(buf, p, end);
                  break;
                }
            }
        }
      if (end)
        z_log(NULL, CORE_ERROR, 0, "%.*s", end, buf);
      close(fd);
    }
  else
    {
      /*LOG
        When error occured Zorp try to gather as many information
        as can. One of it's source is tha maps file. This message
        is about that Zorp cannot open this map file.
       */
      z_log(NULL, CORE_ERROR, 0, "Error opening /proc/self/maps;");
    }
}

/**
 * z_stackdump_log_symbols:
 * 
 * This function uses the libc backtrace() and backtrace_symbols() functions
 * to display a backtrace with resolved names. As this does not always
 * succeed (as it uses malloc()) the alternative implementation without
 * symbols is also present.
 **/
void
z_stackdump_log_symbols(void)
{
  void *bt[256];
  int count, i;
  
  count = backtrace(bt, 256);
  if (count)
    {
      gchar **symbols;
      
      z_log(NULL, CORE_ERROR, 0, "Symbol dump; count='%d'", count);
      symbols = backtrace_symbols(bt, count);
      for (i = 0; i < count; i++)
        {
          z_log(NULL, CORE_ERROR, 0, "%p: %s", bt[i], symbols[i]);
        }
    }
}

/**
 * z_stackdump_log:
 * @p: signal context
 *
 * This function gathers as much information for post-mortem analysis as
 * possible. It is usually called from a fatal signal handler (like SIGSEGV).
 * The current signal context can be queried with the z_stackdump_get_context() macro.
 * This function is Linux & x86 specific.
 **/
void
z_stackdump_log(ZSignalContext *p)
{
  z_stackdump_log_context(p);
  z_stackdump_log_backtrace(p);
  z_stackdump_log_maps();
  z_stackdump_log_stack(p);
  z_stackdump_log_symbols();
}

#else

/**
 * z_stackdump_log:
 * @p: signal context
 *
 * This function is dummy but portable replacement for the x86 specific
 * stackdump function.
 **/
void
z_stackdump_log(ZSignalContext *p)
{
  z_log(NULL, CORE_ERROR, 0, "Stackdump requested, but your platform is not supported;");
}

#endif
