/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: listen.c,v 1.41 2004/10/05 14:06:37 chaoron Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/listen.h>
#include <zorp/io.h>
#include <zorp/log.h>
#include <zorp/socketsource.h>

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <fcntl.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <netinet/tcp.h>
#  include <netinet/in.h>
#endif

#define MAX_ACCEPTS_AT_A_TIME 50

/*
 * This is an implementation of the public ZIOListen interface with
 * some extra hidden attributes. It binds to the given local address,
 * and waits for incoming connections. For each arriving connection a
 * callback is called.
 */
typedef struct _ZIORealListen 
{
  ZIOListen super;
  GSource *watch;
  ZAcceptFunc callback;
  gpointer user_data;
  guint32 sock_flags;
  gint refcnt;
  GStaticRecMutex lock;
  GMainContext *context;
  gchar *session_id;
} ZIORealListen;

/**
 * z_io_listen_accept:
 * @timed_out: specifies whether the operation was timed ot
 * @data: user pointer pointing to self
 *
 * Private callback used as the callback of #ZSocketSource which
 * calls us when the listening socket becomes readable. This function 
 * accepts the connection using z_accept() and calls the callback supplied
 * by our user.
 *
 * Returns: whether further callbacks should be delivered
 *
 **/
static gboolean 
z_io_listen_accept(gboolean timed_out G_GNUC_UNUSED, gpointer data)
{
  ZIORealListen *self = (ZIORealListen *) data;
  ZSockAddr *client;
  gboolean rc = TRUE;
  gint newfd;
  gint accepts = 0;
  GIOStatus res;

  z_enter();
  
  /* NOTE: self->lock protects cancellation, _cancel grabs self->lock thus
   * we either execute first and accept all fds, or self->watch will be
   * NULL and we return */
  
  g_static_rec_mutex_lock(&self->lock);
  if (!self->watch)
    {
      g_static_rec_mutex_unlock(&self->lock);
      z_leave();
      return TRUE;
    }
    
  z_io_listen_ref((ZIOListen *) self);
  
  while (!z_socket_source_is_suspended(self->watch) && rc && accepts < MAX_ACCEPTS_AT_A_TIME)
    {  
      res = z_accept(self->super.fd, &newfd, &client, self->sock_flags);
      if (res == G_IO_STATUS_NORMAL)
        {
#ifdef G_OS_WIN32
          WSAEventSelect(newfd, 0, 0);
#endif
          z_fd_set_nonblock(newfd, 0);
          z_fd_set_keepalive(newfd, 1);
        }
      else if (res == G_IO_STATUS_AGAIN)
        {
          break;
        }
      else
        {
          newfd = -1;
          client = NULL;
        }
      
      rc = self->callback(newfd, client, !!(self->sock_flags & ZSF_ACCEPT_ONE), self->user_data);
      
      accepts++;
      if (self->sock_flags & ZSF_ACCEPT_ONE)
        {
          rc = FALSE;
        }
      if (!self->watch)
        break;
    }
  z_io_listen_unref((ZIOListen *) self);
  g_static_rec_mutex_unlock(&self->lock);
  
  /*LOG
    This message reports the number of accepted connections
    in one poll cycle. If this value is continually high than it
    is likely that the computer can not handle the incoming
    connection rate.
   */
  z_log(self->session_id, CORE_DEBUG, 7, "Accept count; accepts='%d'", accepts);
  
  z_leave();
  return rc;
}

/**
 * z_io_listen_new
 * @local: address to bind to.
 * @sock_flags: a combination of socket flags (ZSF_*)
 * @backlog: backlog value to pass to listen().
 * @callback: function to call, when an incoming connection is accepted.
 * @user_data: opaque pointer passed to callback.
 *
 * This function creates a new ZIOListen instance. Listening to the socket
 * will not be started until z_io_listen_start() is called.
 *
 * Returns: the new ZIOListen instance
 *
 **/
ZIOListen *
z_io_listen_new(const gchar *session_id,
                ZSockAddr *local, 
		guint32 sock_flags,
		gint backlog,
		ZAcceptFunc callback,
		gpointer user_data)
{
  ZIORealListen *self = g_new0(ZIORealListen, 1);

  z_enter();
  self->refcnt = 1;
  self->session_id = session_id ? g_strdup(session_id) : NULL;
  self->callback = callback;
  self->user_data = user_data;
  self->sock_flags = sock_flags;
  self->super.fd = socket(local->sa.sa_family, SOCK_STREAM, 0);
  if (self->super.fd == -1)
    {
      /*LOG
        This message indicate that the creation of a new socket failed
        for the given reason. It is likely that the system is running low
        on memory, or the system is running out of the available fds.
       */
      z_log(self->session_id, CORE_ERROR, 2, "Cannot create socket; error='%s'", g_strerror(errno));
      z_io_listen_unref((ZIOListen *) self);
      z_leave();
      return NULL;
    }
  z_fd_set_nonblock(self->super.fd, 1);
  if (local && z_bind(self->super.fd, local, sock_flags) != G_IO_STATUS_NORMAL)
    { 
      z_io_listen_unref((ZIOListen *) self);
      z_leave();
      return NULL;
    }

  if (z_listen(self->super.fd, backlog, sock_flags) != G_IO_STATUS_NORMAL)
    {
      z_io_listen_unref((ZIOListen *) self);
      z_leave();
      return NULL;
    }

  if (z_getsockname(self->super.fd, &self->super.local, sock_flags) != G_IO_STATUS_NORMAL)
    {
      z_io_listen_unref((ZIOListen *) self);
      z_leave();
      return NULL;
    }
    
  z_leave();

  return (ZIOListen *) self;
}

/**
 * z_io_listen_start:
 * @s: ZIOListen instance
 *
 * Start polling to the listening socket in the main context.
 *
 **/
void
z_io_listen_start(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;
  gchar buf[MAX_SOCKADDR_STRING];
  
  z_enter();
  if (self->watch)
    {
      /*LOG
        This message indicates that the connection was started twice and
        this second attempt is ignored.
       */
      z_log(self->session_id, CORE_ERROR, 4, "Internal error z_io_listen_start called twice, ignoring;");
      z_leave();
      return;
    }
  
  /* our callback might be called immediately, which in turn may free us,
     thus the incremented reference here. */
     

  /*LOG
    This message reports that listening on the given address is
    successfully started.
   */
  z_log(self->session_id, CORE_DEBUG, 7, "Start to listen; fd='%d', address='%s'", self->super.fd, z_sockaddr_format(self->super.local, buf, sizeof(buf)));

  z_io_listen_ref(s);
  self->watch = z_socket_source_new(self->super.fd, Z_SOCKEVENT_ACCEPT, -1);
  g_source_set_callback(self->watch, (GSourceFunc) z_io_listen_accept, self, (GDestroyNotify) z_io_listen_unref);
  g_source_attach(self->watch, self->context);
  
  z_leave();
  return;
}

/**
 * z_io_listen_start_in_context:
 * @s: ZIOListen instance
 * @context: GMainContext to use for polling
 *
 * Start listening to the socket in the specified context.
 *
 **/
void
z_io_listen_start_in_context(ZIOListen *s, GMainContext *context)
{
  ZIORealListen *self = (ZIORealListen *) s;
  
  z_enter();
  g_main_context_ref(context);
  self->context = context;
  z_io_listen_start(s);
  z_leave();
}

/**
 * z_io_listen_suspend:
 * @s: ZIOListen instance
 *
 * Temporarily suspend listening on the socket. Further callbacks will
 * not be delivered until z_io_listen_resume() is called.
 *
 **/
void
z_io_listen_suspend(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;
  
  z_enter();
  
  if (self->watch)
    z_socket_source_suspend(self->watch);
  
  z_leave();
}

/**
 * z_io_listen_resume:
 * @s: ZIOListen instance
 *
 * Resume a suspended listener.
 *
 **/
void
z_io_listen_resume(ZIOListen *l)
{
  ZIORealListen *self = (ZIORealListen *) l;
  
  z_enter();
  
  if (self->watch)
    z_socket_source_resume(self->watch);
  
  z_leave();
}

/**
 * z_io_listen_cancel:
 * @s: ZIOListen instance
 *
 * Cancel listening. No user callbacks will be called after returning from
 * z_io_listen_cancel().
 *
 **/
void
z_io_listen_cancel(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;
  
  z_enter();
  if (self->watch)
    {
      GSource *watch;
      
      /* NOTE: this locks out our accepted callback. We either accept all
       * pending fds, or we NULL out self->watch and our accepted callback
       * won't call any user callbacks. It is therefore guaranteed that no
       * user callbacks will be called after cancellation */
      
      g_static_rec_mutex_lock(&self->lock);
      watch = self->watch;
      self->watch = NULL;
      g_static_rec_mutex_unlock(&self->lock);
      
      g_source_destroy(watch);
      g_source_unref(watch);
    }
  z_leave();
}


/**
 * z_io_listen_free:
 * @s: ZIOListen instance
 *
 * A private function called when the reference count of the ZIOListen
 * instance goes down to zero. It frees all instance variables and the
 * structure itself.
 *
 **/
static void 
z_io_listen_free(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;

  z_enter();
  self->callback = NULL;
  if (self->super.fd != -1)
#ifdef G_OS_WIN32
    closesocket(self->super.fd);
#else
    close(self->super.fd);
#endif
  if (self->context)
    g_main_context_unref(self->context);
  z_sockaddr_unref(self->super.local);
  
  g_free(self->session_id);
  /* we don't need to perform delayed free, as we will only be called by the
     GSource's destroynotify function, which is called when all pending
     source have already been processed.  */
  g_free(self);
  z_leave();
}

/**
 * z_io_listen_ref:
 * @s: ZIOListen instance
 *
 * Increments the reference count of @s
 *
 **/
void
z_io_listen_ref(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;
  
  z_enter();
  g_static_rec_mutex_lock(&self->lock);
  self->refcnt++;
  g_static_rec_mutex_unlock(&self->lock);
  z_leave();
}

/**
 * z_io_listen_unref:
 * @s: ZIOListen instance
 *
 * Decrements the reference count of @s and calls z_io_listen_free() if
 * it goes down to zero.
 *
 **/
void
z_io_listen_unref(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;
  
  z_enter();
  g_static_rec_mutex_lock(&self->lock);
  if (--self->refcnt == 0)
    {
      g_static_rec_mutex_unlock(&self->lock);
      z_io_listen_free(s);
      z_leave();
      return;
    }
  g_static_rec_mutex_unlock(&self->lock);
  z_leave();
}

/**
 * z_io_listen_get_session:
 * @s: ZIOListen instance
 * 
 * Returns: session_id specified at creation
 **/
G_CONST_RETURN gchar *
z_io_listen_get_session(ZIOListen *s)
{
  ZIORealListen *self = (ZIORealListen *) s;
  
  z_enter();
  z_leave();
  return self->session_id;
}

