# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test whether or not a node is compatible with MAAS."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "LocalProxyFixture",
    ]

import errno
import logging
import os.path
import re

from fixtures import Fixture
from maastest import utils
from netaddr import IPNetwork
import netifaces


DISK_CACHE_ROOT = '%(config_dir)s/proxy_cache/'

PROXY_CONFIG_ITEMS = {
    'daemonise': 'true',
    'logFile': '%(log_file)s',
    'pidFile': '%(pid_file)s',
    'proxyAddress': "::0",
    'proxyPort': '%(port)d',
    'allowedClients': '127.0.0.1,%(ip_address)s/%(netmask)s',
    'diskCacheRoot': DISK_CACHE_ROOT
    }

IP_ADDRESS_RE = re.compile(
    ".*inet\s+(\d+\.\d+\.\d+\.\d+)/(\d+)?.*",
    flags=re.DOTALL)


class LocalProxyFixture(Fixture):
    """A test fixture for using a caching proxy in maas-test.

    If no external proxy URL is passed to __init__(), LocalProxyFixture spins
    up a local instance of polipo, a lightweight caching proxy. We use
    polipo here because it's particularly cheap to set up a mayfly
    instance; all of the configuration options can be passed on the
    command line, meaning that we don't need to care about maintaining a
    config file somewhere.

    If LocalProxyFixture is given an external proxy URL, it won't start a
    local proxy and will simply return the external URL when asked.
    """

    #TODO: port should be determined at runtime from the set of
    # available ports rather than being an arbitrary one as it is here.
    def __init__(self, config_dir=None, port=42676, pidfile_dir=None,
                 log_dir=None):
        """Create a `LocalProxyFixture` object.

        :param url: The URL of the proxy to use. If None, LocalProxyFixture
            will create a local caching proxy.
        :type url: string
        :param port: The port on which to start a local caching proxy.
            If URL is specified, this parameter is ignored.
        :type port: int
        :param config_dir: The directory in which to put the proxy's
            configuration and cache.
        :type config_dir: string
        :param pidfile_dir: Optional directory in which to write the proxy's
            pidfile.  Will be created if not present.  Defaults to
            `/run/maas-test`.
        :type pidfile_dir: string
        :param log_dir: Optional directory in which to write the proxy's log
            files.  Will be created if not present.  Defaults to
            `/var/log/maas-test`.
        :type log_dir: string
        """
        super(LocalProxyFixture, self).__init__()
        self.port = port
        if config_dir is None:
            config_dir = utils.DEFAULT_STATE_DIR
        self.config_dir = config_dir
        if pidfile_dir is None:
            pidfile_dir = utils.DEFAULT_PIDFILE_DIR
        if log_dir is None:
            log_dir = utils.DEFAULT_LOG_DIR
        self.pid_file = os.path.join(pidfile_dir, 'proxy.pid')
        self.log_file = os.path.join(log_dir, 'proxy.log')
        self._network_address = None

    def _get_config_arguments(self):
        """Return the proxy config as a sorted list of key=value strings."""
        ip_address = self._get_network_address()
        substitutions = {
            'config_dir': self.config_dir,
            'port': self.port,
            'ip_address': ip_address.ip.format(),
            'netmask': ip_address.prefixlen,
            'pid_file': self.pid_file,
            'log_file': self.log_file,
            }
        config_args = []
        for key, value in PROXY_CONFIG_ITEMS.items():
            arg_string = "%s=%s" % (key, value % substitutions)
            config_args.append(arg_string)
        return sorted(config_args)

    def _get_network_address(self):
        """Return an `IPNetwork` instance for the KVM virtual bridge.

        This allows us to determine which IP address(es) polipo should
        accept connections on.
        """
        if self._network_address is None:
            ipv4_address = netifaces.ifaddresses(
                'virbr0')[netifaces.AF_INET][0]
            self._network_address = IPNetwork(
                "%s/%s" % (ipv4_address['addr'], ipv4_address['netmask']))
        return self._network_address

    def get_url(self):
        """Return the URL of the proxy."""
        ip_address = self._get_network_address()
        return "http://%s:%d" % (ip_address.ip.format(), self.port)

    def setUp(self):
        """Configure the polipo caching http proxy."""
        super(LocalProxyFixture, self).setUp()
        self.create_cache_root()
        # From this point, if further setup fails, we'll need to clean up.
        self.addCleanup(self.kill_running_proxy)

        # Create pidfile and log directories if they did not already exist.
        pidfile_dir = os.path.dirname(self.pid_file)
        if not os.path.isdir(pidfile_dir):
            os.makedirs(pidfile_dir, 0o755)
        log_dir = os.path.dirname(self.log_file)
        if not os.path.isdir(log_dir):
            os.makedirs(log_dir)

        self.start_proxy()

    def start_proxy(self):
        logging.info("Checking for running proxy instance...")
        if os.path.exists(self.pid_file):
            raise Exception(
                "Found pid file %s for running proxy process." %
                self.pid_file)
        logging.info("Starting proxy...")
        utils.run_command(
            ['polipo'] + self._get_config_arguments(), check_call=True)
        logging.info("Done starting proxy.")

    def create_cache_root(self):
        cache_root = DISK_CACHE_ROOT % {'config_dir': self.config_dir}
        try:
            os.makedirs(cache_root)
        except OSError as error:
            if error.errno != errno.EEXIST:
                raise
            # Directory already exists.

    def kill_running_proxy(self):
        """Kill a running proxy."""
        logging.info("Killing proxy...")
        try:
            pid = utils.read_file(self.pid_file).strip()
            utils.run_command(['kill', pid], check_call=True)
            logging.info("Done killing proxy.")
        except Exception as error:
            logging.error("Unable to kill proxy:")
            logging.error(error.message)
