/* This file is part of the KDE project

   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// Own
#include "composer.h"
#include "attachmentwidget.h"
#include "composertextedit.h"
#include "datalistview.h"
#include "addresslineedit.h"
#include "addressbook.h"
#include "mailodybaselistview.h"
#include "tooltip.h"
#include "global.h"
#include "database.h"
#include "messagedata.h"
#include "sendmessage.h"

// Qt
#include <QApplication>
#include <QDockWidget>
#include <QGridLayout>
#include <QLabel>
#include <QMenu>
#include <QPushButton>
#include <QSplitter>
#include <QToolButton>

// KDEPIM
#include <mailtransport/transport.h>
#include <mailtransport/transportbase.h>
#include <mailtransport/transportmanager.h>
#include <mailtransport/transportcombobox.h>
#include <kmime/kmime_dateformatter.h>
#include <kpimidentities/identitycombo.h>
#include <kpimidentities/identity.h>
#include <akonadi/itemcreatejob.h>

// KDE
#include <KAction>
#include <KActionCollection>
#include <KDirLister>
#include <KDirModel>
#include <KMessageBox>
#include <KRun>
#include <KStandardShortcut>
#include <KStatusBar>
#include <KToggleAction>
#include <KToolBar>

// Solid
#include <solid/networking.h>
#include <kstatusbarofflineindicator.h>

using namespace Mailody;

//---------- Composer --------------------------//
Composer::Composer( QWidget* parent, const Akonadi::Collection& mb )
        : KXmlGuiWindow( parent ),
        m_lastState( To ), m_mailbox( mb ),  m_close( false ), m_dirty( false )
{
    setObjectName( "composer#" );
    setCaption( i18n( "Composer" ) );

    if ( !m_mailbox.isValid() )
        kDebug() << "Message can not be stored in " << m_mailbox.remoteId() << endl;

    connect( parent, SIGNAL( setupFinished() ), SLOT( slotStatusChanged() ) );

    m_addressbook = new QTreeWidget( this );
    m_addressbook->setColumnCount( 2 );
    QStringList headers;
    headers << i18n( "Name" ) << i18n( "Email" );
    m_addressbook->setHeaderLabels( headers );
    m_addressbook->setContextMenuPolicy( Qt::CustomContextMenu );
    connect( m_addressbook,
             SIGNAL( itemDoubleClicked( QTreeWidgetItem*,int ) ),
             SLOT( slotAddAddressFromAddressBook( QTreeWidgetItem* ) ) );
    connect( m_addressbook,
             SIGNAL( customContextMenuRequested( const QPoint & ) ),
             SLOT( slotContextMenuAddressBook() ) );

    KDirModel* dirModel = new KDirModel( this );
    m_filebrowser = new QTreeView( this );
    m_filebrowser->setEditTriggers( QAbstractItemView::NoEditTriggers );
    m_filebrowser->setModel( dirModel );
    m_filebrowser->resizeColumnToContents( KDirModel::ModifiedTime );
    m_filebrowser->hideColumn( KDirModel::Owner );
    m_filebrowser->hideColumn( KDirModel::Group );
    m_filebrowser->hideColumn( KDirModel::Type );
    m_filebrowser->hideColumn( KDirModel::Permissions );

    dirModel->dirLister()->openUrl( KUrl( "~" ) );
    connect( m_filebrowser,
             SIGNAL( doubleClicked( const QModelIndex& ) ),
             SLOT( slotAddFile( const QModelIndex& ) ) );

    m_recentbook = new MailodyBaseListView( this );
    m_recentbook->setColumnCount( 4 );
    m_recentbook->setContextMenuPolicy( Qt::CustomContextMenu );
    headers.clear();
    headers << i18n( "Email" ) << i18n( "Name" ) << i18n( "Amount" ) << i18n( "Last" );
    m_recentbook->setHeaderLabels( headers );
    m_recentbook->sortItems( 3, Qt::DescendingOrder );
    connect( m_recentbook,
             SIGNAL( itemDoubleClicked( QTreeWidgetItem*,int ) ),
             SLOT( slotAddAddressFromRecentBook( QTreeWidgetItem* ) ) );
    connect( m_recentbook,
             SIGNAL( customContextMenuRequested( const QPoint & ) ),
             SLOT( slotContextMenuRecentBook() ) );
    connect( m_recentbook,
             SIGNAL( itemEntered( QTreeWidgetItem*, int ) ),
             SLOT( slotShowRecentDetails( QTreeWidgetItem* ) ) );
    connect( m_recentbook, SIGNAL( leftWidget() ), SLOT( slotHideRecentDetails() ) );
    connect( m_recentbook, SIGNAL( viewportEntered() ), SLOT( slotHideRecentDetails() ) );


    // Dockwidgets.
    QDockWidget* addressDock = new QDockWidget( i18n( "Add&ressbook" ), this );
    addressDock->setWidget( m_addressbook );
    addressDock->setObjectName( "addressDock" );
    addressDock->setFeatures( QDockWidget::DockWidgetMovable |
                              QDockWidget::DockWidgetClosable );
    addDockWidget( Qt::LeftDockWidgetArea, addressDock );

    QDockWidget* recentDock = new QDockWidget( i18n( "Re&cent" ), this );
    recentDock->setWidget( m_recentbook );
    recentDock->setObjectName( "recentDock" );
    recentDock->setFeatures( QDockWidget::DockWidgetMovable |
                             QDockWidget::DockWidgetClosable );
    addDockWidget( Qt::LeftDockWidgetArea, recentDock );

    QDockWidget* attachDock = new QDockWidget( i18n( "At&tach File" ), this );
    attachDock->setWidget( m_filebrowser );
    attachDock->setObjectName( "attachDock" );
    attachDock->setFeatures( QDockWidget::DockWidgetMovable |
                             QDockWidget::DockWidgetClosable );
    addDockWidget( Qt::RightDockWidgetArea, attachDock );

    // check that we have state-settings saved. If not hide docks by default.
    KConfigGroup dcfg = KGlobal::config()->group( "Composer" );
    if ( !dcfg.hasKey( "FirstRun" ) ) {
        addressDock->setHidden( true );
        recentDock->setHidden( true );
        attachDock->setHidden( true );
        dcfg.writeEntry( "FirstRun",false );
    }

    tabifyDockWidget( recentDock, addressDock );

    // -------------- main layout ----//

    m_vsplitter = new QSplitter( this );
    m_vsplitter->setOrientation( Qt::Vertical );

    QWidget* widg = new QWidget( m_vsplitter );
    QGridLayout* grid = new QGridLayout( widg );
    grid->setSpacing( 6 );
    grid->setMargin( 6 );

    m_identitylabel = new QLabel( i18n( "Identity:" )+' ', widg );
    m_identityBox = new KPIMIdentities::IdentityCombo( Global::identityManager(),
            widg );
    m_identitylabel->setBuddy( m_identityBox );
    connect( m_identityBox,SIGNAL( identityChanged( uint ) ),
             SLOT( slotReplaceSignature( uint ) ) );

    m_transportlabel = new QLabel( i18n( "Transport:" )+' ', widg );
    m_transportBox = new MailTransport::TransportComboBox( widg );
    m_transportlabel->setBuddy( m_identityBox );
    connect( m_transportBox, SIGNAL( currentIndexChanged( int ) ),
             SLOT( slotTransportChanged() ) );

    QLabel* tolabel = new QLabel( i18n( "Recipient:" )+' ', widg );
    m_edit = new AddressLineEdit( widg );
    tolabel->setBuddy( m_edit );
    connect( m_edit,SIGNAL( shiftAddress() ), SLOT( slotShiftAddress() ) );
    connect( m_edit,SIGNAL( addAddress( const QString& ) ),
             SLOT( slotAddAddress( const QString& ) ) );
    connect( m_edit,SIGNAL( addAddress( const QString& ) ),
             SLOT( slotSetDirty() ) );
    connect( m_edit,SIGNAL( containsValidAddress( bool ) ),
             SLOT( slotContainsValidAddress( bool ) ) );

    m_add = new QPushButton( i18n( "Add" ), widg );
    slotContainsValidAddress( false );
    connect( m_add, SIGNAL( clicked() ), SLOT( slotAddClicked() ) );
    connect( m_add, SIGNAL( clicked() ), SLOT( slotSetDirty() ) );

    m_addressbox = new DataListView( widg,0,5 );
    m_addressbox ->setContextMenuPolicy( Qt::CustomContextMenu );
    m_addressbox->setColumnCount( 5 );
    // since we don't want the pixmaps to be editable, we deal with it
    // ourselfes
    m_addressbox->setEditTriggers( QAbstractItemView::NoEditTriggers );
    m_addressbox->setSelectionMode( QAbstractItemView::SingleSelection );
    m_addressbox->setSelectionBehavior( QAbstractItemView::SelectRows );

    headers.clear();
    headers << i18n( "Address" ) << i18n( "To" ) << i18n( "Cc" ) << i18n( "Bcc" )
    << i18n( "Del" );
    m_addressbox->setHeaderLabels( headers );
    connect( m_addressbox, SIGNAL( itemClicked( QTreeWidgetItem*, int ) ),
             SLOT( slotEditAddress( QTreeWidgetItem*, int ) ) );
    connect( m_addressbox, SIGNAL( itemSelectionChanged() ), SLOT( slotUpdateLineEdit() ) );
    connect( m_addressbox,
             SIGNAL( customContextMenuRequested( const QPoint & ) ),
             SLOT( slotContextMenuAddressList() ) );

    m_alabel = new QLabel( i18n( "Attached:" )+' ', widg );
    m_attachview = new AttachmentWidget( widg );
    connect( m_attachview, SIGNAL( noAttachmentsLeft() ),
             SLOT( slotHideAttachmentWidget() ) );

    m_attachview->hide();
    m_alabel->setBuddy( m_attachview );
    m_alabel->hide();

    grid->addWidget( m_identitylabel,0,0, Qt::AlignRight );
    grid->addWidget( m_identityBox,0,1,1,2 );
    grid->addWidget( m_transportlabel,1,0, Qt::AlignRight );
    grid->addWidget( m_transportBox,1,1,1,2 );
    grid->addWidget( tolabel,2,0, Qt::AlignRight );
    grid->addWidget( m_edit,2,1 );
    grid->addWidget( m_add,2,2 );
    grid->addWidget( m_addressbox,3,1,1,2 );
    grid->addWidget( m_alabel,4,0, Qt::AlignRight );
    grid->addWidget( m_attachview,4,1,1,2 );
    grid->setColumnStretch( 1,10 );

    //--------------------------------------------------------

    QWidget* widg2 = new QWidget( m_vsplitter );
    QGridLayout* grid2 = new QGridLayout( widg2 );
    grid2->setSpacing( 5 );
    grid2->setMargin( 5 );

    QLabel* label = new QLabel( i18n( "Subject:" )+' ', widg );
    m_subject = new KLineEdit( widg );
    m_subject->setObjectName( "subject" );
    label->setBuddy( m_subject );
    connect( m_subject, SIGNAL( textChanged( const QString & ) ),
             SLOT( slotSetDirty() ) );

    m_text = new ComposerTextEdit( widg2 );
    m_text->setRichTextSupport( ComposerTextEdit::FullSupport );
    m_text->createActions( actionCollection() );
    connect( m_text, SIGNAL( textChanged() ), SLOT( slotSetDirty() ) );
    connect( m_text, SIGNAL( addAttachment( const KUrl& ) ),
             SLOT( slotAddFile( const KUrl& ) ) );
    KStandardAction::preferences( this,
                                  SLOT( slotSetup() ), actionCollection() );

    m_textToolBar = new KToolBar( "htmlToolBar", this, Qt::TopToolBarArea );
    m_textToolBar->setToolButtonStyle( Qt::ToolButtonIconOnly );

    m_htmlmode = new KToggleAction( this );
    m_htmlmode->setText( i18n( "HTML Mode" ) );
    actionCollection()->addAction( "toggle_html_mode", m_htmlmode );
    connect( m_htmlmode, SIGNAL( toggled( bool ) ), SLOT( slotHTMLMode( bool ) ) );

    uint identityInt = m_identityBox->currentIdentity();
    const KPIMIdentities::Identity& ident =
        Global::identityManager()->identityForUoidOrDefault( identityInt );
    bool htmlActive = ident.property( s_preferhtml ).toInt() == 1 ? false : true;
    m_htmlmode->setChecked( htmlActive );
    slotHTMLMode( htmlActive );

    m_sendmail = new KAction( this );
    m_sendmail->setText( i18n( "Send Email" ) );
    m_sendmail->setIcon( KIcon( "mail-send" ) );
    m_sendmail->setShortcut( Qt::ControlModifier + Qt::Key_Return );
    m_sendmail->setIcon( KIcon( "mail-send" ) );
    actionCollection()->addAction( "send_message", m_sendmail );
    connect( m_sendmail, SIGNAL( triggered( bool ) ), SLOT( slotSend() ) );

    m_savemail = new KAction( this );
    m_savemail->setText( i18n( "Save Email" ) );
    m_savemail->setIcon( KIcon( "document-save" ) );
    m_savemail->setEnabled( m_mailbox.isValid() );
    m_savemail->setShortcut( Qt::ControlModifier + Qt::Key_S );
    actionCollection()->addAction( "composer_save_message", m_savemail );
    connect( m_savemail, SIGNAL( triggered( bool ) ), SLOT( slotSave() ) );

    m_expandSig = new KAction( this );
    m_expandSig->setText( i18n( "Expand Signature" ) );
    actionCollection()->addAction( "expand_sig", m_expandSig );
    connect( m_expandSig, SIGNAL( triggered( bool ) ), SLOT( slotExpandSignature() ) );

    m_showIdentities = new KToggleAction( this );
    m_showIdentities->setText( i18n( "Show Identities" ) );
    actionCollection()->addAction( "options_show_identities", m_showIdentities );
    connect( m_showIdentities, SIGNAL( triggered( bool ) ),
             SLOT( slotShowIdentities( bool ) ) );

    m_showTransports = new KToggleAction( this );
    m_showTransports->setText( i18n( "Show Transports" ) );
    actionCollection()->addAction( "options_show_transports", m_showTransports );
    connect( m_showTransports, SIGNAL( triggered( bool ) ),
             SLOT( slotShowTransports( bool ) ) );

    // online / offline
    slotStatusChanged();
    connect( Solid::Networking::notifier(),
             SIGNAL( statusChanged( Solid::Networking::Status ) ),
             SLOT( slotStatusChanged() ) );


    KAction *close = new KAction( this );
    close->setText( i18n( "Close" ) );
    close->setIcon( KIcon( "window-close" ) );
    close->setShortcut( KStandardShortcut::close() );
    actionCollection()->addAction( "composer_close", close );
    connect( close, SIGNAL( triggered( bool ) ), SLOT( slotDone() ) );

    KToggleAction *toggleFixed = new KToggleAction( this );
    toggleFixed->setText( i18n( "Fixed Font" ) );
    actionCollection()->addAction( "toggle_fixed_font", toggleFixed );
    connect( toggleFixed, SIGNAL( triggered( bool ) ), SLOT( slotFixedFont( bool ) ) );

    actionCollection()->addAction( "toggle_address_dock",
                                   addressDock->toggleViewAction() );
    actionCollection()->addAction( "toggle_recent_dock",
                                   recentDock->toggleViewAction() );
    actionCollection()->addAction( "toggle_attach_dock",
                                   attachDock->toggleViewAction() );

    KStatusBarOfflineIndicator* nsi =
        new KStatusBarOfflineIndicator( this );

    QToolButton *send = new QToolButton( widg2 );
    send->setDefaultAction( m_sendmail );

    QToolButton *save = new QToolButton( widg2 );
    save->setDefaultAction( m_savemail );

    grid2->addWidget( label,0,0, Qt::AlignRight );
    grid2->addWidget( m_subject,0,1,1,4 );
    grid2->addWidget( m_textToolBar,1,0,1,5 );
    grid2->addWidget( m_text,2,0,1,5 );
    grid2->addWidget( send,3,4 );
    grid2->addWidget( save,3,3 );
    grid2->addWidget( nsi,3,2 );
    grid2->setColumnStretch( 2,10 );

    //--------------------------------------------------------

    m_vsplitter->show();
    setCentralWidget( m_vsplitter );
    setMinimumSize( QSize( 400,300 ) );
    setStandardToolBarMenuEnabled( true );
    toolBar();
    statusBar();
    setupGUI( Keys | StatusBar | ToolBar | Create | Save, "composerui.rc" );

    // set the column width of the subject to the same value
    // as the labels above the splitter. Not sure why 3*spacing is needed.
    m_identitylabel->adjustSize();
    int width = m_identitylabel->width() + ( 3*grid2->spacing() );
    grid2->setColumnMinimumWidth( 0, width );

    // read config for the splitter size
    setAutoSaveSettings( "Composer", true );
    KConfigGroup config = KGlobal::config()->group( "ComposerSplitterSize" );
    slotShowIdentities( config.readEntry( "showIdentities", false ) );
    slotShowTransports( config.readEntry( "showTransports", false ) );

    QList<int> defaultsize = config.readEntry( "vsplitter",QList<int>() );
    if ( !defaultsize.count() ) {
        defaultsize.append( 200 );
        defaultsize.append( 400 );
    }
    m_vsplitter->setSizes( defaultsize );
    m_addressbox->setSortingEnabled( false );
    m_edit->setFocus();

    // When we have time, load the addressess from ab
    QTimer::singleShot( 0, this, SLOT( slotLoadAddressBook() ) );
    QTimer::singleShot( 0, this, SLOT( slotLoadRecentBook() ) );
    m_tip = new ToolTip( this );
    m_timer = new QTimer( this );
    connect( m_timer, SIGNAL( timeout() ), SLOT( slotShowRecentDetailsDoIt() ) );

    // Statusbar stuff
    statusBar()->insertItem( QString(), 1, 10 );
    statusBar()->insertItem( m_identityBox->currentIdentityName(), 2, 1 );
    statusBar()->insertItem( QString(), 3, 1 );
    slotTransportChanged();

    connect( statusBar(), SIGNAL( pressed( int ) ),
             SLOT( slotStatusBarClicked( int ) ) );
}

Composer::~Composer()
{
    KConfigGroup config = KGlobal::config()->group( "ComposerSplitterSize" );
    config.writeEntry( "vsplitter", m_vsplitter->sizes() );
    config.writeEntry( "showIdentities", m_showIdentities->isChecked() );
    config.writeEntry( "showTransports", m_showTransports->isChecked() );
    config.sync();
}

void Composer::slotStatusBarClicked( int item )
{
    QHash<QAction*, QString> temp;
    QMenu *p = new QMenu( this );

    // item 2 is identity and 3 is the transport.
    if ( item ==2 ) {
        foreach( const QString& identity, Global::identityManager()->identities() ) {
            QAction* tmp = new QAction( identity, this );
            p->addAction( tmp );
            temp[tmp] = identity;
        }
    } else if ( item ==3 ) {
        foreach( const QString& transport, MailTransport::TransportManager::self()->transportNames() ) {
            QAction *tmp = new QAction( transport, this );
            p->addAction( tmp );
            temp[tmp] = transport;
        }
    } else {
        return;
    }

    QAction* choice = p->exec( QCursor::pos() );
    const QString newItem = temp.value( choice );

    if ( item == 2 ) {
        m_identityBox->setCurrentIdentity( newItem );
    } else if ( item ==3 ) {
        int id = MailTransport::TransportManager::self()->transportByName( newItem )->id();
        m_transportBox->setCurrentTransport( id );
    }
}
void Composer::slotHideAttachmentWidget()
{
    m_attachview->hide();
    m_alabel->hide();
}

void Composer::slotShowIdentities( bool on )
{
    m_identitylabel->setVisible( on );
    m_identityBox->setVisible( on );
    m_showIdentities->setChecked( on ); // when reading from config...
}

void Composer::slotShowTransports( bool on )
{
    m_transportlabel->setVisible( on );
    m_transportBox->setVisible( on );
    m_showTransports->setChecked( on ); // when reading from config...
}

void Composer::setDirty( bool dirty )
{
    m_dirty = dirty;
}

bool Composer::queryClose()
{
    if ( m_close )
        return true;

    return canClose();
}

bool Composer::canClose()
{
    if ( m_dirty ) {

        if ( m_mailbox.isValid() ) {
            int res = KMessageBox::questionYesNo( this,
                                                  i18n( "You have unsaved changes, do you want to save the draft?" ) );
            if ( res == KMessageBox::Yes ) {
                slotSave();
                return false; /* slotSave will close it if needed */
            } else
                return true;
        } else
            return KMessageBox::Yes == KMessageBox::questionYesNo( this,
                    i18n( "You have unsaved changes, are you sure you want to close this window?" ) );
    }
    return true;
}

void Composer::setRcpt( const QString& address, TypeOfAddress addressType )
{
    m_lastState = addressType;
    addAddress( address );
}

void Composer::setIdentity( KPIMIdentities::Identity identity )
{
    m_identityBox->setCurrentIdentity( identity );
}

void Composer::setSubject( const QString& subject )
{
    m_subject->blockSignals( true );
    m_subject->setText( subject );
    m_origSubject = subject;
    m_subject->blockSignals( false );
}

void Composer::setMsgAndQuote( const QString& plain, const QString& html, bool sign )
{
    if ( m_text->textMode()==KRichTextWidget::Plain ) {
        QString plaintext = i18n( "You wrote:\n" ) + Global::quote( plain ) + '\n';
    }
    setMsg( plain, html, sign );

    bool dirty = m_dirty;
    // Add header.
    m_text->insertHtml( i18n( "<br><hr><b>You wrote:</b><br>" ) );

    // Move to top
    QTextCursor cursor = m_text->textCursor();
    cursor.setPosition( 0 );
    m_text->setTextCursor( cursor );
    setDirty( dirty );
}

void Composer::setMsg( const QString& plain, const QString& html, bool sign )
{
    QString message = ( m_text->textMode()==KRichTextWidget::Rich ) ? html : plain;

    if ( sign )
        addSignaturePlaceholder( message );

    // we do not want to get this dirty
    bool dirty = m_dirty;
    m_text->setTextOrHtml( message );
    ( m_addressbox->topLevelItemCount() > 0 ) ? m_text->setFocus() : m_edit->setFocus();
    setDirty( dirty );
}

void Composer::addAttachment( const KUrl& attachment, const QString& name )
{
    m_attachview->show();
    m_alabel->show();
    m_attachview->addAttachment( attachment, name );
}

void Composer::inReplyTo( const QString& messageID )
{
    m_replyTo = messageID;
}

void Composer::slotUpdateLineEdit()
{
    // we obiously do not want textChanged() emitted here, which would
    // remove all the text accept the P.
    m_edit->blockSignals( true );

    if ( m_addressbox->selectedItems().isEmpty() ) {
        m_edit->clear();
        m_edit->setHelp( false );
        return;
    }
    QTreeWidgetItem* selectedItem = m_addressbox->selectedItems().first();
    if ( !selectedItem->icon( Cc ).isNull() )
        m_edit->setText( i18n( "Press enter to move the address to the bcc field" ) );
    else if ( !selectedItem->icon( Bcc ).isNull() )
        m_edit->setText( i18n( "Press enter to move the address to the to field" ) );
    else if ( !selectedItem->icon( To ).isNull() )
        m_edit->setText( i18n( "Press enter to move the address to the cc field" ) );

    m_edit->blockSignals( false );
    m_edit->setHelp( true );
}

void Composer::slotShiftAddress()
{
    QTreeWidgetItem* selectedItem = m_addressbox->selectedItems().first();
    if ( !selectedItem )
        return;
    if ( !selectedItem->icon( To ).isNull() )
        slotEditAddress( selectedItem,Cc );
    else if ( !selectedItem->icon( Cc ).isNull() )
        slotEditAddress( selectedItem,Bcc );
    else if ( !selectedItem->icon( Bcc ).isNull() )
        slotEditAddress( selectedItem,To );
    slotUpdateLineEdit();
}

void Composer::slotAddAddress( const QString& text )
{
    addAddress( text );
}

void Composer::slotAddClicked()
{
    if ( m_edit->help() )
        slotShiftAddress();
    else
        addAddress( m_edit->text() );
}

void Composer::slotContainsValidAddress( bool active )
{
    m_add->setEnabled( active );
}

void Composer::addSignaturePlaceholder( QString& text )
{
    uint signWith = m_identityBox->currentIdentity();
    KPIMIdentities::Identity ident =
        Global::identityManager()->identityForUoidOrDefault( signWith );
    if ( ident.signature().type() == KPIMIdentities::Signature::Disabled )
        return;

    int pos = ident.property( s_sigpos ).toInt();
    pos == 2 ?  text.prepend( "@@sig@@\n\n" ) : text.append( "@@sig@@" );
}

void Composer::placeSignature( QString& text )
{
    uint signWith = m_identityBox->currentIdentity();
    KPIMIdentities::Identity ident =
        Global::identityManager()->identityForUoidOrDefault( signWith );

    KPIMIdentities::Signature sig = ident.signature();

    // Fetch the signature.
    QString signature = sig.withSeparator();

    // It will be empty if disabled.
    if ( signature.isEmpty() ) {
        text = text.remove( "@@sig@@" );
        return;
    }

    // Replace the placeholder with the signature.
    text = text.replace( "@@sig@@", signature );
}

void Composer::slotReplaceSignature( uint signWith )
{
    Q_UNUSED( signWith );

    // change status bar first...
    statusBar()->changeItem( m_identityBox->currentIdentityName(), 2 );

    // if the signature is expanded already, we don't want this.
    if ( !m_expandSig->isEnabled() )
        return;

    // Remove the current placeholder
    QString msg = m_text->toPlainText();
    msg.remove( "@@sig@@" );

    // Add a new one if needed.
    addSignaturePlaceholder( msg );

    // Set the text in the box again, but don't trigger dirty.
    bool dirty = m_dirty;
    m_text->setText( msg );
    setDirty( dirty );
}

void Composer::addAddress( const QString& text )
{
    if ( text.isEmpty() || text.contains( "@" ) != 1 )
        return;

    // Check for dups.
    for ( int i = 0; i < m_addressbox->topLevelItemCount() ; ++i ) {
        QTreeWidgetItem* item = m_addressbox->topLevelItem( i );
        if ( item->text( 0 ).indexOf( text ) != -1 )
            return;
    }

    m_lastInserted = new QTreeWidgetItem( m_addressbox );
    m_lastInserted->setText( 0, text.trimmed() );

    // although we are in singleSelectionMode, setSelected will happily not
    // clear any previous selection ;-)
    m_addressbox->clearSelection();
    m_lastInserted->setSelected( true );
    m_lastInserted->setIcon( m_lastState, KIcon( "dialog-ok" ) );
    m_lastInserted->setIcon( 4, KIcon( "edit-delete" ) );
    m_addressbox->scrollToItem( m_lastInserted );
    m_lastInserted->setFlags( m_lastInserted->flags() | Qt::ItemIsEditable );
    m_edit->clear();
    m_edit->setFocus();
    slotUpdateLineEdit();
}

void Composer::slotAddFile( const QModelIndex & index )
{
    KDirModel* model = static_cast<KDirModel*>( m_filebrowser->model() );
    KFileItem file = model->itemForIndex( index );
    if ( !file.isFile() )
        return;

    // find existing one
    QList<KUrl> list = m_attachview->attachments().keys();
    foreach( const KUrl& url, list )
    if ( url.path() == file.url().path() )
        return;

    // Add it...
    setDirty( true ); // do it here, as forward message does not come here
    addAttachment( file.url() );
}

void Composer::slotAddFile( const KUrl& file )
{
    setDirty( true );
    addAttachment( file );
}

void Composer::slotEditAddress( QTreeWidgetItem* lvi, int c )
{
    if ( !lvi )
        return;

    if ( c == 0 ) {
        m_addressbox->editItem( lvi );
        return;
    }

    if ( c == 4 ) {
        delete lvi;
        return;
    }

    lvi->setIcon( To,QPixmap() );
    lvi->setIcon( Cc,QPixmap() );
    lvi->setIcon( Bcc,QPixmap() );
    lvi->setIcon( c, KIcon( "dialog-ok" ) );
    m_lastState=c;
}

void Composer::slotSetAddress( const QString& address )
{
    m_edit->setText( address );
}

// -------------- addressbook ---------------------------

void Composer::slotAddAddressFromAddressBook( QTreeWidgetItem* lvi )
{
    if ( !lvi )
        return;

    addAddress( m_abMap[lvi] );
}

void Composer::slotLoadAddressBook()
{
    m_ab = Addressbook::instance()->getAddressbookPointer();
    connect( m_ab, SIGNAL( addressBookChanged( AddressBook* ) ),
             SLOT( slotLoadAddresses( AddressBook* ) ) );
    slotLoadAddresses( m_ab );
}

void Composer::slotLoadAddresses( KABC::AddressBook* ab )
{
    // kDebug() << "Loading Addresses from Addressbook" << endl;
    m_abMap.clear();
    m_addressbook->clear();
    // TODO: don't clear the recent items!!! m_edit->completion()->clear();
    KABC::AddressBook::Iterator it;
    for ( it = ab->begin(); it != ab->end(); ++it ) {
        // For the completion object, i want all addresses....
        QString realName = ( *it ).realName();
        QStringList list = ( *it ).emails();
        QStringList::Iterator its=list.begin();
        for ( ; its!=list.end(); ++its )
            m_edit->completion()->addItem( realName + " <" + ( *its ) + '>' );

        // In the addressbook only show the preferred address...
        QStringList values;
        values << ( *it ).formattedName()
        << ( *it ).preferredEmail()
        << ( *it ).uid();

        QTreeWidgetItem* item = new QTreeWidgetItem( m_addressbook, values );
        m_abMap[item] = ( *it ).fullEmail();
    }
}

void Composer::slotContextMenuAddressBook()
{
    using namespace KABC;

    QTreeWidgetItem* lvi = m_addressbook->currentItem();
    if ( !lvi )
        return;

    QMenu* p = new QMenu( this );
    QAction* open = new QAction( i18n( "Open KAddressBook" ), this );
    QAction* del = new QAction( i18n( "Delete from KAddressBook" ), this );

    p->addAction( open );
    p->addAction( del );
    QAction* choice = p->exec( QCursor::pos() );

    if ( choice == del ) {
        int i = KMessageBox::questionYesNo( this,
                                            i18n( "Do you really want to delete %1 from your "
                                                  "KAddressBook?", lvi->text( 0 ) ) );
        if ( i == KMessageBox::Yes ) {
            Addressee a( m_ab->findByUid( lvi->text( 2 ) ) );
            if ( !a.isEmpty() ) {
                m_ab->removeAddressee( a );
                KABC::Ticket *ticket = m_ab->requestSaveTicket( a.resource() );
                if ( ticket )
                    m_ab->save( ticket );
            } else
                kDebug() << "Address not found!!!" << lvi->text( 2 ) << endl;
        }
    } else if ( choice == open ) {
        KRun::runCommand( "kaddressbook --uid " + lvi->text( 2 ), 0 );
    }
}

void Composer::slotContextMenuAddressList()
{
    using namespace KABC;

    QTreeWidgetItem* lvi = m_addressbox->currentItem();
    if ( !lvi )
        return;

    QMenu* p = new QMenu( this );
    QAction* add = new QAction( i18n( "Add to KAddressBook" ), this );

    p->addAction( add );
    QAction* choice = p->exec( QCursor::pos() );

    if ( choice == add )
        Addressbook::instance()->add( lvi->text( 0 ) );
}

// ---------------------- recent -----------------------//

void Composer::slotLoadRecentBook()
{
    m_recentbook->clear();

    Database* conn = Database::dbinstance();
    QStringList values;
    conn->getRecentList( values );

    QStringList::Iterator its = values.begin();
    while ( its != values.end() ) {
        QString email = ( *its );
        ++its;
        QString name = ( *its );
        ++its;
        QDateTime t;
        t.setTime_t(( *its ).toInt() );
        ++its;
        QString amount = ( *its );
        ++its;

        m_edit->completion()->addItem( name + " <" + email + '>' );
        MailodyBaseListViewItem * i = new MailodyBaseListViewItem( m_recentbook );
        i->setText( 0, email );
        i->setText( 1, name );
        i->setText( 2, amount );

        KMime::DateFormatter df;
        i->setData( 3, Qt::DisplayRole, df.dateString( t ) );
        i->setData( 3, Qt::UserRole, t );

        i->setTextAlignment( 2, Qt::AlignRight );
    }
}

void Composer::slotAddAddressFromRecentBook( QTreeWidgetItem* lvi )
{
    if ( !lvi )
        return;

    addAddress( lvi->text( 0 ) + " <" + lvi->text( 1 ) + '>' );
}

void Composer::slotContextMenuRecentBook()
{
    QTreeWidgetItem* lvi = m_recentbook->currentItem();
    if ( !lvi )
        return;

    QMenu* p = new QMenu( this );
    QAction* del = new QAction( i18n( "Delete" ), this );
    p->addAction( del );
    QAction* choice = p->exec( QCursor::pos() );

    if ( choice == del ) {
        Database* conn = Database::dbinstance();
        conn->deleteFromRecentList( lvi->text( 0 ) );
        delete lvi;
    }
}

void Composer::slotShowRecentDetails( QTreeWidgetItem* lvi )
{
    m_currentItem = lvi;
    m_timer->setSingleShot( true );
    m_timer->start( 500 );
}

void Composer::slotShowRecentDetailsDoIt()
{
    if ( !m_currentItem )
        return;

    QString text = "<table style=\"margin-right: 20px;\" ";
    text.append( "cellpadding=0 cellspacing=0 width=\"100%\">" );
    text.append( "<tr><td colspan=2><b><nobr>" + m_currentItem->text( 1 )
                 + "</nobr></b></td></tr>" );
    text.append( "<tr><td><nobr>"+i18n( "Last mailed at:" )
                 + "</nobr></td><td><nobr>" + m_currentItem->text( 3 )
                 + "</nobr></td><tr>" );
    text.append( "<tr><td><nobr>"+i18n( "Total amount of mails sent:" )
                 + "</nobr></td><td><nobr>" + m_currentItem->text( 2 )
                 + "</nobr></td></tr>" );
    text.append( "</table>" );

    m_tip->setText( QCursor::pos(), text );
}

void Composer::slotHideRecentDetails()
{
    m_timer->stop();
    m_tip->hide();
}

// ---------------------- send -------------------------//

void Composer::addRecipients( Mailody::SendMessage* msg ) //  There's a SendMessage in Win32 API, using Mailody::SendMessage to disambiguate
{
    for ( int i = 0; i < m_addressbox->topLevelItemCount() ; ++i ) {
        QTreeWidgetItem* item = m_addressbox->topLevelItem( i );
        if ( !item->icon( 1 ).isNull() ) {
            msg->addTo( item->text( 0 ).trimmed().toLatin1() );
        } else if ( !item->icon( 2 ).isNull() )
            msg->addCc( item->text( 0 ).trimmed().toLatin1() );
        else if ( !item->icon( 3 ).isNull() )
            msg->addBcc( item->text( 0 ).trimmed().toLatin1() );
    }

}

void Composer::slotSend()
{
    int current = m_transportBox->currentTransportId();
    if ( current == -1 ) {
        KMessageBox::information( this, i18n( "There is no outgoing mailserver configuration found, please set it up before sending" ) );
        slotSetup();
        return;
    }

    if ( !m_addressbox->topLevelItemCount() >= 1 ) {
        KMessageBox::information( this, i18n( "No recipients added. Please add atleast one before sending." ) );
        m_edit->setFocus();
        return;
    }

    if ( !m_origSubject.isEmpty() && m_origSubject != m_subject->text() &&
            !m_replyTo.isEmpty() ) {
        int i = KMessageBox::questionYesNo( 0,
                                            i18n( "You have changed the subject in the thread you "
                                                  "are writing to, do you want to start a new thread?" ),
                                            i18n( "Subject Change" ),
                                            KStandardGuiItem::yes(), KStandardGuiItem::no(),
                                            "break_threads_on_subject_change" );
        if ( i == KMessageBox::Yes )
            m_replyTo.clear();
    }

    // expand signature, since we are sending now.
    slotExpandSignature();

    SendMessage* toSend = new SendMessage( this );
    connect( toSend, SIGNAL( finished() ), SLOT( slotDone() ) );
    connect( toSend, SIGNAL( sendError( const QString& ) ),
             SLOT( slotError( const QString& ) ) );

    addRecipients( toSend );
    toSend->addAttachments( m_attachview->attachments() );

    QString plain= m_text->toPlainText();
    QString html;
    if ( m_text->textMode() == KRichTextWidget::Rich )
        html = m_text->toHtml();

    toSend->setSubject( m_subject->text() );
    toSend->send( m_mailbox, m_identityBox->currentIdentity(),
                  m_transportBox->currentTransportId(), plain, html );
}

void Composer::slotSave()
{
    if ( !m_mailbox.isValid() ) {
        kWarning() << m_mailbox.name() << " is not valid. Not saving.";
        return;
    }

    SendMessage* toSend = new SendMessage( this );
    addRecipients( toSend );
    toSend->addAttachments( m_attachview->attachments() );

    QString plain= m_text->toPlainText();
    QString html;
    if ( m_text->textMode() == KRichTextWidget::Rich )
        html = m_text->toHtml();

    KMime::Message* kmimeMessage = toSend->compileMessage( SendMessage::Draft,
                                   m_identityBox->currentIdentity(), plain, html );
    const QString flags = "\\Draft";

    if ( !kmimeMessage->hasContent() ) {
        kWarning() << "Message has no content, aborting";
        return;
    }

    Akonadi::Item item( "message/rfc822" );
    item.setPayload( MessagePtr( kmimeMessage ) );

    Akonadi::ItemCreateJob* job = new Akonadi::ItemCreateJob( item, m_mailbox, 0 );
    connect( job, SIGNAL( result( KJob* ) ), this, SLOT( slotDone( KJob* ) ) );
}

void Composer::slotFixedFont( bool toggle )
{
    toggle ? m_text->setFont( KGlobalSettings::fixedFont() )
    : m_text->setFont( KGlobalSettings::generalFont() );
}

void Composer::slotExpandSignature()
{
    m_expandSig->setEnabled( false );

    QString msg = m_text->textOrHtml();
    placeSignature( msg );

    // if this is rich text, switch to rich text mode first...
    ComposerTextEdit::Mode mode = m_text->textMode();
    bool dirty = m_dirty;
    m_text->setTextOrHtml( msg );
    setDirty( dirty );

    // If the signature is made up from html, the textedit is now
    // automatically set to html mode, revert that.
    // It kind of sucks, because formatting could be lost here.
    if ( mode == ComposerTextEdit::Plain && m_text->textMode() == ComposerTextEdit::Rich ) {
        m_text->switchToPlainText();
    }
}

void Composer::slotHTMLMode( bool toggle )
{
    if ( !toggle && m_text->textMode() == KRichTextEdit::Rich && !m_text->toPlainText().isEmpty() ) {
        int choice = KMessageBox::warningContinueCancel( this, i18n( "Switching from HTML mode "
                     "to Plain mode will cause the text to loose the formatting. Are you sure?" ),
                     i18n( "Loose the formatting?" ), KStandardGuiItem::cont(), KStandardGuiItem::cancel(),
                     "switchhtml2plainisok" );
        if ( choice != KMessageBox::Continue ) {
            m_htmlmode->setChecked( true );
            return;
        }
    }
    m_textToolBar->setEnabled( toggle );
    m_textToolBar->setVisible( toggle );

    toggle ? m_text->enableRichTextMode() : m_text->switchToPlainText();
}

void Composer::slotSetup()
{
    emit showSettings();
}

void Composer::slotSetDirty()
{
    setDirty( true );
}

void Composer::slotDone()
{
    setDirty( true );
    close();
}

void Composer::slotDone( KJob* job )
{
    //TODO: KMessageBox...
    QApplication::restoreOverrideCursor();
    if ( job->error() ) {
        kWarning() << job->errorString();
    } else {
        m_close = true;
        close();
    }
}

void Composer::slotError( const QString& error )
{
    QApplication::restoreOverrideCursor();
    KMessageBox::information( this, error );
}

void Composer::slotStatusChanged()
{
    m_sendmail->setEnabled( Global::connectionPossible() );

    // if changed to offline, enable the sendmail button when the
    // mailtransport is set to a sendmail job.
    if ( !Global::connectionPossible() )
        slotTransportChanged();
}

void Composer::slotTransportChanged()
{
    int current = m_transportBox->currentTransportId();
    if ( current == -1 ) {
        KMessageBox::information( this, i18n( "There is no outgoing mailserver configuration found, please set it up before sending" ) );
        // as this can be called from the constructor, wait for the eventloop, so the mainwindow is
        // connected to the signal.
        QTimer::singleShot( 0, this, SLOT( slotSetup() ) );
        statusBar()->changeItem( i18nc( "No valid mail transport selected","None" ), 3 );
        return;
    }

    // set the status bar.
    const QString t = MailTransport::TransportManager::self()->transportById( current )->name();
    statusBar()->changeItem( t, 3 );

    // the idea is that the 'sendmail' transports should be possible,
    // regardless of the online/offline state.
    if ( Global::connectionPossible() )
        return;

    bool sendmailTransport = m_transportBox->transportType() ==
                             MailTransport::Transport::EnumType::Sendmail;
    m_sendmail->setEnabled( sendmailTransport );
}

#include "composer.moc"
