/* This file is part of the KDE project

   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>

   This file is based on digikams albumdb which is
   Copyright 2004 by Renchi Raju <no_working@address.known>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// Own
#include "database.h"

// C ANSI
extern "C"
{
#include <sys/time.h>
#include <time.h>
}

// C++
#include <cstdio>
#include <cstdlib>

// Qt
#include <QDir>
#include <QMap>
#include <QSqlQuery>
#include <QVariant>
#include <QSqlError>
#include <QSqlDriver>

// KDE
#include <kdebug.h>
#include <KStandardDirs>

namespace Mailody
{

Database *Database::m_dbinstance = 0;

Database::Database()
{
    m_db = QSqlDatabase::addDatabase( "QSQLITE", "mailodymain" );
    setDatabasePath( KStandardDirs::locateLocal( "appdata","mailody4.db" ) );
}

Database::~Database()
{
    if ( m_db.isOpen() )
        m_db.close();
}

Database *Database::dbinstance()
{
    kDebug() << !m_dbinstance;
    if ( !m_dbinstance )
        m_dbinstance = new Database;

    return m_dbinstance;
}

void Database::setDatabasePath( const QString& path )
{
    if ( m_db.isOpen() )
        m_db.close();


    m_db.setDatabaseName( path );
    bool open = m_db.open();
    if ( !open )
        kFatal() << "Cannot open database " << path
        << m_db.lastError().text();
    else {
        kDebug() << "Database is opened: " << path;
        initDatabase();
    }
}

void Database::initDatabase()
{

    // Check if we have the required tables

    QSqlQuery res = execSql( "SELECT name FROM sqlite_master"
                             " WHERE type='table'"
                             " ORDER BY name;" );
    QStringList values;
    while ( res.next() )
        values.append( res.value( 0 ).toString() );

    if ( !values.contains( "recent" ) )
        execSql( QString( "CREATE TABLE recent"
                          " (email TEXT, "
                          " name TEXT, "
                          " last TEXT, "
                          " amount integer, "
                          " UNIQUE( email ) );" ) );
}

QSqlQuery Database::execSql( const QString& sql )
{
    kDebug() << "SQL-query: " << sql << endl;

    if ( !m_db.isOpen() ) {
        kFatal() << "Database is not open";
        return false;
    }

    QSqlQuery query( sql, m_db );
    if ( query.lastError().isValid() )
        kWarning() << "Error in the query:" << query.lastError().text() << sql;

    return( query );
}

void Database::beginTransaction()
{
    m_db.transaction();
}

void Database::commitTransaction()
{
    m_db.commit();
}

QString Database::escapeString( const QString& str ) const
{
    QString sting = str;
    sting.replace( "'", "''" );
    return sting;
}

// -------------------- recent queries ----------------- //

void Database::getRecentList( QStringList& values )
{
    QSqlQuery res = execSql( QString( "SELECT email,name,last,amount FROM recent" ) );
    while ( res.next() )
        for ( int i = 0; i<4 ; ++i )
            values.append( res.value( i ).toString() );
}

void Database::getTopTenRecentList( QStringList& values )
{
    QSqlQuery res = execSql( QString( "SELECT name,email FROM recent "
                                      "order by amount desc limit 10" ) );
    while ( res.next() )
        for ( int i = 0; i<2 ; ++i )
            values.append( res.value( i ).toString() );
}

void Database::addToRecentList( const QString& email, const QString& name )
{
    // first see if it is already in the database
    QSqlQuery res = execSql( QString( "SELECT last from recent where email='%1'" )
                             .arg( escapeString( email ) ) );

    if ( res.next() ) {
        // It exists so update the data
        execSql( QString( "update recent set amount=amount+1, "
                          "last=strftime(\'%s\',\'now\') where "
                          "email='%1'" ).arg( escapeString( email ) ) );
    } else {
        // does not exist, add it.
        execSql( QString( "insert into recent values('%1', '%2', "
                          "strftime(\'%s\',\'now\'), 1)" )
                 .arg( escapeString( email ), escapeString( name ) ) );
    }
}

void Database::deleteFromRecentList( const QString& email )
{
    execSql( QString( "delete from recent where email = '%1'" )
             .arg( escapeString( email ) ) );
}

}
