/* This file is part of the KDE project

   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef MAILODYBASELISTVIEW_H
#define MAILODYBASELISTVIEW_H

#include <QTreeWidget>

namespace Mailody
{

class MailodyBaseListViewItem;

/**
 * @class MailodyBaseListView
 *
 * This is a drop in replacement for a QTreeWidget. It extends the
 * QTreeWidget with a couple of convenience functions and some defaults.
 */

class MailodyBaseListView : public QTreeWidget
{
    Q_OBJECT

public:
    /** Constructor */
    explicit MailodyBaseListView( QWidget* parent );

    /** Returns a list of column sizes so you can easily store them */
    QList<int> columnSizes();

    /** Sets the column sizes based on the passed list */
    void setColumnSizes( QList<int> );

protected:
    void leaveEvent( QEvent * );
    void mousePressEvent( QMouseEvent * event );

signals:
    /** emitted when the mouse leaves the widget*/
    void leftWidget();

    /**
     * emitted on left click, can be emitted when not over an item, in that
     * case there is a click in an empty area. So check if item is null.
     * @param item the item, 0 when no item is clicked
     * @param column the column
     * @param point coordinated based on the viewport
     */
    void selectButton( MailodyBaseListViewItem* item, const int column );

    /**
     * emitted on middle click, can be emitted when not over an item, in that
     * case there is a click in an empty area. So check if item is null.
     * @param item the item, 0 when no item is clicked
     * @param column the column
     * @param point coordinated based on the viewport
     */
    void midButton( MailodyBaseListViewItem* item, const int column );

    /**
     * emitted on right click, can be emitted when not over an item, in that
     * case there is a click in an empty area. So check if item is null.
     * @param item the item, 0 when no item is clicked
     * @param column the column
     * @param point coordinated based on the viewport
     */
    void contextButton( MailodyBaseListViewItem* item, const int column );
};

/**
 * @class MailodyBaseListViewItem
 *
 * This is a drop in replacement for a QTreeWidgetItem. It extends the
 * QTreeWidgetItem with two features: levels and fancy dates.
 *
 * If you give an item a level it will always appear above all items with
 * a lower level.
 *
 * If you call setFancyDate(), that column will hold a fancy date.
 *
 * @author Tom Albers <tomalbers@kde.nl>
 */
class MailodyBaseListViewItem : public QTreeWidgetItem
{
public:
    /** Constructor, the new item will have level 1 by default */
    explicit MailodyBaseListViewItem( QTreeWidget* parent );

    /** Constructor, the new item will have the level of @p parent
     * by default */
    explicit MailodyBaseListViewItem( MailodyBaseListViewItem* parent );

    /**
     * returns the current level
     */
    int level() const {
        return m_level;
    }

    /**
     * sets @p level to this item. By default all new items are
     * created with level 1.
     */
    void setLevel( int level ) {
        m_level = level;
    }

    /**
     * reimplemented to sort first on the level, if two items
     * are on the same level, it sorts on the key as usual.
     */
    virtual bool operator<( const QTreeWidgetItem &other ) const;

private:
    int         m_level;
};

}

#endif
