/* This file is part of the KDE project

   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef MESSAGEVIEW_H
#define MESSAGEVIEW_H

#include "messagedata.h"
#include <QWidget>
#include <QScrollArea>
#include <QGridLayout>
#include <khtml_part.h>

class QLabel;
namespace Mailody
{

class MessageHeaderView;
class MessageBodyView;
class AddressLabel;


/**
 * @class MessageHeaderView
 * This class will show the headers.
 * @author Tom Albers <tomalbers@kde.nl>
 */
class MessageHeaderView : public QScrollArea
{
    Q_OBJECT

public:

    /**
     * Constructor
     */
    explicit MessageHeaderView( QWidget* );

    /**
     * Destructor
     */
    ~MessageHeaderView();

    /**
     * this will fill the widget with the info from @p msg
     * you probably don't want to call this, you probably want
     * TotalView::setMsg() so it also updates the body
     */
    void setMsg( MessageData* msg );

    /**
     * remove the existing info.
     */
    void clearView();

    /**
     * Reread config for the autosaving feature for attachments
     */
    void autoSaveAttachments();

private slots:
    void slotLeftMouseClick( const QString& );
    void slotRightMouseClick( const QString& );
    void updateHeaders();

signals:
    void openComposer( const Akonadi::Collection&, const QString& );

private:

    void getFreeFilename( KUrl &fileLocation );
    void autoSaveAttachment( const KUrl&, const QString& );
    MessageData*                m_currentMessage;
    QGridLayout*                m_layout;
    AddressLabel*               m_label;
    QLabel*                     m_foto;
    bool                        m_storeAttachments;
    KUrl                        m_attachmentFolder;
};

/**
 * @class MessageBodyView
 * This class is responsible for displaying the body of the message
 * to achieve this, it inherits a KHTMLPart
 */
class MessageBodyView : public KHTMLPart
{
    Q_OBJECT

public:
    /**
    * Constructor
    */
    explicit MessageBodyView( QWidget* );

    /**
    * Destructor
    */
    ~MessageBodyView();

    /**
     * Scrolls to the top of the message
     */
    void top()             { /*TODO: setContentsPos(0,0); */};

    /**
     * Returns if the message html will be interepreted or displayed as
     * plain text (escaped).
     */
    bool html() const      {
        return m_bodyType == MessageData::HTML;
    };

    /**
     * Returns if the source of the message will be showed
     */
    bool source() const    {
        return m_bodyType == MessageData::Source;
    };

    /**
     * Returns if the message will be shown in fixed font, has no use
     * in html mode of course
     */
    bool fixedfont() const {
        return m_fixedFont;
    };

    /**
     * Returns if the message will load external images in html-mode
     */
    bool externalImage() const {
        return m_externalImage;
    };

    /**
     * Sets the message to be shown, remember that you probably want to
     * use TotalView::setMsg() to also get updated headers
     */
    void setMsg( MessageData* msg );

    /**
     * set @p i to true to render the html in the message
     */
    void setHTML( bool i );

    /**
     * set @p i to true to load external images in the message
     * it will update the message immediately, unless @p updateScreen
     * is set to false.
     */
    void setExternalImage( bool i, bool updateScreen=true );

    /**
     * set this so the links will open directly, normaly it would
     * open a link in a external browser or emit the signal on middle
     * click
     */
    void setOpenLinksDirectly( bool i );

    /**
     * set @p i to true to see the message in a fixed font
     */
    void setFixedFont( bool i );

    /**
     * Call this to see the source of the message
     */
    void setViewSource();

    /**
     * this will clear the body.
     */
    void clearView();

    /**
     * Zoom in
     */
    void increaseZoomLevel();

    /**
     * Zoom out
     */
    void decreaseZoomLevel();

protected:
    virtual bool urlSelected( const QString&, int, int, const QString &,
                              const KParts::OpenUrlArguments& args=KParts::OpenUrlArguments(),
                              const KParts::BrowserArguments& browserArgs=KParts::BrowserArguments() );

private slots:
    void slotShowMsg( const MessageData* );
    void slotPopupMenu( const QString&, const QPoint& );
    void slotSetCaption( const QString& title );

signals:
    void openComposer( const Akonadi::Collection&, const QString& );
    void openInTab( const QString& );
    void pageTitle( const QString& );

private:
    MessageData*             m_currentMessage;
    MessageData::bodyType    m_bodyType;
    bool                     m_fixedFont;
    bool                     m_externalImage;
    bool                     m_openDirectly;
};

/** @class TotalView
 * This class is responsible for displaying messages. It consits of three
 * classes. There is a TotalView, which displays two widgets: the MessageHeaderView
 * for all the headers and the MessageBodyView which contains the body of the
 * message
 * @author Tom Albers <tomalbers@kde.nl>
 */
class TotalView : public QWidget
{
    Q_OBJECT
public:
    /**
     * Constructor
     */
    explicit TotalView( QWidget* );

    /**
     * Destructor
     */
    ~TotalView();

    /**
     * returns a pointer to the header widget
     */
    MessageHeaderView*  headers() const     {
        return m_headers;
    };

    /**
     * returns a pointer to the body widgget
     */
    MessageBodyView* body() const        {
        return m_body;
    };

    /**
     * returns a pointer to the currently shown message, or 0.
     */
    MessageData* msg() const         {
        return m_msg;
    };

    /**
     * this call asks to show @p msg in the view.
     */
    void setMsg( const Akonadi::Item&, const Akonadi::Collection& );

    /**
     * this will hide the headers and clear the body.
     */
    void clearView();

    /**
     * guess what...
     */
    void print();

    /**
     * Guess again...
     */
    void save( const QString& filename );

    /**
     * Reread config for the autosaving feature for attachments
     */
    void autoSaveAttachments() {
        m_headers->autoSaveAttachments();
    };

private:
    MessageData             *m_msg;
    MessageHeaderView       *m_headers;
    MessageBodyView         *m_body;
};
}

#endif
